#**********************************************************************************************************************************
# File          :   DXCluster.py
# Project       :   Telnet DX Cluster
# Description   :   DX Spider Telnet client, can be run standalone, or from within MiniBook with Spot click -> Logbook and Rigcontrol
# Date          :   27-05-2025
# Authors       :   Bjorn Pasteuning - PD5DJ
# Website       :   https://wwww.pd5dj.nl
#
# Version history
#   27-05-2025  :   1.0.0   -   Initial basics running
#   28-05-2025  :   1.0.1   -   All functions working
#   30-05-2025  :   1.0.2   -   Add spot function added
#                           -   cty.dat download added
#                           -   MODES catagories expanded
#   01-06-2025  :   1.0.3   -   UI layout changed, Exit button added
#   07-06-2025  :   1.0.4   -   Fixed focus when clicked on Send spot button
#   22-06-2025  :   1.0.5   -   Custom filter added, based of REGEX
#                           -   Users can now see theirselves in the tree when they are spotted. (green)
#                           -   Auto reconnect function added, when connection drops, app will try to reconnect every 5 seconds, when Auto Reconnect is enabled.
#   11-07-2025  :   1.0.6   -   Band Combobox changed for band buttons.
#   10-08-2025  :   1.0.7   -   Changed filepath structure
#   29-08-2025  :   1.0.8   -   Send Spot handling changed
#   25-10-2026  :   1.0.9   -   Treeview build rewritten, now fully buffered to maximize update speed.
#                   1.1.0   -   Callsign alert added
#   29-11-2025  :   1.1.1   -   ADD -   Alert color added to tree and legend
#   06-12-2025  :   1.1.2   -   FIX -   UI fix, windows are now properly aligned to app window.
#**********************************************************************************************************************************

import asyncio
import json
import threading
import tkinter as tk
from tkinter import ttk, messagebox
import winsound  # For Windows alert sounds
import re
import configparser
import os
import socket
from pathlib import Path
from datetime import datetime, timezone
from cty_parser import parse_cty_file
import requests

VERSION_NUMBER = ("v1.1.2")

SETTINGS_FOLDER     = Path.cwd() / "settings"
DATA_FOLDER         = Path.cwd() / "data"

INI_FILE        = SETTINGS_FOLDER / "dxcluster.ini"
ALERT_FILTER    = SETTINGS_FOLDER / "alert_filter.json"
CLUSTER_FILE    = SETTINGS_FOLDER / "clusters.json"
DXCC_FILE       = DATA_FOLDER / "cty.dat"

ctydat_url      = "https://www.country-files.com/bigcty/cty.dat"

BANDS = {
    "HF": (1.8, 30.0),
    "VHF": (30.0, 300.0),
    "UHF": (300.0, 1000.0),
    "SHF": (1000.0, 3000.0),    
    "160m": (1.8, 2.0),
    "80m": (3.5, 3.8),
    "60m": (5.3305, 5.4065),
    "40m": (7.0, 7.3),
    "30m": (10.1, 10.15),
    "20m": (14.0, 14.35),
    "17m": (18.068, 18.168),
    "15m": (21.0, 21.45),
    "12m": (24.89, 24.99),
    "10m": (28.0, 29.7),
    "6m": (50.0, 54.0),
    "4m": (70.0, 70.5),
    "40MHz": (40.66, 40.7),
    "2m": (144.0, 148.0),
    "70cm": (420.0, 450.0),
    "23cm": (1240.0, 1300.0),
    "13cm": (2300.0, 2450.0),
}


MODES = {
    "CW": [
        (1.800, 1.829),
        (3.500, 3.559),
        (7.000, 7.039),
        (10.100, 10.139),
        (14.000, 14.069),
        (18.068, 18.109),
        (21.000, 21.069),
        (24.890, 24.909),
        (28.000, 28.069),
        (50.000, 50.099),
        (70.000, 70.099),
        (144.000, 144.099),
        (432.000, 432.099),
        (1296.000, 1296.099),
        (2300.000, 2300.199),
    ],
    "FT8": [
        (1.840, 1.841),
        (3.573, 3.574),
        (7.074, 7.077),
        (10.136, 10.137),
        (14.074, 14.077),
        (18.100, 18.101),
        (21.074, 21.075),
        (24.915, 24.916),
        (28.074, 28.075),
        (50.313, 50.314),
        (70.154, 70.155),
        (144.174, 144.177),
        (432.174, 432.177),
    ],
    "FT4": [
        (3.575, 3.576),
        (7.047, 7.048),
        (14.080, 14.081),
        (21.140, 21.141),
        (50.318, 50.319),
        (144.170, 144.171),
    ],
    "DIGITAL": [
        (1.830, 1.839),
        (1.842, 1.849),
        (3.560, 3.572),
        (3.577, 3.599),
        (7.040, 7.046),
        (7.049, 7.069),
        (10.140, 10.135),
        (10.138, 10.149),
        (14.070, 14.073),
        (14.076, 14.119),
        (18.110, 18.099),
        (18.102, 18.167),
        (21.070, 21.073),
        (21.076, 21.139),
        (24.910, 24.914),
        (24.917, 24.989),
        (28.070, 28.073),
        (28.076, 28.119),
        (50.300, 50.312),
        (50.315, 50.317),
        (50.320, 50.499),
        (70.200, 70.153),
        (70.156, 70.299),
        (144.800, 144.169),
        (144.172, 144.989),
        (432.600, 432.173),
        (432.176, 432.999),
        (1296.200, 1296.499),
        (2400.000, 2402.999),
    ],
    "PHONE": [
        (1.850, 1.999),
        (3.600, 3.799),
        (7.070, 7.299),
        (14.150, 14.349),
        (21.200, 21.449),
        (28.400, 29.699),
        (50.100, 53.999),
        (70.100, 70.499),
        (144.200, 147.999),
        (430.000, 439.999),
        (1297.000, 1299.999),
        (2403.000, 2449.999),
    ],
}



class DXClusterApp:
    def __init__(self, root, rigctl_host="127.0.0.1", rigctl_port=4532,
                 tracking_var=None, on_callsign_selected=None,
                 get_worked_calls=None, get_worked_calls_today=None,
                 get_last_qso_callsign=None, get_current_frequency=None):
        self.root = root
        self.root.title(f"DX Cluster Telnet Client - {VERSION_NUMBER} - by PD5DJ")

        self._open_windows = {
            "manage_filters": None,
            "regex_help": None,
            "edit_filter": None
        }


        # MiniBook integration
        self.rigctl_host = rigctl_host
        self.rigctl_port = rigctl_port
        self.tracking_var = tracking_var
        self.on_callsign_selected = on_callsign_selected
        self.get_worked_calls = get_worked_calls
        self.get_worked_calls_today = get_worked_calls_today
        self.get_last_qso_callsign = get_last_qso_callsign
        self.get_current_frequency = get_current_frequency

        # Buffering voor smoother realtime updates
        self._initial_buffer = []          # SH/DX historical buffer (will be sorted before single UI build)
        self._live_buffer = []             # ongoing live spots buffer (flushed periodically)
        self._buffer_lock = threading.Lock()
        self._buffering_initial = False    # True while we collect initial SH/DX lines
        self._live_flush_interval = 150    # ms between live flushes (tweakable)

        self.all_spots = []  # permanente buffer van alle ontvangen spots

 
        self.custom_filters = self.load_custom_filters()


        # Telnet/async setup
        self.reconnect_attempts = 0
        self.user_callsign = ""
        self.manual_disconnect = False

        self.connected = False
        self.writer = None
        self.loop = asyncio.new_event_loop()
        threading.Thread(target=self.loop.run_forever, daemon=True).start()

        # UI setup
        self.setup_ui()

        # Make sure that external .destroy() calls also trigger on_close
        self.root.destroy_original = self.root.destroy
        self.root.destroy = self.on_close_wrapper

        self.root.protocol("WM_DELETE_WINDOW", self.on_close)

        # Load Hosts en DXCC
        self.hosts = []
        self.spots = []
        self.load_host_file(default=CLUSTER_FILE)

        # Checks if cty.dat is present, if not download. and parse it
        self.check_ctydat_file()

        # Tracking status label
        #self.update_tracking_status()
        
        self._last_worked_calls_today = set()
        self.schedule_worked_calls_check()

        self.restore_window_position()

        # --- Fix window size after building UI ---
        self.root.update_idletasks()  # Zorg dat layout klaar is
        width = self.root.winfo_width()
        height = self.root.winfo_height()
        x = self.root.winfo_x()
        y = self.root.winfo_y()

        # Startgrootte instellen, maar wel verticaal resizable
        self.root.geometry(f"{width}x{height}+{x}+{y}")
        self.root.resizable(False, True)




    def on_close_wrapper(self):
        self.on_close() # Process everything through the standard shutdown routine



    def start_loop(self):
        asyncio.set_event_loop(self.loop)
        self.loop.run_forever()        






    def setup_ui(self):
        top_frame = tk.Frame(self.root)
        top_frame.pack(padx=10, pady=5, fill=tk.X)

        top_frame.columnconfigure(0, weight=1)
        top_frame.columnconfigure(1, weight=0)

        filters_frame = tk.LabelFrame(top_frame, font=('Arial', 10, 'bold'), text="Spot Filters", bd=2)
        filters_frame.grid(row=0, column=0, padx=(0, 10), pady=(0, 10), sticky="nsew")

        # --- Band filter ---
        band_frame = tk.LabelFrame(filters_frame, font=('Arial', 10, 'bold'), text="Filter by band", labelanchor="n")
        band_frame.grid(row=1, column=0, columnspan=5, sticky="ew", pady=(5, 0))

        self.band_var = tk.StringVar(value="ALL")

        def get_band_low(b):
            return BANDS[b][0] if b in BANDS else float('inf')

        # Sorteer banden op frequentie
        sorted_bands = sorted(
            [b for b in BANDS if b not in ("HF", "VHF", "UHF", "SHF")],
            key=get_band_low
        )
        bands_list = ["ALL"] + sorted_bands
        self.band_buttons = {}

        band_buttons_frame = tk.Frame(band_frame)
        band_buttons_frame.grid(row=0, column=0, columnspan=10, sticky="w", pady=(2, 5))

        def update_band_highlight():
            for b, btn in self.band_buttons.items():
                if self.band_var.get() == b:
                    btn.config(bg="blue", fg="white")
                else:
                    btn.config(bg="SystemButtonFace", fg="black")

        def set_band_and_filter(band):
            self.band_var.set(band)
            update_band_highlight()
            self.refresh_filtered_view()  # ✅ gebruik nieuwe methode

        buttons_per_row = 9
        for i, band in enumerate(bands_list):
            row = i // buttons_per_row
            col = i % buttons_per_row
            btn = tk.Button(band_buttons_frame, text=band, width=5, command=lambda b=band: set_band_and_filter(b))
            btn.grid(row=row, column=col, padx=1, pady=1)
            self.band_buttons[band] = btn

        update_band_highlight()

        # --- Mode filter ---
        mode_frame = tk.LabelFrame(filters_frame, text="Filter by mode & Custom regex", font=('Arial', 10, 'bold'), labelanchor="n")
        mode_frame.grid(row=2, column=0, columnspan=5, sticky="ew", pady=(5, 0))        

        # --- Mode filter combobox ---
        tk.Label(mode_frame, text="Mode:").grid(row=2, column=0, sticky="w")
        self.mode_var = tk.StringVar()
        modes_with_all = ["ALL"] + list(MODES.keys()) + list(self.custom_filters.keys())

        import tkinter.font as tkfont
        font_small = tkfont.Font(family="Arial", size=12)

        self.mode_menu = ttk.Combobox(mode_frame, textvariable=self.mode_var, values=modes_with_all, state="readonly", width=10)
        self.mode_menu.configure(font=font_small)
        self.mode_menu.grid(row=2, column=1, padx=(2, 10), pady=5, sticky=tk.W)
        self.mode_menu.current(0)
        self.mode_menu.bind("<<ComboboxSelected>>", lambda e: self.refresh_filtered_view())  # ✅

        # --- Custom filters beheer ---
        tk.Label(mode_frame, text="Manage Custom Filters:").grid(row=2, column=2, sticky="e")
        tk.Button(mode_frame, text="⚙", width=2, command=self.manage_custom_filters).grid(row=2, column=3, padx=(0, 5))
        tk.Button(mode_frame, text="?", width=2, command=self.show_regex_help).grid(row=2, column=4, padx=(0, 5))

        # --- Callsign alert filter ---
        alert_frame = tk.LabelFrame(filters_frame, text="Callsign Alert", font=('Arial', 10, 'bold'), labelanchor="n")
        alert_frame.grid(row=3, column=0, columnspan=5, sticky="ew", pady=(5, 0))


        # Loads alert_filter.json directly at start
        self.alert_callsigns = self.load_alert_callsigns()
        
        # Centrer buttons inside alert_frame
        btn_frame = tk.Frame(alert_frame)
        btn_frame.pack(pady=8)

        edit_btn = tk.Button(btn_frame, text="✏️ Edit Filter", width=14, command=self.edit_alert_filter)
        edit_btn.pack(side="left", padx=10)

        clear_btn = tk.Button(btn_frame, text="🗑️ Clear Alerts", width=14, command=self.clear_alerts)
        clear_btn.pack(side="left", padx=10)


        # Checkbox: enable/disable sound on new alert
        sound_value = self.load_alert_sound_setting()  # Load from INI
        self.alert_sound = tk.BooleanVar(value=sound_value)

        sound_checkbox = tk.Checkbutton(
            btn_frame,
            text="Play sound on alert",
            variable=self.alert_sound,
            command=lambda: self.safe_write_to_ini("Settings", {"alert_sound": str(self.alert_sound.get())})
        )
        sound_checkbox.pack(side="left", padx=10)


        buttons_frame = tk.LabelFrame(top_frame, font=('Arial', 10, 'bold'), text="Spotting", bd=2, labelanchor="n")
        buttons_frame.grid(row=0, column=1, padx=(0, 0), pady=(0, 10), sticky="nsew")
        top_frame.columnconfigure(1, weight=1)

        self.spot_callsign_var = tk.StringVar()
        self.spot_freq_var = tk.StringVar()
        self.spot_comment_var = tk.StringVar()

        tk.Label(buttons_frame, text="Callsign:").grid(row=0, column=0, sticky="e", padx=5, pady=2)
        tk.Entry(buttons_frame, textvariable=self.spot_callsign_var).grid(row=0, column=1, padx=5, pady=2, sticky="ew")

        tk.Label(buttons_frame, text="Freq(MHz):").grid(row=1, column=0, sticky="e", padx=5, pady=2)
        tk.Entry(buttons_frame, textvariable=self.spot_freq_var).grid(row=1, column=1, padx=5, pady=2, sticky="ew")

        tk.Label(buttons_frame, text="Comment:").grid(row=2, column=0, sticky="e", padx=5, pady=2)
        tk.Entry(buttons_frame, textvariable=self.spot_comment_var).grid(row=2, column=1, padx=5, pady=2, sticky="ew")

        btn_frame = tk.Frame(buttons_frame)
        btn_frame.grid(row=3, column=0, columnspan=2, pady=5, padx=10, sticky="ew")

        send_spot_btn = tk.Button(btn_frame, text="Send Spot", command=self.send_spot, fg="white", bg="green", width=12)
        send_spot_btn.pack(side="left", expand=True, fill="x", padx=(0, 5))

        clear_btn = tk.Button(btn_frame, text="Clear", command=self.clear_spot_fields, fg="black", bg="lightgrey", width=12)
        clear_btn.pack(side="left", expand=True, fill="x", padx=(5, 0))

        # Treeview header bold
        style = ttk.Style(self.root)
        style.theme_use("clam")
        default_font = tkfont.nametofont("TkHeadingFont")
        bold_font = default_font.copy()
        bold_font.configure(weight="bold")
        style.configure("Treeview.Heading", font=bold_font)

        # Notebook
        self.notebook = ttk.Notebook(self.root)
        self.notebook.pack(padx=10, pady=5, fill=tk.BOTH, expand=True)

        self.tab_spots = tk.Frame(self.notebook)
        self.notebook.add(self.tab_spots, text="Spots")

        self.tab_alerts = tk.Frame(self.notebook)
        self.notebook.add(self.tab_alerts, text="Alerts")

        self.tab_clusters = tk.Frame(self.notebook)
        self.notebook.add(self.tab_clusters, text="Clusters")


        self.cluster_tree = ttk.Treeview(self.tab_clusters, columns=("prefix", "host", "port"), show="headings")
        self.cluster_tree.heading("prefix", text="Prefix")
        self.cluster_tree.heading("host", text="Host")
        self.cluster_tree.heading("port", text="Port")
        self.cluster_tree.column("prefix", width=70, anchor="center")
        self.cluster_tree.column("host", width=100, anchor="center")
        self.cluster_tree.column("port", width=70, anchor="center")
        self.cluster_tree.tag_configure('oddrow', background='white')
        self.cluster_tree.tag_configure('evenrow', background='#f0f0f0')
        self.cluster_tree.pack(fill=tk.BOTH, expand=True, padx=10, pady=5)
        self.cluster_tree.bind("<Double-1>", self.on_cluster_double_click)

        # Buttons for editing
        cluster_btn_frame = tk.Frame(self.tab_clusters)
        cluster_btn_frame.pack(pady=5)
        tk.Button(cluster_btn_frame, text="➕ Add", command=self.add_cluster).pack(side=tk.LEFT, padx=5)
        tk.Button(cluster_btn_frame, text="✏️ Edit", command=self.edit_cluster).pack(side=tk.LEFT, padx=5)
        tk.Button(cluster_btn_frame, text="❌ Delete", command=self.delete_cluster).pack(side=tk.LEFT, padx=5)

        # Connection settings
        conn_frame = tk.LabelFrame(self.tab_clusters, text="Connection")
        conn_frame.pack(fill="x", padx=10, pady=5)

        tk.Label(conn_frame, text="Host:").grid(row=0, column=0, sticky="e")
        self.hostport_var = tk.StringVar()
        tk.Label(conn_frame, textvariable=self.hostport_var, width=30, anchor="w", relief="sunken").grid(row=0, column=1, padx=(2, 10), sticky=tk.W)

        tk.Label(conn_frame, text="Login:").grid(row=1, column=0, sticky="e")
        self.login_var = tk.StringVar()
        tk.Entry(conn_frame, textvariable=self.login_var, width=10).grid(row=1, column=1, sticky=tk.W, pady=(0, 5))

        self.connect_button = tk.Button(conn_frame, text="Connect", command=self.toggle_connection)
        self.connect_button.grid(row=0, column=2, rowspan=2, padx=10, sticky="w")

        self.auto_reconnect_var = tk.BooleanVar()
        self.auto_reconnect_var.set(self.load_ini_setting("AutoReconnect", "0") == "1")
        self.auto_reconnect_checkbox = tk.Checkbutton(conn_frame, text="Auto reconnect", variable=self.auto_reconnect_var, command=self.save_auto_reconnect_setting)
        self.auto_reconnect_checkbox.grid(row=0, column=3, rowspan=2, sticky="w", pady=(0, 5))

        if self.load_ini_setting("AutoReconnect", "") == "":
            self.save_auto_reconnect_setting()

        last_used = self.load_last_used_cluster()
        if last_used:
            self.hostport_var.set(last_used)
        if self.auto_reconnect_var.get():
            self.manual_disconnect = False
            self.root.after(100, self.connect)

        self.tab_output = tk.Frame(self.notebook)
        self.notebook.add(self.tab_output, text="Console")

        # ==================== SPOTS TAB (Grid lay-out) ====================
        # Make the Spots tab grid-resizable
        self.tab_spots.grid_rowconfigure(0, weight=1)
        self.tab_spots.grid_columnconfigure(0, weight=1)

        # Container for Treeview + bottom
        spots_container = tk.Frame(self.tab_spots)
        spots_container.grid(row=0, column=0, sticky="nsew")

        # Treeview gets the rack space
        spots_container.grid_rowconfigure(0, weight=1)
        spots_container.grid_columnconfigure(0, weight=1)
        # Reserve a minimum height for the bottom frame so that it remains visible
        spots_container.grid_rowconfigure(1, weight=0, minsize=46)  # customize to your liking

        # ---- Treeview ----
        tree_frame = tk.Frame(spots_container)
        tree_frame.grid(row=0, column=0, sticky="nsew")

        tree_scrollbar = tk.Scrollbar(tree_frame)
        tree_scrollbar.pack(side=tk.RIGHT, fill=tk.Y)

        self.tree = ttk.Treeview(
            tree_frame,
            columns=("time", "freq", "dx", "country", "spotter", "comment"),
            show="headings",
            yscrollcommand=tree_scrollbar.set,
            height=1,   # minimum height in rows -> prevents large min-height
        )
        self.tree.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        tree_scrollbar.config(command=self.tree.yview)

        columns = {
            "time": {"width": 30},
            "freq": {"width": 50},
            "dx": {"width": 60},
            "country": {"width": 100},
            "spotter": {"width": 50},
            "comment": {"width": 150},
        }
        for col, options in columns.items():
            self.tree.heading(col, text=col.upper())
            self.tree.column(col, anchor=tk.CENTER, width=options["width"])

        self.tree.tag_configure('oddrow', background='white')
        self.tree.tag_configure('evenrow', background='#f0f0f0')
        self.tree.tag_configure('worked_today', background='#ffd9b3')
        self.tree.tag_configure('worked', background='#b3e6ff')
        self.tree.tag_configure('owncall', background='#c6f5c6')
        self.tree.tag_configure('alert', background='#ff9999')


        # ==================== Context menu for Spots Tree ====================
        self.spot_menu = tk.Menu(self.root, tearoff=0)
        self.spot_menu.add_command(label="Add this DX to Alerts", command=self.add_spot_to_alerts)

        # Bind left-click to spot_clicked (MiniBook)
        self.tree.bind("<Button-1>", self.spot_clicked)

        # Bind right-click to context menu (Alerts)
        self.tree.bind("<Button-3>", self.show_spot_context_menu)


        # ---- Bottom frame: always remains visible ----
        bottom_frame = tk.Frame(spots_container)
        bottom_frame.grid(row=1, column=0, sticky="ew", padx=10, pady=(5, 10))

        legend_left = tk.Frame(bottom_frame)
        legend_left.pack(side="left")

        tk.Label(legend_left, text="Worked before", bg="#b3e6ff", relief="ridge", borderwidth=1, width=15).pack(side="left", padx=5)
        tk.Label(legend_left, text="Worked today",  bg="#ffd9b3", relief="ridge", borderwidth=1, width=15).pack(side="left", padx=5)
        tk.Label(legend_left, text="Self spot",     bg="#c6f5c6", relief="ridge", borderwidth=1, width=15).pack(side="left", padx=5)
        tk.Label(legend_left, text="Alert",         bg="#ff9999", relief="ridge", borderwidth=1, width=15).pack(side="left", padx=5)


        exit_right = tk.Frame(bottom_frame)
        exit_right.pack(side="right")
        tk.Button(exit_right, text="Exit", command=self.on_close, fg="black", bg="lightgrey", width=12).pack(padx=10)

        # Console tab
        self.output = tk.Text(self.tab_output, fg="white", bg="black", height=10, width=80)
        self.output.pack(fill=tk.BOTH, expand=True, padx=10, pady=5)

        # ==================== ALERTS TREE (same columns & style as Spots) ====================
        alerts_container = tk.Frame(self.tab_alerts)
        alerts_container.pack(fill=tk.BOTH, expand=True)

        alerts_tree_frame = tk.Frame(alerts_container)
        alerts_tree_frame.pack(fill=tk.BOTH, expand=True)

        alerts_scrollbar = tk.Scrollbar(alerts_tree_frame)
        alerts_scrollbar.pack(side=tk.RIGHT, fill=tk.Y)

        self.alerts_tree = ttk.Treeview(
            alerts_tree_frame,
            columns=("time", "freq", "dx", "country", "spotter", "comment"),
            show="headings",
            yscrollcommand=alerts_scrollbar.set,
            height=1
        )
        self.alerts_tree.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        alerts_scrollbar.config(command=self.alerts_tree.yview)

        for col, options in columns.items():
            self.alerts_tree.heading(col, text=col.upper())
            self.alerts_tree.column(col, anchor=tk.CENTER, width=options["width"])

        # Tags (zelfde kleuren als Spots)
        self.alerts_tree.tag_configure('oddrow', background='white')
        self.alerts_tree.tag_configure('evenrow', background='#f0f0f0')
        self.alerts_tree.tag_configure('worked_today', background='#ffd9b3')
        self.alerts_tree.tag_configure('worked', background='#b3e6ff')
        self.alerts_tree.tag_configure('owncall', background='#c6f5c6')

        # Klikgedrag
        self.alerts_tree.bind("<<TreeviewSelect>>", self.alert_clicked)
        self.alerts_tree.bind("<Double-1>", self.alert_clicked)


    def clear_spot_fields(self):
        """Clears the spot entry fields: callsign, frequency, comment."""
        self.spot_callsign_var.set("")
        self.spot_freq_var.set("")
        self.spot_comment_var.set("")

    # Shows a little help windows how to use REGEX
    def show_regex_help(self):

        # voorkomen dat er meerdere vensters openen
        if hasattr(self, "_open_windows") and self._open_windows.get("regex_help"):
            win = self._open_windows["regex_help"]
            if win and win.winfo_exists():
                win.lift()
                win.focus_force()
                return

        win = tk.Toplevel(self.root)
        self._open_windows["regex_help"] = win

        def on_close():
            self._open_windows["regex_help"] = None
            win.destroy()

        win.protocol("WM_DELETE_WINDOW", on_close)
        win.title("Regex Help")
        self.center_window(win, 600, 400)

        text = tk.Text(win, wrap="word")
        text.pack(expand=True, fill="both", padx=10, pady=10)

        help_content = """
        REGEX FILTER GUIDE
        --------------------------

        EXAMPLES (CALLSIGNS):
        ^I             — starts with 'I' (e.g. Italy: I0AAA)
        ^DL            — starts with 'DL' (e.g. Germany: DL1ABC)
        \\bREF\\b        — exact word 'REF' as a standalone word
        ^F.*           — starts with 'F' (e.g. France: F4XYZ)
        ^OH[0-9]       — Finnish station (OH + digit, e.g. OH2ZZ)
        [A-Z]{1,2}[0-9]{1,2}[A-Z]{1,3}  — generic callsign format (e.g. PA3XYZ)
        .*25.*         — contains '25' anywhere (e.g. LZ25AA)
        [IU][0-9]      — starts with 'I' or 'U' followed by a digit
        .*/P$          — ends with /P (portable station)
        .*/MM$         — ends with /MM (maritime mobile)
        ^EA[1-9]/      — Spanish prefix with slash (e.g. EA1/ON4ZZZ)
        ^ON[0-9]{1}.*  — Belgian station with 1 digit (e.g. ON3AAA)
        .*\\/.*         — contains a slash (e.g. PA3XYZ/P)
        ^HB9           — Swiss station (e.g. HB9ABC)
        ^JA[0-9]       — Japanese callsign (e.g. JA1XYZ)
        ^ZS[1-6]       — South Africa (e.g. ZS6AAA)
        ^VK[1-9]       — Australia (e.g. VK2ABC)
        ^K[0-9]        — USA (e.g. K3LR, K9CT)
        .*(\\bYL\\b|\\bOM\\b).*  — contains YL or OM as whole words
        ^3D2.*         — Fiji Islands (e.g. 3D2XYZ)
        ^(5Z|9J|9G)    — Kenya, Zambia or Ghana (multiple prefixes)

        SYMBOLS
        --------
        ^    — start of line
        $    — end of line
        .    — any character
        *    — zero or more times
        +    — one or more times
        []   — character set (e.g. [A-Z])
        {}   — repeat count (e.g. {2} = exactly 2 times)
        \\b   — word boundary
        |    — OR (e.g. PA|PD)

        NOTES
        ------
        [A-Z]{1,2}[0-9]{1,2}[A-Z]{1,3}
        Matches most standard callsign formats:
        - 1–2 letters, 1–2 digits, 1–3 letters
        - Examples: PA3XYZ, ON4ZZ, DL1A, K9AA

        Use https://regex101.com to test your expressions.
        """
        text.insert("1.0", help_content)
        text.config(state="disabled", font=("Courier New", 10))

        tk.Button(win, text="Close", command=on_close).pack(pady=5)




    # Function to check if cty.dat file is present in the root folder
    def check_ctydat_file(self):
        try:
            if not os.path.exists(DXCC_FILE):
                messagebox.showinfo(title="File Not Found",
                                    message="The file cty.dat was not found. It will now be downloaded.",
                                    parent=self.root)
                self.download_ctydat_file()
            self.dxcc_data = parse_cty_file(DXCC_FILE)
        except Exception as e:
            messagebox.showerror(title="Error",
                                message=f"Error loading data: {e}",
                                parent=self.root)
            self.dxcc_data = {}

    # Function to download cty.dat file
    def download_ctydat_file(self):
        try:
            response = requests.get(ctydat_url)
            response.raise_for_status()
            with open(DXCC_FILE, "wb") as f:
                f.write(response.content)
            messagebox.showinfo(title="Download",
                                message="The file cty.dat has been downloaded successfully.",
                                parent=self.root)
            self.reconnect()                                
        except Exception as e:
            messagebox.showerror(title="Download Error",
                                message=f"Failed to download file: {e}",
                                parent=self.root)



    # Sends spot to Telnet cluster server
    def send_spot(self):
        callsign = self.spot_callsign_var.get().strip()
        freq_str = self.spot_freq_var.get().strip()
        comment = self.spot_comment_var.get().strip()

        if not callsign or not freq_str:
            messagebox.showwarning("Missing data", "Please enter both callsign and frequency.")
            return

        try:
            # Zet MHz naar kHz
            freq_mhz = float(freq_str)
            freq_khz = int(round(freq_mhz * 1000))

            spot_cmd = f"DX {freq_khz} {callsign} {comment}\n"
            if self.writer:
                self.writer.write(spot_cmd.encode())
                asyncio.run_coroutine_threadsafe(self.writer.drain(), self.loop)
                self.output.insert(tk.END, f"Sent spot: {spot_cmd}")
                self.output.see(tk.END)
                self.clear_spot_fields()
        except ValueError:
            messagebox.showerror("Error", f"Invalid frequency: {freq_str}")
        except Exception as e:
            messagebox.showerror("Error", f"Failed to send spot:\n{e}")



    async def _async_send(self, message):
        if self.writer is not None:
            self.writer.write(message.encode())
            await self.writer.drain()

    # Auto reconnect setting
    def save_auto_reconnect_setting(self):
        self.safe_write_to_ini("Settings", {
            "AutoReconnect": "1" if self.auto_reconnect_var.get() else "0"
        })


    # Loads Settings from ini file
    def load_ini_setting(self, key, default=""):
        config = configparser.ConfigParser()
        config.optionxform = str
        if os.path.exists(INI_FILE):
            config.read(INI_FILE)
            return config.get("Settings", key, fallback=default)
        return default

    
    def reconnect(self):
        if self.connected:
            return
        self.manual_disconnect = False
        self.connect()


    def on_cluster_double_click(self, event=None):
        selected = self.cluster_tree.selection()
        if not selected:
            return
        values = self.cluster_tree.item(selected[0], "values")
        if len(values) >= 3:
            prefix, host, port = values
            if self.connected:
                self.disconnect()
            self.hostport_var.set(f"{host}:{port}")

            

    def save_clusters_to_file(self):
        try:
            with open(CLUSTER_FILE, "w", encoding="utf-8") as f:
                json.dump(self.clusters_data, f, indent=2)
            self.load_host_file()
        except Exception as e:
            messagebox.showerror("Save Error", str(e))

    def add_cluster(self):
        self.add_cluster_dialog()


    def edit_cluster(self):
        selected = self.cluster_tree.selection()
        if not selected:
            return

        values = self.cluster_tree.item(selected[0], "values")
        self.edit_cluster_dialog(values=values)



    def delete_cluster(self):
        selected = self.cluster_tree.selection()
        if not selected:
            return

        prefix, host, port = self.cluster_tree.item(selected[0], "values")

        confirm = tk.Toplevel(self.root)
        confirm.title("Delete Cluster")
        confirm.transient(self.root)
        confirm.grab_set()
        self.center_window(confirm, 340, 130)

        msg = tk.Label(confirm, text=f"Delete cluster\n{host}:{port} ?", pady=10)
        msg.pack()

        def do_delete():
            self.clusters_data = [
                c for c in self.clusters_data
                if not (c["host"] == host and str(c["port"]) == str(port))
            ]
            self.save_clusters_to_file()
            confirm.grab_release()
            confirm.destroy()

        def cancel():
            confirm.grab_release()
            confirm.destroy()

        btn_frame = tk.Frame(confirm)
        btn_frame.pack(pady=5)

        tk.Button(btn_frame, text="Yes", width=10, command=do_delete).pack(side="left", padx=10)
        tk.Button(btn_frame, text="No", width=10, command=cancel).pack(side="right", padx=10)

    def add_cluster_dialog(self):
        key = "add_cluster_dialog"

        if not hasattr(self, "_open_windows"):
            self._open_windows = {}

        # Single-instance
        if key in self._open_windows and self._open_windows[key] and self._open_windows[key].winfo_exists():
            w = self._open_windows[key]
            w.lift()
            w.focus_force()
            return

        win = tk.Toplevel(self.root)
        self._open_windows[key] = win

        def on_close():
            self._open_windows[key] = None
            win.destroy()

        win.protocol("WM_DELETE_WINDOW", on_close)
        win.title("Add Cluster")
        win.resizable(False, False)

        # ---------------------------------------------------
        # VASTE VENSTERGROOTTE
        # ---------------------------------------------------
        FIXED_W = 300
        FIXED_H = 150
        self.center_window(win, FIXED_W, FIXED_H)

        # ---------------------------------------------------
        # GRID CONFIG → GECENTREERDE UI
        # ---------------------------------------------------
        win.grid_columnconfigure(0, weight=1)
        win.grid_columnconfigure(1, weight=0)
        win.grid_columnconfigure(2, weight=1)
        win.grid_columnconfigure(3, weight=1)
        win.grid_columnconfigure(4, weight=1)

        entry_width = 20
        pady = 4

        # ---------- PREFIX ----------
        tk.Label(win, text="Prefix:").grid(row=0, column=1, sticky="e", padx=4, pady=pady)
        prefix_var = tk.StringVar()
        tk.Entry(win, textvariable=prefix_var, width=entry_width).grid(
            row=0, column=2, columnspan=2, sticky="we"
        )

        # ---------- HOST ----------
        tk.Label(win, text="Host:").grid(row=1, column=1, sticky="e", padx=4, pady=pady)
        host_var = tk.StringVar()
        tk.Entry(win, textvariable=host_var, width=entry_width).grid(
            row=1, column=2, columnspan=2, sticky="we"
        )

        # ---------- PORT ----------
        tk.Label(win, text="Port:").grid(row=2, column=1, sticky="e", padx=4, pady=pady)
        port_var = tk.StringVar()
        tk.Entry(win, textvariable=port_var, width=entry_width).grid(
            row=2, column=2, columnspan=2, sticky="we"
        )

        # ---------------------------------------------------
        # SAVE BUTTON (GECENTREERD)
        # ---------------------------------------------------
        def save():
            prefix = prefix_var.get().strip()
            host = host_var.get().strip()
            port_str = port_var.get().strip()

            if not prefix or not host or not port_str:
                messagebox.showerror("Invalid", "All fields are required.", parent=win)
                return

            try:
                port = int(port_str)
            except ValueError:
                messagebox.showerror("Invalid", "Port must be a number.", parent=win)
                return

            new_entry = {"prefix": prefix, "host": host, "port": port}
            self.clusters_data.append(new_entry)
            self.save_clusters_to_file()

            on_close()

        tk.Button(win, text="Save", width=12, command=save).grid(
            row=3, column=0, columnspan=5, pady=10
        )


    def edit_cluster_dialog(self, values=None):
        key = "cluster_dialog"

        if not hasattr(self, "_open_windows"):
            self._open_windows = {}

        if key in self._open_windows and self._open_windows[key] and self._open_windows[key].winfo_exists():
            self._open_windows[key].lift()
            self._open_windows[key].focus_force()
            return

        win = tk.Toplevel(self.root)
        self._open_windows[key] = win

        def on_close():
            self._open_windows[key] = None
            win.destroy()

        win.protocol("WM_DELETE_WINDOW", on_close)
        win.title("Edit Cluster" if values else "Add Cluster")
        win.resizable(False, False)

        FIXED_W = 300
        FIXED_H = 150
        self.center_window(win, FIXED_W, FIXED_H)

        # ---------------------------------------------------
        # GRID CONFIG → MAAKT ALLES MOOI GECENTREERD
        # ---------------------------------------------------
        win.grid_columnconfigure(0, weight=1)   # leeg links
        win.grid_columnconfigure(1, weight=0)   # labels
        win.grid_columnconfigure(2, weight=1)   # midden
        win.grid_columnconfigure(3, weight=1)   # entries
        win.grid_columnconfigure(4, weight=1)   # leeg rechts

        entry_width = 26
        pady = 6

        # ---------------------------------------------------
        # PREFIX
        # ---------------------------------------------------
        tk.Label(win, text="Prefix:").grid(row=0, column=1, sticky="e", pady=pady, padx=4)
        prefix_var = tk.StringVar(value=values[0] if values else "")
        tk.Entry(win, textvariable=prefix_var, width=entry_width).grid(
            row=0, column=2, columnspan=2, sticky="we", pady=pady
        )

        # ---------------------------------------------------
        # HOST
        # ---------------------------------------------------
        tk.Label(win, text="Host:").grid(row=1, column=1, sticky="e", pady=pady, padx=4)
        host_var = tk.StringVar(value=values[1] if values else "")
        tk.Entry(win, textvariable=host_var, width=entry_width).grid(
            row=1, column=2, columnspan=2, sticky="we", pady=pady
        )

        # ---------------------------------------------------
        # PORT
        # ---------------------------------------------------
        tk.Label(win, text="Port:").grid(row=2, column=1, sticky="e", pady=pady, padx=4)
        port_var = tk.StringVar(value=values[2] if values else "")
        tk.Entry(win, textvariable=port_var, width=entry_width).grid(
            row=2, column=2, columnspan=2, sticky="we", pady=pady
        )

        # ---------------------------------------------------
        # SAVE BUTTON (GECENTREERD)
        # ---------------------------------------------------
        def save():
            prefix = prefix_var.get().strip()
            host = host_var.get().strip()

            try:
                port = int(port_var.get().strip())
            except ValueError:
                messagebox.showerror("Invalid", "Port must be a number", parent=win)
                return

            if not host:
                messagebox.showerror("Invalid", "Host is required", parent=win)
                return

            new_entry = {"prefix": prefix, "host": host, "port": port}

            if values:
                old_host = values[1]
                old_port = str(values[2])
                self.clusters_data = [
                    c for c in self.clusters_data
                    if not (c["host"] == old_host and str(c["port"]) == old_port)
                ]

            self.clusters_data.append(new_entry)
            self.save_clusters_to_file()

            on_close()

        tk.Button(win, text="Save", width=12, command=save).grid(
            row=3, column=0, columnspan=5, pady=12
        )





    def load_alert_callsigns(self):
        """Loads saved alert callsigns from alert_filter.json."""
        path = os.path.join(os.path.dirname(__file__), ALERT_FILTER)
        try:
            if os.path.exists(path):
                with open(path, "r", encoding="utf-8") as f:
                    data = json.load(f)
                    if isinstance(data, list):
                        return data
            return []
        except Exception as e:
            print(f"Error in spot_clicked_from_popup: {e}")
            return []

    def save_alert_callsigns(self):
        """Saves alert callsigns to alert_filter.json."""
        path = os.path.join(os.path.dirname(__file__), ALERT_FILTER)
        try:
            with open(path, "w", encoding="utf-8") as f:
                json.dump(self.alert_callsigns, f, indent=2)
        except Exception as e:
            print(f"Error saving alert_filter.json: {e}")

    def edit_alert_filter(self):
        """Open a window to edit the comma-separated callsign list."""
        win = tk.Toplevel(self.root)
        win.title("Edit Callsign Alert Filter")
        win.geometry("480x360")
        win.minsize(480, 360)
        win.configure(bg="#f2f2f2")
        win.grab_set()
        self.center_window(win, 480, 360)

        # Title and explanation
        lbl = tk.Label(
            win,
            text="Enter your callsigns here, separated by commas:",
            font=("Arial", 10, "bold"),
            bg="#f2f2f2"
        )
        lbl.pack(pady=(10, 5))

        # Frame for text field + scroll bar
        frame_text = tk.Frame(win, bg="#f2f2f2")
        frame_text.pack(fill="both", expand=True, padx=10, pady=5)

        text_scroll = tk.Scrollbar(frame_text)
        text_scroll.pack(side="right", fill="y")

        text = tk.Text(
            frame_text,
            wrap="word",
            yscrollcommand=text_scroll.set,
            font=("Consolas", 11),
            height=12,
            bg="white",
            relief="sunken",
            bd=1
        )
        text.pack(side="left", fill="both", expand=True)
        text_scroll.config(command=text.yview)

        # Post existing list
        if self.alert_callsigns:
            text.insert("1.0", ", ".join(self.alert_callsigns))

        # Function: store
        def save_and_close():
            content = text.get("1.0", "end").strip()

            # --- If field is empty, treat it as an empty list ---
            if not content:
                self.alert_callsigns = []
                self.save_alert_callsigns()
                messagebox.showinfo("Filter saved", "The alert filter has been cleared (no callsigns).")
                win.destroy()
                return

            # Disallow obvious wrong separators
            disallowed = set(";:.|\\\t")
            if any(ch in content for ch in disallowed):
                messagebox.showwarning(
                    "Invalid separator",
                    "Only commas are allowed as separators.\nDo not use semicolons, dots, pipes, backslashes, colons or tabs.\n\n"
                    "Example: PA6Y, PD5DJ, PD5DJ/P"
                )
                return

            # Split on commas (allow spaces after commas)
            parts = [p.strip() for p in content.split(",")]
            parts = [p for p in parts if p]  # remove empties

            # Validate each callsign token
            import re
            token_re = re.compile(r"^[A-Za-z0-9]+(?:/[A-Za-z0-9]+){0,2}$")

            for tok in parts:
                if " " in tok:
                    messagebox.showwarning(
                        "Invalid format",
                        f"Invalid callsign '{tok}': spaces are only allowed after commas, not inside a callsign."
                    )
                    return
                if not token_re.match(tok):
                    messagebox.showwarning(
                        "Invalid callsign",
                        f"Invalid callsign '{tok}'. Callsigns must consist of letters/digits and up to two slashes (e.g. PREFIX/CALL/SUFFIX).\n"
                        "Example: PA6Y, PD5DJ, PD5DJ/P, 3D2/PA3XYZ"
                    )
                    return

            # Save uppercase callsigns
            self.alert_callsigns = [c.upper() for c in parts]
            self.save_alert_callsigns()
            messagebox.showinfo("Filter saved", "The new callsigns have been saved.")
            win.destroy()

        # Function: cancel
        def cancel():
            win.destroy()

        # Buttons at the bottom
        btn_frame = tk.Frame(win, bg="#f2f2f2")
        btn_frame.pack(fill="x", side="bottom", pady=10)

        btn_save = tk.Button(
            btn_frame,
            text="💾 Save",
            bg="#2e8b57",
            fg="white",
            font=("Arial", 10, "bold"),
            width=12,
            command=save_and_close
        )
        btn_save.pack(side="left", expand=True, padx=(60, 10))

        btn_cancel = tk.Button(
            btn_frame,
            text="Cancel",
            bg="#d9d9d9",
            font=("Arial", 10),
            width=12,
            command=cancel
        )
        btn_cancel.pack(side="right", expand=True, padx=(10, 60))






    def update_treeview_colors(self):
        """
        Ensures that:
        - Rows with special status ('owncall', 'worked_today', 'worked') are given that color
        - All other rows retain an alternate (even/odd) background
        Call this function after adding/removing/filtering rows.
        """
        worked_calls_today = self.get_worked_calls_today() if self.get_worked_calls_today else set()
        current_worked_calls = self.get_worked_calls() if self.get_worked_calls else set()

        children = self.tree.get_children()
        for i, item in enumerate(children):
            try:
                values = self.tree.item(item, "values")
                dx_call = values[2].upper() if len(values) >= 3 and values[2] is not None else ""
            except Exception:
                dx_call = ""

            # check if own call (case-insensitive)
            callmatch = None
            if getattr(self, "user_callsign", None):
                try:
                    callmatch = re.fullmatch(rf"(.*[/])?{re.escape(self.user_callsign)}([/].*)?", dx_call, flags=re.IGNORECASE)
                except re.error:
                    callmatch = None

            # Highlight lines are prioritized; otherwise even/odd
            if callmatch:
                tag = "owncall"
            elif dx_call in [c.upper() for c in self.alert_callsigns]:
                tag = "alert"
            elif dx_call in worked_calls_today:
                tag = "worked_today"
            elif dx_call in current_worked_calls:
                tag = "worked"
            else:
                tag = "evenrow" if (i % 2) == 0 else "oddrow"


            # set the tag (overwrites previous tags)
            self.tree.item(item, tags=(tag,))


    def insert_spot_row(self, spot, at_top=True):
        """
        Helps you insert a single spot and then refresh the colors.
        - spot: dict with keys time, freq, dx, country, spotter, comment (same structure as in your code)
        - at_top: if True, insert at the top (index 0), otherwise append (end)
        This function doesn't replace the existing insert_spot, but you can modify 'add_spot_to_treeview'
        to use it.
        """
        try:
            freq_str = f"{float(spot.get('freq', '')):.4f}" if spot.get('freq', '') != "" else spot.get('freq', '')
        except Exception:
            freq_str = spot.get('freq', '')

        values = (
            spot.get("time", ""),
            freq_str,
            spot.get("dx", ""),
            spot.get("country", ""),
            spot.get("spotter", ""),
            spot.get("comment", "")
        )

        if at_top:
            # index 0 = on top
            self.tree.insert("", 0, values=values)
        else:
            self.tree.insert("", "end", values=values)

        # if necessary, limit to 100 rows (as you did before)
        children = self.tree.get_children()
        if len(children) > 100:
            self.tree.delete(children[-1])

        # recalculate tags/colors for ALL rows
        self.update_treeview_colors()


    def insert_alert_row(self, spot, at_top=True):
        """Adds spot in Alerts-tree (newest at top, max 100 spots)."""
        try:
            freq_str = f"{float(spot.get('freq','')):.4f}" if spot.get('freq','') else spot.get('freq','')
        except Exception:
            freq_str = spot.get('freq','')

        values = (
            spot.get("time",""),
            freq_str,
            spot.get("dx",""),
            spot.get("country",""),
            spot.get("spotter",""),
            spot.get("comment","")
        )

        try:
            if at_top:
                self.alerts_tree.insert("", 0, values=values)
            else:
                self.alerts_tree.insert("", "end", values=values)

            # max 100
            children = self.alerts_tree.get_children()
            if len(children) > 100:
                self.alerts_tree.delete(children[-1])

            # kleurentags
            for i, item in enumerate(self.alerts_tree.get_children()):
                vals = self.alerts_tree.item(item, "values")
                dx_call = vals[2].upper() if len(vals) > 2 else ""
                tag = "evenrow" if (i % 2 == 0) else "oddrow"
                self.alerts_tree.item(item, tags=(tag,))
            
            # Play sound only if the alert sound option is enabled
            if getattr(self, "alert_sound", None) and self.alert_sound.get():
                winsound.MessageBeep(winsound.MB_ICONEXCLAMATION)

        except Exception:
            pass



    def alert_clicked(self, event):
        """Handle selection (or double-click) in the Alerts tree.
        Sends callsign+freq+mode to MiniBook (on_callsign_selected) and, if tracking is enabled,
        sends frequency and mode to rig via rigctl.
        """
        try:
            # get selected item(s)
            selected = self.alerts_tree.selection()
            if not selected:
                return

            # Use the first selected row
            values = self.alerts_tree.item(selected[0])["values"]
            # values layout: (time, freq, dx, country, spotter, comment)
            raw_freq = values[1] if len(values) > 1 else ""
            to_call = values[2] if len(values) > 2 else ""

            # normalize freq to string and attempt to parse to Hz
            try:
                # allow freq like "14.0740" or "14074" etc.
                freq_mhz = float(raw_freq)
                hz = int(freq_mhz * 1_000_000)
                freq_mhz_str = f"{freq_mhz:.5f}"
            except Exception:
                hz = None
                freq_mhz_str = ""
            
            # determine GUI mode / resolved mode
            gui_mode = self.mode_var.get() if hasattr(self, "mode_var") else "ALL"
            if gui_mode == "PHONE" or gui_mode == "ALL":
                # USB if above ~10 MHz (10000000 Hz), else LSB
                resolved_mode = "USB" if (hz and hz > 10_000_000) else "LSB"
            else:
                resolved_mode = gui_mode

            # call MiniBook callback if available
            if self.on_callsign_selected:
                try:
                    # prefer full signature (callsign, freq, mode)
                    self.on_callsign_selected(to_call, freq=freq_mhz_str, mode=resolved_mode)
                except TypeError:
                    # fallback if older MiniBook version expects only callsign
                    try:
                        self.on_callsign_selected(to_call)
                    except Exception:
                        pass
                except Exception:
                    pass

            # if tracking is not enabled (or tracking_var missing) do not talk to rig
            if not (self.tracking_var and self.tracking_var.get()):
                return

            # send frequency + mode to rig via rigctl if we have a valid hz
            if hz:
                try:
                    with socket.create_connection((self.rigctl_host, self.rigctl_port), timeout=2) as s:
                        s.sendall(f"F {hz}\n".encode())
                        _ = s.recv(1024)
                        if resolved_mode and resolved_mode not in ("ALL", "DIGITAL"):
                            s.sendall(f"M {resolved_mode} -1\n".encode())
                            _ = s.recv(1024)
                except Exception:
                    # don't propagate exceptions; fail silently as the main app does
                    pass

        except Exception:
            # swallow any unexpected error to avoid crashing UI on click
            pass


    def show_spot_context_menu(self, event):
        """Show right-click menu only when a valid row in the Spots Tree is clicked."""
        try:
            row_id = self.tree.identify_row(event.y)
            if not row_id:
                return  # No valid row -> do nothing

            # Select clicked row (so user sees which one they right-clicked)
            self.tree.selection_set(row_id)
            self.selected_spot = self.tree.item(row_id, "values")

            # Show context menu at cursor position
            self.spot_menu.tk_popup(event.x_root, event.y_root)
        finally:
            self.spot_menu.grab_release()



    def add_spot_to_alerts(self):
        """Add the selected DX call from the Spots Tree to alert_filter.json."""
        try:
            if not hasattr(self, "selected_spot") or not self.selected_spot:
                return

            dx_call = self.selected_spot[2].strip().upper()
            if not dx_call:
                return

            alert_file = ALERT_FILTER
            alerts = []

            # Load existing alerts if file exists
            if os.path.exists(alert_file):
                with open(alert_file, "r", encoding="utf-8") as f:
                    alerts = json.load(f)

            # Only add if not already present
            if dx_call not in [a.upper() for a in alerts]:
                alerts.append(dx_call)
                with open(alert_file, "w", encoding="utf-8") as f:
                    json.dump(alerts, f, indent=2)

                # Reload list into memory
                self.alert_callsigns = alerts

                # Optional feedback
                messagebox.showinfo("Alert Added", f"{dx_call} added to alert list.")
            else:
                messagebox.showinfo("Already Exists", f"{dx_call} is already in the alert list.")
        except Exception as e:
            messagebox.showerror("Error", f"Failed to add alert: {e}")




    def clear_alerts(self):
        """Ask for confirmation and clear all entries from the Alerts TreeView."""
        confirm = messagebox.askyesno(
            "Confirm",
            "Are you sure you want to clear all Alert spots?",
            parent=self.root
        )
        if confirm:
            # Delete all items from the Alerts TreeView
            for item in self.alerts_tree.get_children():
                self.alerts_tree.delete(item)



    def schedule_worked_calls_check(self):
        current = self.get_worked_calls_today() if self.get_worked_calls_today else set()
        if current != self._last_worked_calls_today:
            self._last_worked_calls_today = current
            self.update_treeview_colors()
        self.root.after(10000, self.schedule_worked_calls_check)







    def load_host_file(self, default=CLUSTER_FILE):
        try:
            # Doesn't the cluster file exist? Create a default one
            if not os.path.exists(default):
                default_data = [
                    {"prefix": "PI4CC", "host": "dcx.pi4cc.nl", "port": 8000}
                ]
                with open(default, "w", encoding="utf-8") as f:
                    json.dump(default_data, f, indent=4)

            with open(default, "r", encoding="utf-8") as f:
                data = json.load(f)

            self.hosts = []
            self.clusters_data = []

            if hasattr(self, 'cluster_tree'):
                self.cluster_tree.delete(*self.cluster_tree.get_children())

            for idx, item in enumerate(data):
                if isinstance(item, dict):
                    prefix = item.get("prefix", "")
                    host = item.get("host", "")
                    port = item.get("port", 0)
                    if host and port:
                        self.clusters_data.append({"prefix": prefix, "host": host, "port": port})
                        self.hosts.append(f"{host}:{port}")
                        if hasattr(self, 'cluster_tree'):
                            tag = 'evenrow' if idx % 2 == 0 else 'oddrow'
                            self.cluster_tree.insert("", "end", values=(prefix, host, port), tags=(tag,))

            if self.hosts and not self.hostport_var.get():
                self.hostport_var.set(self.hosts[0])

        except Exception as e:
            messagebox.showerror("Host File Error", str(e))








    # Last used cluster storage
    def save_last_used_cluster(self, host, port, login):
        config = configparser.ConfigParser()
        config.optionxform = str

        # Read existing INI (if any)
        if os.path.exists(INI_FILE):
            config.read(INI_FILE, encoding="utf-8")

        # Add sections if they don't already exist
        if "LAST" not in config:
            config["LAST"] = {}
        if "LOGIN" not in config:
            config["LOGIN"] = {}

        # Enter values
        config["LAST"]["host"] = host
        config["LAST"]["port"] = str(port)
        config["LOGIN"]["user"] = login

        # Write back to ini
        with open(INI_FILE, "w", encoding="utf-8") as f:
            config.write(f)



    # Last used cluster retrieval
    def load_last_used_cluster(self):
        config = configparser.ConfigParser()
        config.optionxform = str
        if not os.path.exists(INI_FILE):
            return None
        config.read(INI_FILE, encoding="utf-8")

        host = ""
        port = ""
        login = ""

        if "LAST" in config:
            host = config["LAST"].get("host", "")
            port = config["LAST"].get("port", "")
        if "LOGIN" in config:
            login = config["LOGIN"].get("user", "")

        if host and port:
            hostport = f"{host}:{port}"
            self.hostport_var.set(hostport)
            self.login_var.set(login)
            return hostport
        return None







    def select_hostport(self, event=None):
        value = self.hostport_var.get()
        if ':' in value:
            host, port_str = value.split(':', 1)
            self.host_var = host.strip()
            try:
                port = int(port_str.strip())
                self.port_var = port
            except ValueError:
                self.port_var = None
        else:
            self.host_var = value.strip()
            self.port_var = None




    def select_host(self, event=None):
        index = self.host_menu.current()
        if index >= 0 and index < len(self.hosts):
            host = self.hosts[index]
            self.host_var.set(host["host"])
            self.port_var.set(str(host["port"]))





    def toggle_connection(self):
        if self.connected:
            self.disconnect()
        else:
            self.connect()





    def connect(self):
        if self.connected:
            return  # Avoid duplicate connections

        self.manual_disconnect = False  # reset at explicit connect
        hostport = self.hostport_var.get()
        login = self.login_var.get()
        self.user_callsign = login.strip().upper()

        if ':' not in hostport:
            messagebox.showerror("Error", "Host:Port must be in format hostname:port")
            return

        host, port_str = hostport.split(":", 1)
        try:
            port = int(port_str)
        except ValueError:
            messagebox.showerror("Error", "Port must be a number")
            return

        self.output.insert(tk.END, f"Connecting to {host}:{port}...\n")
        self.output.see(tk.END)

        future = asyncio.run_coroutine_threadsafe(self.telnet_client(host, port, login), self.loop)
        self.save_last_used_cluster(host, port, login)
        self.connect_button.config(text="Disconnect")


    def disconnect(self):
        self.output.insert(tk.END, "[DEBUG] Disconnect called.\n")
        self.output.insert(tk.END, f"[DEBUG] connected={self.connected}, writer={self.writer is not None}, manual_disconnect={self.manual_disconnect}, auto_reconnect={self.auto_reconnect_var.get()}\n")
        self.output.see(tk.END)

        self.manual_disconnect = True

        if self.writer:
            self.writer.close()
            asyncio.run_coroutine_threadsafe(self.writer.wait_closed(), self.loop)

        self.writer = None
        self.connected = False
        
        self.connect_button.config(text="Connect")

        self.auto_reconnect_var.set(False)
        self.auto_reconnect_checkbox.update()
        self.save_auto_reconnect_setting()
        self.output.insert(tk.END, "[INFO] Auto-reconnect disabled due to manual disconnect.\n")
        self.output.see(tk.END)



       







    async def telnet_client(self, host, port, login):
        """
        Connects to the DX Cluster via Telnet, buffers initial SH/DX spots in the background,
        builds the TreeView once (sorted newest-first), then processes live spots smoothly in real time.
        """
        self.reconnect_attempts = 0

        while True:
            try:
                reader, writer = await asyncio.open_connection(host, port)
                self.writer = writer
                self.output.insert(tk.END, f"✅ Connected to {host}:{port}\n")
                self.output.see(tk.END)

                prompt = await reader.readuntil(b"login:")
                self.output.insert(tk.END, prompt.decode(errors="ignore"))
                self.output.see(tk.END)

                writer.write((login + "\n").encode())
                await writer.drain()

                # Request historical spots
                writer.write(b"SH/DX\n")
                await writer.drain()

                # Start initial buffering
                with self._buffer_lock:
                    self._initial_buffer.clear()
                    self._buffering_initial = True
                self.output.insert(tk.END, "[INFO] Collecting initial SH/DX spots...\n")
                self.output.see(tk.END)

                # Flush after short delay
                self.root.after(800, self._flush_initial_buffer)

                self.connected = True
                self.reconnect_attempts = 0
                if self.connect_button.winfo_exists():
                    self.connect_button.config(text="Disconnect")

                # Start flush loop for live spots
                self.root.after(self._live_flush_interval, self._flush_live_buffer)

                # Read incoming data
                while True:
                    line = await reader.readline()
                    if not line:
                        raise ConnectionResetError("Connection closed by remote host")

                    decoded = line.decode(errors="ignore").strip()
                    if decoded:
                        self.output.insert(tk.END, decoded + "\n")
                        self.output.see(tk.END)
                        self.parse_spot(decoded)

            except Exception as e:
                self.connected = False
                self.writer = None
                self.output.insert(tk.END, f"❌ Connection error: {e}\n")
                self.output.see(tk.END)
                if self.connect_button.winfo_exists():
                    self.connect_button.config(text="Connect")

            # Auto reconnect or manual stop
            if self.manual_disconnect or not self.auto_reconnect_var.get():
                self.output.insert(tk.END, "🔌 Auto-reconnect is off or manually disconnected. Staying disconnected.\n")
                self.output.see(tk.END)
                break

            self.reconnect_attempts += 1
            self.output.insert(tk.END, f"🔁 Reconnect attempt {self.reconnect_attempts} in 5 seconds...\n")
            self.output.see(tk.END)
            await asyncio.sleep(5)











    def parse_spot(self, line):
        # SH/DX: starts with frequency and contains a date in the format 28-May-2025
        if re.match(r"^\d+\.\d+", line) and re.search(r"\d{1,2}-[A-Za-z]{3}-\d{4}", line):
            self.parse_shdx_line(line)
        elif line.startswith("DX de"):
            self.parse_live_line(line)
        else:
            pass





    def get_country_from_call(self, call):
        if not call:
            return ("", "")

        call = call.strip().upper()
        best_match = None
        best_prefix_len = 0

        for entry in self.dxcc_data:
            for raw_prefix in entry.prefixes:
                # Strip formatting marks: ; = ( ) * [ ]
                prefix = re.sub(r'[=;()\[\]*]', '', raw_prefix).strip().upper()

                if call.startswith(prefix) and len(prefix) > best_prefix_len:
                    best_match = entry
                    best_prefix_len = len(prefix)

        if best_match:
            return (best_match.name, best_match.continent)  # Or flag, cq_zone, etc.

        return ("", "")




    def parse_shdx_line(self, line):
        match = re.match(
            r"^(?P<freq>\d+\.\d+)\s+(?P<dx>\S+)\s+(?P<date>\d{1,2}-[A-Za-z]{3}-\d{4})\s+(?P<time>\d{4})Z\s+(?P<comment>.*?)\s*<(?P<spotter>[^>]+)>",
            line
        )
        if not match:
            return

        try:
            freq_khz = float(match.group("freq"))
            freq = f"{freq_khz / 1000:.4f}"
            dx = match.group("dx")
            date_str = match.group("date")
            time_str = match.group("time")
            comment = match.group("comment").strip()
            spotter = match.group("spotter")

            # Make correct UTC datetime from cluster data
            try:
                # Datum + time in UTC
                dt_utc = datetime.strptime(f"{date_str} {time_str}", "%d-%b-%Y %H%M")
                dt_utc = dt_utc.replace(tzinfo=datetime.timezone.utc)
               # Convert to local time (optional, so times remain recognizable)
                dt = dt_utc.astimezone().replace(tzinfo=None)
            except Exception:
                dt = datetime.now(timezone.utc)

            timestr = dt.strftime("%H:%M")
            country, flag = self.get_country_from_call(dx)

            spot = {
                "freq": freq,
                "dx": dx,
                "time": timestr,
                "spotter": spotter,
                "comment": comment,
                "country": country,
                "flag": flag,
                "datetime": dt,
            }

            with self._buffer_lock:
                if getattr(self, "_buffering_initial", False):
                    self._initial_buffer.append(spot)
                else:
                    self._live_buffer.append(spot)

        except Exception as e:
            print(f"Parse error in parse_shdx_line: {e}")



    def parse_live_line(self, line):
        if not line.startswith("DX de"):
            return
        try:
            parts = line.split()
            spotter = parts[2].rstrip(":")
            freq_khz = float(parts[3])
            freq = f"{freq_khz / 1000.0:.4f}"
            dx = parts[4]
            comment = " ".join(parts[5:-1]) if len(parts) > 6 else ""
            time_raw = parts[-1]
            cleaned_time = re.sub(r"[^\d]", "", time_raw)
            time = f"{cleaned_time[:2]}:{cleaned_time[2:]}" if len(cleaned_time) == 4 else ""

           # Create datetime with daterestore after midnight
            now = datetime.now(timezone.utc)

            try:
                if time:
                    t = datetime.strptime(time, "%H:%M").time()
                    dt = datetime.combine(now.date(), t)
                    if (dt - now).total_seconds() > 12 * 3600:
                        dt -= timedelta(days=1)
                else:
                    dt = now
            except Exception:
                dt = now

            # Determine mode via frequency
            freq_mhz = float(freq)
            mode = ""
            for m, ranges in MODES.items():
                for low, high in ranges:
                    if low <= freq_mhz <= high:
                        mode = m
                        break
                if mode:
                    break

            country, flag = self.get_country_from_call(dx)

            spot = {
                "freq": freq,
                "dx": dx,
                "spotter": spotter,
                "mode": mode,
                "comment": comment,
                "time": time,
                "country": country,
                "flag": flag,
                "datetime": dt,
            }

            with self._buffer_lock:
                self._live_buffer.append(spot)

        except Exception as e:
            print(f"Parse error in parse_live_line: {e}")





    
    def handle_new_spot(self, spot):
        if self.spot_matches_filters(spot):
            self.add_spot_to_treeview(spot)




    def add_spot_to_treeview(self, spot):
        if not self.spot_matches_filters(spot):
            return

        dx_call = spot.get("dx", "").upper()
        worked_calls_today = self.get_worked_calls_today() if self.get_worked_calls_today else set()
        current_worked_calls = self.get_worked_calls() if self.get_worked_calls else set()

        try:
            freq_str = f"{float(spot['freq']):.4f}"
        except (ValueError, TypeError):
            freq_str = spot['freq']

        callmatch = re.fullmatch(rf"(?i)(.*[/])?{re.escape(self.user_callsign)}([/].*)?", dx_call)
        if callmatch:
            tag = 'owncall'
        elif dx_call in [c.upper() for c in self.alert_callsigns]:
            tag = 'alert'
        elif dx_call in worked_calls_today:
            tag = 'worked_today'
        elif dx_call in current_worked_calls:
            tag = 'worked'
        else:
            tag = ''




        self.tree.insert(
            "", 0,
            values=(
                spot["time"],
                freq_str,
                spot["dx"],
                spot.get("country", ""),
                spot["spotter"],
                spot["comment"]
            ),
            tags=(tag,)
        )

        children = self.tree.get_children()
        if len(children) > 100:
            self.tree.delete(children[-1])

        self.update_treeview_colors()






    def filter_spots(self):
        self.tree.delete(*self.tree.get_children())
        filtered_spots = []

        for spot in self.spots:
            if self.spot_matches_filters(spot):
                filtered_spots.append(spot)

        for row_index, spot in enumerate(filtered_spots):
            tag = 'evenrow' if row_index % 2 == 0 else 'oddrow'
            freq_str = f"{float(spot['freq']):.4f}"
            self.tree.insert("", "end", values=(
                spot["time"],
                freq_str,
                spot["dx"],
                spot.get("country", ""),
                spot["spotter"],
                spot["comment"]
            ), tags=(tag,))

        self.update_treeview_colors()

        children = self.tree.get_children()
        if children:
            self.tree.see(children[0])





    def spot_matches_filters(self, spot):
        band = self.band_var.get()
        mode = self.mode_var.get()

        try:
            freq_mhz = float(spot["freq"])
        except (ValueError, TypeError):
            return False

        # Bandfilter
        if band != "ALL":
            low, high = BANDS.get(band, (0, float("inf")))
            if not (low <= freq_mhz <= high):
                return False

        # Modefilter (based on frequency-analyses)
        if mode != "ALL":
            for m, ranges in MODES.items():
                if m == mode:
                    for low, high in ranges:
                        if low <= freq_mhz <= high:
                            return True
                    return False
            return False

        return True



    def _flush_initial_buffer(self):
        """Sort initial SH/DX buffer by datetime (newest first) and build the tree once."""
        try:
            with self._buffer_lock:
                buf = list(self._initial_buffer)
                self._initial_buffer.clear()
                self._buffering_initial = False

            if not buf:
                return

            buf.sort(key=lambda s: s.get("datetime", datetime.min), reverse=True)
            # Add all to persistent buffer
            self.all_spots.extend(buf)
            self.all_spots.sort(key=lambda s: s.get("datetime", datetime.min), reverse=True)
            # Keep buffer limited (e.g. 2000 spots max)
            self.all_spots = self.all_spots[:2000]

            # Create filtered view
            self.spots = [s for s in self.all_spots if self.spot_matches_filters(s)][:100]


            self.tree.delete(*self.tree.get_children())
            for idx, spot in enumerate(self.spots):
                tag = 'evenrow' if idx % 2 == 0 else 'oddrow'
                self.tree.insert("", "end", values=(
                    spot["time"],
                    spot["freq"],
                    spot["dx"],
                    spot["country"],
                    spot["spotter"],
                    spot["comment"]
                ), tags=(tag,))

            self.update_treeview_colors()
            self.root.after(self._live_flush_interval, self._flush_live_buffer)

        except Exception as e:
            print(f"_flush_initial_buffer error: {e}")


    def _flush_live_buffer(self):
        """Flush live spots periodically to the tree in sorted order."""
        try:
            with self._buffer_lock:
                buf = list(self._live_buffer)
                self._live_buffer.clear()

            if not buf:
                self.root.after(self._live_flush_interval, self._flush_live_buffer)
                return

            buf.sort(key=lambda s: s.get("datetime", datetime.min), reverse=True)

            # Add new spots to permanent buffer
            self.all_spots = (buf + self.all_spots)[:2000]

            # Show only spots that fall within the active filter
            filtered_buf = [s for s in buf if self.spot_matches_filters(s)]
            for spot in reversed(filtered_buf):
                self.add_spot_to_treeview(spot)

                # === Alert-filter check ===
                dx = spot.get("dx", "").upper()
                for call in self.alert_callsigns:
                    if dx == call.upper():
                        self.insert_alert_row(spot)
                        break




            self.root.after(self._live_flush_interval, self._flush_live_buffer)

        except Exception as e:
            print(f"_flush_live_buffer error: {e}")
            self.root.after(self._live_flush_interval, self._flush_live_buffer)



    def refresh_filtered_view(self):
        """Herbouw TreeView op basis van huidige filter uit all_spots."""
        try:
            self.tree.delete(*self.tree.get_children())
            filtered = [s for s in self.all_spots if self.spot_matches_filters(s)]
            filtered.sort(key=lambda s: s.get("datetime", datetime.min), reverse=True)
            self.spots = filtered[:100]

            for idx, spot in enumerate(self.spots):
                tag = 'evenrow' if idx % 2 == 0 else 'oddrow'
                self.tree.insert("", "end", values=(
                    spot["time"],
                    spot["freq"],
                    spot["dx"],
                    spot["country"],
                    spot["spotter"],
                    spot["comment"]
                ), tags=(tag,))
            self.update_treeview_colors()
        except Exception as e:
            print(f"refresh_filtered_view error: {e}")





    def on_close(self): 
        self.connected = False
        if hasattr(self, 'writer') and self.writer is not None:
            try:
                future = asyncio.run_coroutine_threadsafe(self.disconnect_async(), self.loop)
                future.result(timeout=1)
            except Exception as e:
                print(f"Error disconnecting during closing: {e}")

        x = self.root.winfo_x()
        y = self.root.winfo_y()
        self.save_window_position(x, y)

        # Always call the original destroy (even if it comes via MiniBook)
        if hasattr(self.root, 'destroy_original'):
            self.root.destroy_original()
        else:
            self.root.destroy()



    def save_window_position(self, x, y):
        self.safe_write_to_ini("WINDOW", {
            "x": str(x),
            "y": str(y)
        })


    def restore_window_position(self):
        print("[DEBUG] Attempting to restore window position...")
        config = configparser.ConfigParser()
        config.optionxform = str

        if not os.path.exists(INI_FILE):
            print(f"[DEBUG] INI file '{INI_FILE}' does not exist. Aborting restore.")
            return

        print(f"[DEBUG] INI file '{INI_FILE}' found. Reading contents...")
        try:
            config.read(INI_FILE, encoding="utf-8")
            print("[DEBUG] INI file successfully read.")

            if "WINDOW" not in config:
                print("[DEBUG] Section [WINDOW] not found in INI file.")
                return

            x_str = config["WINDOW"].get("x", "100")
            y_str = config["WINDOW"].get("y", "100")
            print(f"[DEBUG] Retrieved position strings: x='{x_str}', y='{y_str}'")

            x = int(x_str)
            y = int(y_str)
            print(f"[DEBUG] Parsed position: x={x}, y={y}")

            self.root.geometry(f"+{x}+{y}")
            print(f"[DEBUG] Window position restored to: x={x}, y={y}")

        except Exception as e:
            print(f"[ERROR] Failed to restore window position: {e}")

        


    async def disconnect_async(self):
        if self.writer:
            try:
                self.writer.close()
                await self.writer.wait_closed()
            except Exception as e:
                print(f"Error during async disconnect:{e}")
            finally:
                self.writer = None





    def update_tracking_status(self):
            if not hasattr(self, 'tracking_status_label'):
                self.tracking_status_label = ttk.Label(self.root, text="", font=("Segoe UI", 9, "bold"))
                self.tracking_status_label.pack(pady=(0, 5))
            if self.tracking_var and self.tracking_var.get():
                self.tracking_status_label.config(text="Hamlib Tracking: ✅ Enabled", foreground="green")
            else:
                self.tracking_status_label.config(text="Hamlib Tracking: ❌ Disabled", foreground="red")
            self.root.after(500, self.update_tracking_status)


    
    def spot_clicked_from_popup(self, spot):
        """looks spot at radio and MiniBook — identical to click in TreeView."""
        try:
            raw_freq = spot.get("freq")
            to_call = spot.get("dx")
            if not raw_freq or not to_call:
                return

            hz = int(float(raw_freq) * 1_000_000)
            gui_mode = self.mode_var.get()

            try:
                freq_mhz = f"{float(raw_freq):.5f}"
            except ValueError:
                freq_mhz = ""

            # Determine mode (same logic as spot_clicked)
            if gui_mode == "PHONE" or gui_mode == "ALL":
                resolved_mode = "USB" if hz > 10000000 else "LSB"
            else:
                resolved_mode = gui_mode

            # --- MiniBook link ---
            if self.on_callsign_selected:
                try:
                    self.on_callsign_selected(to_call, freq=freq_mhz, mode=resolved_mode)
                except TypeError:
                    # Some versions of MiniBook have a simpler signature
                    self.on_callsign_selected(to_call)

            # --- Tracking & radio frequency ---
            if not (self.tracking_var and self.tracking_var.get()):
                return

            with socket.create_connection((self.rigctl_host, self.rigctl_port), timeout=2) as s:
                s.sendall(f"F {hz}\n".encode())
                _ = s.recv(1024)
                if resolved_mode and resolved_mode not in ("ALL", "DIGITAL"):
                    s.sendall(f"M {resolved_mode} -1\n".encode())
                    _ = s.recv(1024)

        except Exception as e:
            print(f"Fout bij spot_clicked_from_popup: {e}")




    
    def spot_clicked(self, event):
        
            # Only process left-click (button 1)
            if event.num != 1:
                return

            try:
                selected = self.tree.identify_row(event.y)
                if not selected:
                    return

                self.tree.selection_set(selected)
                values = self.tree.item(selected)["values"]
                raw_freq = values[1]
                to_call = values[2]
                gui_mode = self.mode_var.get()
                hz = int(float(raw_freq) * 1_000_000)
                
                try:
                    freq_mhz = f"{float(raw_freq):.5f}"

                except ValueError:
                    freq_mhz = ""
                if gui_mode == "PHONE" or gui_mode == "ALL":
                    resolved_mode = "USB" if hz > 10000000 else "LSB"
                else:
                    resolved_mode = gui_mode
                if self.on_callsign_selected:
                    self.on_callsign_selected(to_call, freq=freq_mhz, mode=resolved_mode)
                if not (self.tracking_var and self.tracking_var.get()):
                    return
                                
                with socket.create_connection((self.rigctl_host, self.rigctl_port), timeout=2) as s:
                    s.sendall(f"F {hz}\n".encode())
                    _ = s.recv(1024)
                    if resolved_mode and resolved_mode not in ("ALL", "DIGITAL"):
                        s.sendall(f"M {resolved_mode} -1\n".encode())
                        _ = s.recv(1024)
            except Exception:
                pass

# Spot Custom Filter Functions
    def center_window(self, win, width, height):
        self.root.update_idletasks()
        x = self.root.winfo_rootx() + (self.root.winfo_width() - width) // 2
        y = self.root.winfo_rooty() + (self.root.winfo_height() - height) // 2
        win.geometry(f"{width}x{height}+{x}+{y}")
    def load_custom_filters(self):
        config = configparser.ConfigParser()
        config.optionxform = str
        if os.path.exists(INI_FILE):
            config.read(INI_FILE)
            if "CustomFilters" in config:
                return {k: v for k, v in config["CustomFilters"].items()}
        return {}

    def save_custom_filters(self):
        config = configparser.ConfigParser()
        config.optionxform = str

        if os.path.exists(INI_FILE):
            config.read(INI_FILE, encoding="utf-8")

        # Remove CustomFilters section if it exists
        if "CustomFilters" in config:
            del config["CustomFilters"]

        # Only restore if there are still filters
        if self.custom_filters:
            config["CustomFilters"] = {}
            for k, v in self.custom_filters.items():
                config["CustomFilters"][k] = v

        with open(INI_FILE, "w", encoding="utf-8") as f:
            config.write(f)
            self.safe_write_to_ini("CustomFilters", self.custom_filters)




    def update_mode_combobox(self):
        custom_keys = list(self.custom_filters.keys())
        self.mode_menu["values"] = ["ALL"] + list(MODES.keys()) + custom_keys



    def manage_custom_filters(self):
        if self._open_windows["manage_filters"] and self._open_windows["manage_filters"].winfo_exists():
            self._open_windows["manage_filters"].lift()
            return

        win = tk.Toplevel(self.root)
        self._open_windows["manage_filters"] = win

        def on_close():
            self._open_windows["manage_filters"] = None
            win.destroy()

        win.protocol("WM_DELETE_WINDOW", on_close)
        win.title("Custom Filters")
        self.center_window(win, 400, 300)

        listbox = tk.Listbox(win)
        listbox.pack(fill="both", expand=True, padx=10, pady=5)

        for name in self.custom_filters:
            listbox.insert("end", name)

        def refresh_list():
            listbox.delete(0, tk.END)
            for name in self.custom_filters:
                listbox.insert("end", name)

        def add_filter():
            def refresh_and_close():
                listbox.delete(0, tk.END)
                for name in self.custom_filters:
                    listbox.insert("end", name)
            self.edit_filter_dialog(callback=refresh_and_close, external_window=win)


        def edit_selected():
            selected = listbox.curselection()
            if not selected:
                return

            name = listbox.get(selected[0])
            pattern = self.custom_filters[name]

            def refresh_and_close():
                listbox.delete(0, tk.END)
                for name in self.custom_filters:
                    listbox.insert("end", name)

            self.edit_filter_dialog(name, pattern, callback=refresh_and_close, external_window=win)


        def delete_selected():
            selected = listbox.curselection()
            if not selected:
                return

            name = listbox.get(selected[0])

            # --- Custom confirm dialoog gekoppeld aan het juiste venster (win) ---
            confirm = tk.Toplevel(win)
            confirm.title("Delete filter")
            confirm.transient(win)        # <<< BELANGRIJK: niet self.root
            confirm.grab_set()            # modaal
            self.center_window(confirm, 360, 140)

            msg = tk.Label(
                confirm,
                text=f"Delete filter '{name}'?",
                anchor="center",
                justify="center",
                padx=10,
                pady=15
            )
            msg.pack(fill="both", expand=True)

            btn_frame = tk.Frame(confirm)
            btn_frame.pack(pady=(0, 10))

            def do_delete():
                # voer delete uit
                del self.custom_filters[name]
                self.save_custom_filters()
                refresh_list()
                self.update_mode_combobox()

                confirm.grab_release()    # <<< BELANGRIJK
                confirm.destroy()         # sluit alleen de dialog

            def cancel():
                confirm.grab_release()    # <<< BELANGRIJK
                confirm.destroy()

            tk.Button(btn_frame, text="Yes", width=10, command=do_delete).pack(side="left", padx=10)
            tk.Button(btn_frame, text="No", width=10, command=cancel).pack(side="right", padx=10)



        btn_frame = tk.Frame(win)
        btn_frame.pack(pady=5)

        tk.Button(btn_frame, text="➕ Add", command=add_filter).pack(side="left", padx=5)
        tk.Button(btn_frame, text="✏️ Edit", command=edit_selected).pack(side="left", padx=5)
        tk.Button(btn_frame, text="❌ Delete", command=delete_selected).pack(side="left", padx=5)

        tk.Button(win, text="Close", command=on_close).pack(pady=(0, 10))

        

    def edit_filter_dialog(self, name="", pattern="", callback=None, external_window=None):
        if self._open_windows["edit_filter"] and self._open_windows["edit_filter"].winfo_exists():
            self._open_windows["edit_filter"].lift()
            return

        win = tk.Toplevel(self.root if external_window is None else external_window)
        self._open_windows["edit_filter"] = win

        def on_close():
            self._open_windows["edit_filter"] = None
            win.destroy()

        win.protocol("WM_DELETE_WINDOW", on_close)
        win.title("Edit Filter" if name else "Add Filter")
        self.center_window(win, 420, 220)

        tk.Label(win, text="Name:").pack(anchor="w", padx=10, pady=(10, 0))
        name_var = tk.StringVar(value=name)
        tk.Entry(win, textvariable=name_var).pack(fill="x", padx=10)

        tk.Label(win, text="Regex Pattern:").pack(anchor="w", padx=10, pady=(10, 0))
        pattern_var = tk.StringVar(value=pattern)
        tk.Entry(win, textvariable=pattern_var).pack(fill="x", padx=10)

        def save():
            new_name = name_var.get().strip()
            new_pattern = pattern_var.get().strip()

            if not new_name or not new_pattern:
                messagebox.showwarning("Invalid", "Name and pattern cannot be empty.", parent=win)
                return

            self.custom_filters[new_name] = new_pattern
            self.save_custom_filters()
            self.update_mode_combobox()

            if callback:
                callback()

            on_close()

        tk.Button(win, text="Save", command=save).pack(pady=10)


    # Adjusting spot_matches_filters():
    def spot_matches_filters(self, spot):
        band = self.band_var.get()
        mode = self.mode_var.get()

        try:
            freq_mhz = float(spot["freq"])
        except (ValueError, TypeError):
            return False

        if band != "ALL":
            low, high = BANDS.get(band, (0, float("inf")))
            if not (low <= freq_mhz <= high):
                return False

        if mode != "ALL":
            if mode in MODES:
                for low, high in MODES[mode]:
                    if low <= freq_mhz <= high:
                        return True
                return False
            elif mode in self.custom_filters:
                pattern = self.custom_filters[mode]
                spot_text = f"{spot.get('dx','')} {spot.get('comment','')} {spot.get('spotter','')}"
                return re.search(pattern, spot_text, re.IGNORECASE) is not None
            else:
                return False

        return True


    def load_alert_sound_setting(self):
        """Load the alert sound checkbox state from INI using safe read."""
        try:
            config = configparser.ConfigParser()
            config.optionxform = str
            if os.path.exists(INI_FILE):
                config.read(INI_FILE, encoding="utf-8")
                if config.has_option("Settings", "alert_sound"):
                    return config.getboolean("Settings", "alert_sound")
        except Exception:
            pass
        return True  # Default enabled



    def safe_write_to_ini(self, section, keyvals):
        """
        Safely write to an INI file without losing other sections or capitalization.
        - section: name of the INI section (string)
        - keyvals: dict with the key/values you want to set in that section
        """
        config = configparser.ConfigParser()
        config.optionxform = str

        if os.path.exists(INI_FILE):
            config.read(INI_FILE, encoding="utf-8")

        if section not in config:
            config[section] = {}

        config[section].update(keyvals)

        # ✅ Ensure that the directory for the INI file exists
        ini_dir = os.path.dirname(INI_FILE)
        os.makedirs(ini_dir, exist_ok=True)

        with open(INI_FILE, "w", encoding="utf-8") as f:
            config.write(f)



    def show_alert_popup(self, spot):
        """Shows popup when spot matches alert filter"""
        popup = tk.Toplevel(self.root)
        popup.title("📡 Callsign Alert!")
        popup.geometry("340x220")
        popup.configure(bg="#f9f9f9")
        popup.grab_set()
        self.center_window(popup, 340, 220)

        msg = (
            f"Callsign: {spot.get('dx','')}\n"
            f"Frequency: {spot.get('freq','')}\n"
            f"Spotter: {spot.get('spotter','')}\n"
            f"Time: {spot.get('time','')}\n"
            f"Comment: {spot.get('comment','')}\n"
            f"Country: {spot.get('country','')}"
        )

        # Title
        tk.Label(popup, text="MATCH FOUND!", fg="red", font=("Arial", 12, "bold"), bg="#f9f9f9").pack(pady=(10, 5))

        # Message
        tk.Label(popup, text=msg, justify="left", bg="#f9f9f9", font=("Arial", 10)).pack(padx=10, pady=5)

        # Buttonframe
        btn_frame = tk.Frame(popup, bg="#f9f9f9")
        btn_frame.pack(side="bottom", pady=10)

        def close_popup(event=None):
            popup.destroy()

        def set_freq():
            """Sends spot to radio (same a click in TreeView)."""
            try:
                self.spot_clicked_from_popup(spot)
            except Exception as e:
                messagebox.showerror("MiniBook", f"Could not set frequency:\n{e}")
            popup.destroy()

        # OK-button
        ok_btn = tk.Button(btn_frame, text="OK", width=10, command=close_popup, bg="lightgrey")
        ok_btn.pack(side="left", padx=10)

        # Set Freq Button (Only if MiniBook is active)
        if getattr(self, "minibook_connected", False):
            set_btn = tk.Button(btn_frame, text="🎯 Set Freq", width=10, bg="#2e8b57", fg="white", command=set_freq)
            set_btn.pack(side="left", padx=10)

        # Keybindings
        popup.bind("<Escape>", close_popup)
        popup.bind("<Return>", close_popup)



    def center_window(self, win, width=None, height=None):
        """Center a Toplevel window above the main window"""
        win.update_idletasks()
        root_x = self.root.winfo_rootx()
        root_y = self.root.winfo_rooty()
        root_w = self.root.winfo_width()
        root_h = self.root.winfo_height()

        # Request current windows size if no fixed size is available
        if width is None or height is None:
            width = win.winfo_width()
            height = win.winfo_height()

        # Calculate centered position
        x = root_x + (root_w // 2) - (width // 2)
        y = root_y + (root_h // 2) - (height // 2)

        win.geometry(f"{width}x{height}+{x}+{y}")



# Used when running stand alone
if __name__ == "__main__":
    root = tk.Tk()
    root.title(f"DX Cluster Telnet Client - {VERSION_NUMBER}")
    app = DXClusterApp(root)
    root.mainloop()
  



# Used when opened from MiniBook
def launch_dx_spot_viewer(
    rigctl_host="127.0.0.1",
    rigctl_port=4532,
    tracking_var=None,
    on_callsign_selected=None,
    get_worked_calls=None,
    get_worked_calls_today=None,
    get_last_qso_callsign=None,
    get_current_frequency=None,
    parent_window=None,
):



    if parent_window is None:
        parent_window = tk.Toplevel()
        parent_window.resizable(True, True)

    app = DXClusterApp(
        parent_window,
        rigctl_host,
        rigctl_port,
        tracking_var,
        on_callsign_selected,
        get_worked_calls,
        get_worked_calls_today,
        get_last_qso_callsign,
        get_current_frequency
    )

    app.minibook_connected = True


    # Save the instance in the parent window so MiniBook can access it
    parent_window.dxcluster_app = app

    return app
