param (
    [Parameter(ValueFromRemainingArguments = $true)]
    [string[]]$Apps
)

# ==================================================
# Setup
# ==================================================
$WshShell = New-Object -ComObject WScript.Shell
$Desktop  = [Environment]::GetFolderPath("Desktop")

# App root = one folder above this script (INSTALL -> app root)
$appRoot = Resolve-Path (Join-Path $PSScriptRoot "..")

# ==================================================
# Find pythonw.exe or python.exe (PowerShell 5.1 safe)
# ==================================================
$python = $null

$cmd = Get-Command pythonw.exe -ErrorAction SilentlyContinue
if ($cmd) {
    $python = $cmd.Source
} else {
    $cmd = Get-Command python.exe -ErrorAction SilentlyContinue
    if ($cmd) {
        $python = $cmd.Source
    }
}

if (-not $python) {
    Write-Host "ERROR: Python not found in PATH"
    exit 1
}

Write-Host "Using Python: $python"
Write-Host "App root: $appRoot"
Write-Host ""

# ==================================================
# Process apps
# ==================================================
foreach ($app in $Apps) {

    if (-not $app) { continue }

    $parts = $app.Split("|")
    if ($parts.Count -lt 2) {
        Write-Host "WARNING: Invalid definition: $app"
        continue
    }

    $scriptName   = $parts[0].Trim()
    $shortcutName = $parts[1].Trim()
    $iconName     = if ($parts.Count -ge 3) { $parts[2].Trim() } else { "" }

    $scriptPath = Join-Path $appRoot $scriptName
    if (-not (Test-Path $scriptPath)) {
        Write-Host "WARNING: Script not found: $scriptPath"
        continue
    }

    $lnkPath = Join-Path $Desktop ($shortcutName + ".lnk")
    $shortcut = $WshShell.CreateShortcut($lnkPath)

    $shortcut.TargetPath = $python
    $shortcut.Arguments  = "`"$scriptPath`""
    $shortcut.WorkingDirectory = Split-Path $scriptPath

    if ($iconName) {
        $iconPath = Join-Path $appRoot $iconName
        if (Test-Path $iconPath) {
            $shortcut.IconLocation = $iconPath
        } else {
            Write-Host "WARNING: Icon not found: $iconPath"
        }
    }

    $shortcut.Save()
    Write-Host "Created shortcut: $shortcutName"
}

Write-Host ""
Write-Host "Done."
