#**********************************************************************************************************************************
# File          :   MiniBook.py
# Project       :   A JSON Based logbook for portable use.
# Description   :   Logs basic contact information to json
# Date          :   18-10-2024
# Authors       :   Bjorn Pasteuning - PD5DJ
# Website       :   https://wwww.pd5dj.nl
#
# Version history
#   18-10-2024  :   1.0.0   -   Initial basics running
#   20-10-2024  :   1.0.7   -   Added DXCC lookup
#   20-10-2024  :   1.0.8   -   If DXCC lookup file is missing, added download option
#   21-10-2024  :   1.0.9   -   Basic Log viewer added
#   22-10-2024  :   1.1.0   -   Search function added
#   22-10-2024  :   1.1.1   -   Sorting function added
#   24-10-2024  :   1.1.2   -   Scrollbar fixed in log viewer, layout changed, row color change fixed after search
#   25-10-2024  :   1.1.3   -   Country column added
#   31-10-2024  :   1.1.4   -   ADIF logbook process is changed in JSON
#                           -   ADIF Export function added
#   01-11-2024  :   1.1.5   -   Sorting fixed in logbook viewer
#                           -   Locator field added to main window
#   02-11-2024  :   1.1.6   -   Add flag images that matches with DXCC Entity
#                   1.1.7   -   Added QSO delete option in Edit window
#                           -   Improved prefix filtering
#   03-11-2024  :   1.1.8   -   Yet another prefix filter improvement
#                   1.1.9   -   Cosmetic improvements in main window
#                           -   Major improvement prefix filter, source for dcxx.json is change to my own repo on github
#   06-11-2024  :   1.1.9a  -   New Logbook structure
#                           -   Configuration preferences: Station Callsign and Locator are now moved into new logbook structure
#                           -   Detection/Conversion of older logbooks implemented
#   07-11-2024  :   1.1.9b  -   Edit select & Delete functions changed to right-mouseclick in log viewer
#                           -   Time Tracking checkbox add to menu, handy if you want to add QSO's afterwards
#   09-11-2024  :   1.1.9c  -   Several bugs fixed:
#                           -   If no call matched with dxcc.json then it logged the previous dxcc. Now loggin "[None]" when not found
#                           -   Comboboxes, Band and Mode are now prohibited to edit manually
#                           -   If you now go through all the entry fields with TAB and arrive at frequency, the cursor will be directly at the end so that it can be changed immediately
#                           -   After logging, immediately move the cursor to the callsign entry
#                           -   If you enter a frequency that does not match the band, adjust the band automatically and Visa Versa
#  10-11-2024   :   1.2.1   -   ADIF Import added
#                           -   File encoding fixed when reading ADIF files, now checking UTF-8 and LATIN-1
#                           -   Comboboxes added in Edit QSO window
#                           -   valid locator check added in Edit QSO window
#                           -   Delete QSO button added in Edit QSO window
#  13-11-2024   :   1.2.2   -   Row colors fixed when Searching/Sorting
#                           -   When loading new logbook, treeview is now closed before loading. Not showing old loaded content
#  14-11-2024   :   1.2.3   -   First implementation of Hamlib RigCtlD with frequency and mode readout
#                           -   Error handling RigCTLD added
#                           -   Locator check in Edit QSO fixed
#                           -   Free port detection added, for multiple deamon instances on network
#  20-11-2024   :   1.2.3c  -   Threading added for rigctld deamon
#                   1.2.3d  -   Thread added for update frequency_mode
#                   1.2.4   -   First implementation of WSJT-X ADIF UDP support
#  23-11-2024   :   1.2.4a  -   Added posibility to connect to external rigctld server
#  24-11-2024   :   1.2.5   -   Fix, When changing UDP port, it is was not updated until restart program
#                           -   Confirmation on quitting program
#                           -   Window position save / load added for Main and Logbook window
#  25-11-2024   :   1.2.6   -   Parameters are now stored in user appdata folder
#                           -   Export ADIF back to user select where to save method
#                           -   Name entry added
#  01-12-2024   :   1.2.7   -   More robust Config.ini handler added
#                           -   Added load last logbook function upon start of MiniBook
#  19-04-2025   :   1.2.9   -   Multiselect added in logbook for record deletion
#  21-04-2025   :           -   Added Worked Before Frame at bottom of main window
#  27-04-2025   :   1.3.0   -   MAJOR CHANGE:
#                           -   Due many issues with integrating a Hamlib server, I have decided to remove all server related parts from MiniBook.
#                           -   Minibook can now only connect to an external Hamlib service.
#                           -   Example Batch files are added how to start a service in the background.
#                           -   Fixed RST format 59/599 on mode change
#                           -   Fixed manual RST edit in edit window
#                           -   Fixed Date / Time entry, if incorrect format was entered logbook could not be loaded correctly anymore.
#                           -   Added Satellite Entry field, names are stored in satellites.txt and can be updated with getsatnames.py
#                               Satellite fields are now imported from ADIF, and Exported to ADIF
#  01-05-2025   :   1.3.1   -   QRZ Lookup added, credentials entered in preference menu.
#                           -   Layout change to support QRZ Lookup, Button added for Lookup
#                               QRZ Lookup ofcourse only works with a valid QRZ Lookup subscription
#                               QRZ Lookup only works with internet connection.
#                           -   Added better Callsign filtering for QRZ lookup
#  03-05-2025               -   Now when pressing TAB in callsign entry a QRZ Lookup will be performed automaticly.
#  05-05-2025   :   1.3.2   -   Fix in ADIF export Submode.
#                           -   Export to ADIF is 3 formats, POTA, WWFF and Normal
#                           -   QSO Edit window re-arranged
#                           -   Openarator field added in My Station Setup
#                           -   POTA and WWFF entries added in logbook window
#                           -   Hamlib status and loaded logbook now moven to title bar
# 09-05-2025    :   1.3.3   -   QRZ API Upload added
#                           -   ADIF Import Progressbar added
#                           -   Frequency Input control added, type frequency in KHZ
#                           -   Added "Dupe check of today", callsign will show red on orange when already worked that day.
#                           -   Fix in logbook load function, Clearing old logbook upon cancelation of loading logbook
# 13-05-2025    :   1.3.4   -   Export Selected records to ADIF added in Logbook Window
#                           -   Added frequency offset control, user now can shift frequecy up and down in khz with - or +, ie +15 is 15khz up
#                           -   log backup function added, when opening logbooks a copy is stored in backup-logs folder with timestamp
#                           -   Fix QRZ upload record, Operator field was missing
# 17-05-2025    :   1.3.5   -   Added, DX Summit DX Cluster window.
#                           -   Click on spot, will send frequency and mode to logbook and sent to Hamlib
# 21-05-2025    :   1.3.6   -   Bulkedit added when selecting multiple records.
#                           -   Duplicate QSO finder added
# 29-05-2025    :   1.3.7   -   Password(s) & API Keys have show buttons
#                           -   dxcc.json is replaced by cty.dat, unfortunally cty.dat does not provide dxcc entity numbers, so Flag image had to be removed.
#                           -   Worked before expanded with more matching colors
# 01-06-2025    :   1.3.8   -   QSO's are now processed fully in cache, minimized disk writing time.. worked b4 lookup speed up.
# 02-06-2025    :   1.3.9   -   Locator lookup added. When locator is entered heading and distance always calculated from these entries.
#                           -   Fix Importing ADIF Duplicate records, when selecting ignore.. no files where added at all
#                           -   Fix in QRZ Lookup when entering multiple / call was not found.. i.e. DK/PD5DJ/P was not found.
# 15-06-2025    :   1.4.0   -   QRZ Lookup Fix, better filtering with dashes in callsigns.
#                               When no match, it will try to retrieve first and last name using base callsign.
#                               For example: if VK/PD5DJ is not found, it will use names from PD5DJ only.
#                           -   Also when callsign retrieves no qrz data, fields are wiped also.
#                           -   Heading and Distance calculation improved
# 05-07-2025    :   1.4.2   -   ADIF import is now be processed threaded
# 03-08-2025    :   1.4.3   -   Edit QSO, time validation was HH:MM now HH:MM:SS
# 09-08-2025    :           -   Added WWFF/POTA/BOTA Prefix retreive function
#                           -   Added lookup for Park/Bunker names
# 10-08-2025    :           -   Changed filepath structure
# 11-08-2025    :   1.4.4   -   Added mapping option to view references on map
# 17-08-2025    :   1.4.5   -   Major Update!
#                           -   Added IOTA/SOTA/WLOTA Entry fields
#                           -   Added Owner information for contest ADIF Export
#                           -   Simple Serial contest logging added
#                           -   Added menu Radio, Frequency & Mode Polling moved.
#                           -   Duplicate finder upgraded.
# 25-08-2025    :   1.4.6   -   SOTA reference lookup added
#                           -   Serial / Receive exchange some fixes
# 27-08-2025    :               RST Pulldown changed for button window
# 29-08-2025    :   1.4.7   -   Cosmetics, Station info and Activation info now notebook style.
#                           -   When callsign entered, and TAB, callsign + freq is sent to DXCluster app if opened.
# 31-08-2025    :   1.4.8   -   Fix in update_frequency_and_mode_thread() where enter cleared callsign
#                           -   Backup folder now default .\backup user can alsways change it.
# 19-09-2025    :   1.4.9   -   Added QSL Sent and Received
#                           -   Added Logbook converter, Checks version of logbook format and auto updates it to newest format
# 21-10-2025    :           -   Invalid Backup folder path detection added
#                           -   WSJT-X naming changed to UDP ADIF
#                           -   More Mode & Sub Modes added
#                           -   Automatic UTC correction added
#                           -   Cosmetic change Station setup menu
#                           -   Find duplicates features expanded, more functionality
#                    1.4.9d -   ADD -       LOTW Records added
#                    1.4.9e -   CHANGED -   Log record names shortened to decrease byte size with bigger logs.
#                                           Due to that, log converter is updated too
#                           -   CHANGED -   Satellite list is now user editable, celestrak download removed
#                           -   CHANGED -   Reference fields in logbook are only added when record contains something.
#                           -   ADD -       Reference frame in main window now collapsable, only visible when needed
#                           -   FIX -       Adif Export did not contain country field.
#                           -   MOVED -     Tracking Time now moved from menu to main window next to  time
#                           -   ADD -       DXCC Entity lookup file added in DATA folder: prefix_to_dxcc.json
#                           -   ADD -       DXCC Statistics page.
#                           -   FIX -       When trying to edit records after search query, index error apeared.
#                           -   CHANGED -   My Station / Activation fields are now collapsable
#                           -   MOVED -     Logbook and DX Cluster now Buttons instead of menu items
#                           -   ADD -       DXCC Flag reintroduced on main window next to country name
#                    1.4.9g -   CHANGED -   Loading and saving json file is now optimized with Async
#                           -   CLEANED -   Code cleaned, unused functions/variables removed
# 27-11-2025    :    1.4.9h -   ADD -       Fast Disk I/O read/write for json file introduced.
#                           -   ADD -       QRZ, Json, Hamlib Logging
#                           -   CHANGED -   Radio Frequency & Mode Polling menu removed.
#                                           Rig Enable option added in preference menu, now fully autoconnect
# 01-12-2025    :    1.4.9i -   CHANGED -   Unified callsign/frequency/mode handler for both manual entry and DXCluster spots. source: "manual" or "spot"
# 02-12-2025    :    1.4.9m -   ADD -       Quick Confirm options added to Logbook, Rightclick mouse
# 10-12-2025    :    1.5.0a -   ADD -       Wordpress dashboard updater
#                               FIX -       Reload last loaded logbook
# 13-12-2025    :    1.5.0b -   FIX -       Sent exchange number was messed up, now fixed, also when deleting qso with highest STX wil trigger Sent exchange refresh
# 15-12-2025    :    1.5.0d -   ADD -       Column View selector added in logbook window
#                               ADD -       ADIF Import now checks for APP_N1MM_EXCHANGE1,CQZ,ITUZ field, wil ask to store it in SRX (Receive Exchange) field
#                    1.5.0e     ADD -       Logbook fields CQZ & ITUZ added, Book version 3.3
# 01-01-2026    :    1.5.0g     FIX -       Dupe check on import, time_off now used
# 02-01-2026    :    1.5.1      RELEASE     Migration support, All settings are now stored in Application Data folder of Windows user.
#**********************************************************************************************************************************

from datetime import datetime, timedelta, date, timezone
from pathlib import Path
from tkcalendar import DateEntry
from tkinter import filedialog, messagebox, ttk
from PIL import Image, ImageTk

# ==================== LEGACY MIGRATION (SAFE) ====================
# MiniBook SAFE migration:
# - Move only known legacy files
# - NEVER delete settings/data folders (shared app root possible)

from pathlib import Path
import shutil
import os
import sys

def migrate_minibook_legacy_safe():
    app_root = Path(sys.argv[0]).resolve().parent

    legacy_settings = app_root / "settings"
    legacy_data     = app_root / "data"
    legacy_logs     = app_root / "logs"
    legacy_backup   = app_root / "backup"

    appdata_base = Path(os.getenv("APPDATA")) / "MiniBook"
    appdata_settings = appdata_base / "settings"
    appdata_data     = appdata_base / "data"

    docs_base   = Path.home() / "Documents" / "MiniBook"
    docs_logs   = docs_base / "logs"
    docs_backup = docs_base / "backup"

    # --- SETTINGS ---
    if legacy_settings.exists():
        appdata_settings.mkdir(parents=True, exist_ok=True)
        for fname in (
            "minibook.ini",
        ):
            src = legacy_settings / fname
            dst = appdata_settings / fname
            if src.exists() and not dst.exists():
                shutil.move(str(src), str(dst))

    # --- DATA (only cache / downloadable files) ---
    if legacy_data.exists():
        appdata_data.mkdir(parents=True, exist_ok=True)
        for fname in (
            "cty.dat",
            "wwff_directory.csv",
            "WCA_list.csv",
            "all_parks.csv",
            "wwbota.csv",
            "fulllist.json",
            "summitslist.csv"
        ):
            src = legacy_data / fname
            dst = appdata_data / fname
            if src.exists() and not dst.exists():
                shutil.move(str(src), str(dst))

    # --- LOGS ---
    if legacy_logs.exists():
        docs_logs.mkdir(parents=True, exist_ok=True)
        for f in legacy_logs.glob("*.log"):
            dst = docs_logs / f.name
            if not dst.exists():
                shutil.move(str(f), str(dst))

    # --- BACKUP ---
    if legacy_backup.exists():
        docs_backup.mkdir(parents=True, exist_ok=True)
        for f in legacy_backup.iterdir():
            dst = docs_backup / f.name
            if f.is_file() and not dst.exists():
                shutil.move(str(f), str(dst))

migrate_minibook_legacy_safe()
# ================== END LEGACY MIGRATION ==================


import tkinter.font as tkFont
import configparser
import csv
import ctypes
import html
import ipaddress
import json
import math
import msvcrt
import logging
import os
import platform
import re
import requests
import socket
import subprocess
import sys
import tempfile
import threading
import time
import tkinter as tk
import traceback
import urllib.parse
import webbrowser
import xml.etree.ElementTree as ET
from logging.handlers import RotatingFileHandler
from DXCluster import launch_dx_spot_viewer
from cty_parser import parse_cty_file

VERSION_NUMBER              = ("v1.5.1")  # Application version
BOOK_VERSION                = 3.3          # Internal logbook version




# ---------------------------------------------------
#  PATHS & FOLDERS (Windows best practice)
# ---------------------------------------------------

APP_ROOT = Path(__file__).resolve().parent

# AppData (per user)
APPDATA_DIR = Path(os.environ["APPDATA"]) / "MiniBook"

SETTINGS_FOLDER = APPDATA_DIR / "settings"
APPDATA_DATA_FOLDER = APPDATA_DIR / "data"

# Documents (logs)
DOCUMENTS_DIR = Path.home() / "Documents" / "MiniBook"
LOG_FOLDER = DOCUMENTS_DIR / "logs"
BACKUP_FOLDER = DOCUMENTS_DIR / "backup"

# App root (static data & backups)
DATA_FOLDER = APP_ROOT / "data"

# Create folders
SETTINGS_FOLDER.mkdir(parents=True, exist_ok=True)
APPDATA_DATA_FOLDER.mkdir(parents=True, exist_ok=True)
LOG_FOLDER.mkdir(parents=True, exist_ok=True)
DATA_FOLDER.mkdir(exist_ok=True)
BACKUP_FOLDER.mkdir(parents=True, exist_ok=True)

CONFIG_FILE                 = SETTINGS_FOLDER / "minibook.ini"  # User settings file
LOCK_FILE                   = SETTINGS_FOLDER / "_minibook.lck" # Prevent multiple instances

MINIBOOK_LOG_FILE           = LOG_FOLDER / "MiniBook.log"       # Main application log
HAMLIB_LOG_FILE             = LOG_FOLDER / "Hamlib.log"         # Hamlib communication log
QRZ_LOOKUP_LOG_FILE         = LOG_FOLDER / "qrzlookup.log"      # QRZ lookup log file
QRZ_UPLOAD_LOG_FILE         = LOG_FOLDER / "qrzupload.log"      # QRZ upload log file
QRZ_UPLOAD_BULK_LOG_FILE    = LOG_FOLDER / "qrzbulkupload.log"  # QRZ bulk upload log file



CURRENT_JSON_FILE           = None  # Current logbook file

DXCC_FILE                   = APPDATA_DATA_FOLDER / "cty.dat"  # Local DXCC data file
ctydat_url                  = "https://www.country-files.com/bigcty/cty.dat"  # DXCC download URL
dxcc_data                   = []  # Parsed DXCC data

WWFF_FILE                   = APPDATA_DATA_FOLDER / "wwff_directory.csv"  # Local WWFF list
wwff_references             = {}  # Parsed WWFF data
wwffref_url                 = "https://wwff.co/wwff-data/wwff_directory.csv"  # WWFF download URL

COTA_FILE                   = APPDATA_DATA_FOLDER / "WCA_list.csv"  # Local COTA list
cota_references             = {}  # Parsed COTA data
cota_url                    = "https://wcagroup.org/FORMS/WCA_list.csv"  # COTA download URL

POTA_FILE                   = APPDATA_DATA_FOLDER / "all_parks.csv"  # Local POTA list
pota_references             = {}  # Parsed POTA data
pota_url                    = "https://pota.app/all_parks.csv"  # POTA download URL

BOTA_FILE                   = APPDATA_DATA_FOLDER / "wwbota.csv"  # Local BOTA list
bota_references             = {}  # Parsed BOTA data
bota_url                    = "https://drive.usercontent.google.com/u/0/uc?id=1wZAOObnUmJTXFYPCAAqQPHgOnT3gaMa7&export=download"  # BOTA download URL

IOTA_FILE                   = APPDATA_DATA_FOLDER / "fulllist.json"  # Local IOTA list
iota_url                    = "https://www.iota-world.org/islands-on-the-air/downloads/download-file.html?path=fulllist.json"  # IOTA download URL
iota_references             = {}  # Parsed IOTA data

SOTA_FILE                   = APPDATA_DATA_FOLDER / "summitslist.csv"  # Local SOTA list
sota_url                    = "https://storage.sota.org.uk/summitslist.csv"  # SOTA download URL
sota_references             = {}  # Parsed SOTA data

ENTITY_FILE                 = DATA_FOLDER / "prefix_to_dxcc.json"  # Local prefix lookup file
prefix_to_dxcc              = {}  # Maps callsign prefixes to DXCC entities

SAT_FILE                    = DATA_FOLDER / "satellites.txt"  # Satellites definition file
POTA_MAP_FILE               = DATA_FOLDER / "dxcc_to_pota_map.json"  # DXCC→POTA reference file

FLAG_IMAGES                 = {}  # Maps DXCC to PhotoImage

tree_to_log_index           = {}  # Maps Treeview item IDs to log indices

# Global variables
tree                        = None  # Main Logviewer Tree
Logbook_Window              = None  # State: Logbook window
Edit_Window                 = None  # State: Edit window
Preference_Window           = None  # State: Preferences window
Station_Setup_Window        = None  # State: Station setup window
About_Window                = None  # State: About window
workedb4_tree               = None  # Worked Before Treeview
workedb4_frame              = None  # Worked Before frame
dxspotviewer_window         = None  # DXCluster.py window
selector_window             = None  # RST selector window
Statistics_Window           = None  # Statistics window
tracking_enabled            = True  # Frequency tracking checkbox state


# ---------------------------------------------------
#  LOGGING SETUP
# ---------------------------------------------------
log = logging.getLogger("MiniBook")
log.setLevel(logging.DEBUG)

handler = logging.FileHandler(MINIBOOK_LOG_FILE, encoding="utf-8")
formatter = logging.Formatter('%(asctime)s [%(levelname)s] %(message)s')
handler.setFormatter(formatter)

if not log.hasHandlers():
    log.addHandler(handler)

# ===============================
# HAMLIB SECTION (ERROR LOGGING)
# ===============================

hamlib_logger = logging.getLogger("Hamlib")
hamlib_logger.setLevel(logging.INFO)  # We log only errors + important state changes

hamlib_handler = RotatingFileHandler(HAMLIB_LOG_FILE, maxBytes=200_000, backupCount=5, encoding='utf-8')

hamlib_handler.setFormatter(logging.Formatter("%(asctime)s %(levelname)s: %(message)s"))
if not any(isinstance(h, RotatingFileHandler) for h in hamlib_logger.handlers):
    hamlib_logger.addHandler(hamlib_handler)


#########################################################################################
#    _   ___ ___    ___   _____ 
#   /_\ | _ \ _ \  /_\ \ / / __|
#  / _ \|   /   / / _ \ V /\__ \
# /_/ \_\_|_\_|_\/_/ \_\_| |___/
#
#########################################################################################                               

FIELD_LAYOUT = {
    # -------------------------------------------------
    # Main QSO fields
    # -------------------------------------------------
    'Date':              (0, 0, 'date'),
    'Time':              (1, 0, 'entry'),
    'Callsign':          (2, 0, 'entry'),
    'Name':              (3, 0, 'entry'),

    'Country':           (4, 0, 'entry'),
    'DXCC':              (5, 0, 'entry'),
    'CQZ':               (6, 0, 'entry'),
    'ITUZ':              (7, 0, 'entry'),

    'Band':              (8, 0, 'band'),
    'Mode':              (9, 0, 'mode'),
    'Submode':           (10, 0, 'submode'),
    'Sent':              (11, 0, 'entry'),
    'Received':          (12, 0, 'entry'),
    'Sent Exchange':     (13, 0, 'entry'),
    'Receive Exchange':  (14, 0, 'entry'),
    'Frequency':         (15, 0, 'entry'),
    'Locator':           (16, 0, 'entry'),
    'Comment':           (17, 0, 'entry'),
    'Satellite':         (18, 0, 'entry'),
    'Propagation':       (19, 0, 'prop'),

    # -------------------------------------------------
    # Award / reference fields (worked station)
    # -------------------------------------------------
    'WWFF':              (0, 2, 'entry'),
    'POTA':              (1, 2, 'entry'),
    'BOTA':              (2, 2, 'entry'),
    'COTA':              (3, 2, 'entry'),
    'IOTA':              (4, 2, 'entry'),
    'SOTA':              (5, 2, 'entry'),
    'WLOTA':             (6, 2, 'entry'),

    # -------------------------------------------------
    # My station fields
    # -------------------------------------------------
    'My Callsign':       (7, 2, 'entry'),
    'My Operator':       (8, 2, 'entry'),
    'My Locator':        (9, 2, 'entry'),
    'My Location':       (10, 2, 'entry'),
    'My WWFF':           (11, 2, 'entry'),
    'My POTA':           (12, 2, 'entry'),
    'My BOTA':           (13, 2, 'entry'),
    'My COTA':           (14, 2, 'entry'),
    'My IOTA':           (15, 2, 'entry'),
    'My SOTA':           (16, 2, 'entry'),
    'My WLOTA':          (17, 2, 'entry'),

    # -------------------------------------------------
    # QSL (paper)
    # -------------------------------------------------
    'QSL Sent':          (20, 0, 'qsl'),
    'QSL Sent Date':     (20, 2, 'date'),
    'QSL Sent Via':      (20, 4, 'qslvia'),
    'QSL Received':      (21, 0, 'qsl'),
    'QSL Received Date': (21, 2, 'date'),
    'QSL Received Via':  (21, 4, 'qslvia'),

    # -------------------------------------------------
    # LoTW
    # -------------------------------------------------
    'LoTW Sent':          (22, 0, 'qsl'),
    'LoTW Sent Date':     (22, 2, 'date'),
    'LoTW Received':      (23, 0, 'qsl'),
    'LoTW Received Date': (23, 2, 'date'),
}


# --- GUI ↔ internal field mapping for QSL, LoTW, and Serial Exchange fields ---
FIELD_ALIAS = {
    # QSL
    "QSL Sent": "QS",
    "QSL Sent Date": "QSD",
    "QSL Sent Via": "QSV",
    "QSL Received": "QR",
    "QSL Received Date": "QRD",
    "QSL Received Via": "QRV",

    # Serial Exchange
    "Sent Exchange": "STX",
    "Receive Exchange": "SRX",

    # LoTW
    "LoTW Sent": "LWS",
    "LoTW Sent Date": "LWSD",
    "LoTW Received": "LWR",
    "LoTW Received Date": "LWRD",
    "Propagation": "Prop",
    "DXCC": "dxcc",

    "CQZ":  "CQZ",
    "ITUZ": "ITUZ"
}




# -------- Operating mode options --------
mode_options = [
    "",
    "AM", "AMTORFEC", "ARDOP", "ASCI", "ATV",
    "C4FM", "CHIP", "CHIP128", "CHIP64", "CLO", "CONTESTI", "CW",
    "DIGITALVOICE", "DOMINO", "DOMINOF", "DSTAR", "DYNAMIC",
    "FAX", "FM", "FMHELL", "FSK", "FSK31", "FSK441", "FT8",
    "GTOR",
    "HELL", "HELL80", "HFSK",
    "ISCAT",
    "JT4", "JT4A", "JT4B", "JT4C", "JT4D", "JT4E", "JT4F", "JT4G",
    "JT6M", "JT9", "JT44", "JT65", "JT65A", "JT65B", "JT65C",
    "LSB",
    "MFSK", "MFSK16", "MFSK8", "MSK144", "MT63", "MTONE",
    "OLIVIA", "OPERA",
    "PAC", "PAC2", "PAC3", "PAX", "PAX2", "PCW", "PKT",
    "PSK", "PSK10", "PSK125", "PSK31", "PSK63", "PSK63F",
    "PSKAM10", "PSKAM31", "PSKAM50", "PSKFEC31", "PSKHELL",
    "Q15", "Q65", "QPSK125", "QPSK31", "QPSK63", "QRA64",
    "ROS", "RTTY", "RTTYM",
    "SSB", "SSTV",
    "T10", "THOR", "THRB", "THRBX", "TOR",
    "USB",
    "V4", "VOI",
    "WINMOR", "WSPR"
]

submode_options = [
    "",
    "AMTOR_A", "AMTOR_AUTO", "AMTOR_B", "AMTOR_FEC",
    "ARDOP_AUTO", "ARDOP_FEC", "ARDOP_PKT", "ARDOP_P2P",
    "ASCI_FAST", "ASCI_SLOW", "ASCI_ULTRA",
    "BAUDOT_45_45", "BAUDOT_50_50",
    "CHIP128", "CHIP64",
    "CLO",
    "CONTESTI",
    "DSTAR_D", "DSTAR_G", "DSTAR_R", "DSTAR_S",
    "DOMINO_11", "DOMINO_16", "DOMINO_22", "DOMINO_4", "DOMINO_44",
    "DOMINO_5", "DOMINO_8", "DOMINO_88", "DOMINO_EX", "DOMINO_F", "DOMINO_M",
    "DMR_AMBE", "DMR_Data1", "DMR_Data2", "DMR_Data3",
    "DSC",
    "DV_C4FM", "DV_DMR", "DV_DSTAR", "DV_FREEDV", "DV_M17",
    "FT4", "FT8", "FTx",
    "HELL_100", "HELL_150", "HELL_200", "HELL_50", "HELL_75",
    "MFSK_1024", "MFSK_128", "MFSK_16", "MFSK_256", "MFSK_32",
    "MFSK_512", "MFSK_64", "MFSK_8", "MFSK_FST4", "MFSK_Q65",
    "MSK144", "MSK441",
    "PSK_1000", "PSK_125", "PSK_250", "PSK_31", "PSK_500", "PSK_63",
    "QPSK_1000", "QPSK_125", "QPSK_250", "QPSK_31", "QPSK_500", "QPSK_63",
    "RTTY45", "RTTY50", "RTTY75",
    "SSTV_BW256", "SSTV_BW512",
    "SCAMP_FAST", "SCAMP_SLOW", "SCAMP_VSLOW",
    "TOR_1000", "TOR_200", "TOR_500",
    "VARA_FM_1200", "VARA_FM_9600", "VARA_HF", "VARA_SATELLITE",
    "WINMOR_1700", "WINMOR_900",
    "WSPR_1", "WSPR_2", "WSPR_4", "WSPR_8"
]

# Initialize last_passband globally, starting with a default value (e.g., 0 or your choice)
last_passband = 0

# -------- RST Sent/Received options -------
rst_options         = [str(i) for i in range(51, 60)] + ["59+10dB", "59+20dB", "59+30dB", "59+40dB"]



# Frequency ranges and default frequencies for each band
band_ranges = {
    "2200m": (0.1357, 0.1358),
    "160m": (1.8, 2.0),
    "80m": (3.5, 4.0),
    "60m": (5.3515, 5.3665),
    "40m": (7.0, 7.3),
    "30m": (10.1, 10.15),
    "20m": (14.0, 14.35),
    "17m": (18.068, 18.168),
    "15m": (21.0, 21.45),
    "12m": (24.89, 24.99),
    "11m": (26.0, 27.99),
    "10m": (28.0, 29.97),
    "6m": (50.0, 54.0),
    "4m": (70.0, 70.5),
    "2m": (144.0, 148.0),
    "1.25m": (220.0, 225.0),
    "70cm": (420.0, 450.0),
    "33cm": (902.0, 928.0),
    "23cm": (1200.0, 1300.0),
    "13cm": (2300.0, 2450.0),
}

# Mapping from band to default frequency
band_to_frequency = {
    "2200m": "0.1356",
    "160m": "1.8500",
    "80m": "3.6500",
    "60m": "5.3600",
    "40m": "7.0000",
    "30m": "10.1000",
    "20m": "14.0000",
    "17m": "18.1000",
    "15m": "21.0000",
    "12m": "24.8900",
    "11m": "27.0000",
    "10m": "28.0000",
    "6m": "50.0000",
    "4m": "70.0000",
    "2m": "144.0000",
	"1.25m": "220.0000",
    "70cm": "430.0000",
	"33cm": "902.0000",
    "23cm": "1296.0000",
    "13cm": "2300.0000"
}

# Function to update frequency based on band selection
def update_frequency_from_band(*args):
    selected_band = qso_band_var.get()
    default_frequency = band_to_frequency.get(selected_band, "")
    qso_frequency_var.set(default_frequency)

# Function to check frequency and update the band if needed
def update_band_from_frequency(*args):
    frequency = qso_frequency_var.get()
    
    try:
        frequency = float(frequency)
        for band, (low, high) in band_ranges.items():
            if low <= frequency <= high:
                # Frequency is within the band range, set the band
                qso_band_var.set(band)
                break
        else:
            qso_band_var.set("OOB")
            # If frequency is outside the known bands, don't change the band show Out Of Band
            pass
    except ValueError:
        pass  # Ignore invalid frequency input






#########################################################################################
#  ___ _   _ _  _  ___ _____ ___ ___  _  _ ___ 
# | __| | | | \| |/ __|_   _|_ _/ _ \| \| / __|
# | _|| |_| | .` | (__  | |  | | (_) | .` \__ \
# |_|  \___/|_|\_|\___| |_| |___\___/|_|\_|___/
#                                              
#########################################################################################


def update_datetime():
    """Update QSO date and time depending on auto/manual UTC settings."""
    if datetime_tracking_enabled.get():  # Only update if tracking is enabled
        try:
            utc_auto = config.getboolean("Global_settings", "utc_auto", fallback=True)

            if utc_auto:
                # Auto mode: show true UTC time
                now = datetime.now(timezone.utc)
            else:
                # Manual mode: show local time + manual offset
                try:
                    offset_hours = int(utc_offset_var.get())
                except ValueError:
                    offset_hours = 0
                    print("Invalid UTC offset, defaulting to 0.")
                now = datetime.now() + timedelta(hours=offset_hours)

            qso_time_var.set(now.strftime("%H:%M:%S"))
            qso_date_var.set(now.strftime("%Y-%m-%d"))

        except Exception as e:
            print(f"Error updating datetime: {e}")

        root.after(1000, update_datetime)




def open_log_folder():
    """Open the MiniBook log folder in Explorer."""
    try:
        if os.path.exists(LOG_FOLDER):
            os.startfile(LOG_FOLDER)   # Windows native
        else:
            messagebox.showwarning("Logs", "Log folder does not exist.")
    except Exception as e:
        messagebox.showerror("Logs", f"Unable to open log folder:\n{e}")

# Creates Tooltip information window
# Example: ToolTip(callsign_entry,"TEXT")
class ToolTip:
    def __init__(self, widget, text, delay=1000):
        self.widget = widget
        self.text = text
        self.delay = delay      # delay in ms
        self.tipwindow = None
        self.after_id = None

        widget.bind("<Enter>", self.schedule)
        widget.bind("<Leave>", self.hide)
        widget.bind("<ButtonPress>", self.hide)  # close when clicked

    def schedule(self, event=None):
        self.after_id = self.widget.after(self.delay, self.show)

    def show(self, event=None):
        if self.tipwindow or not self.widget.winfo_exists():
            return

        x = self.widget.winfo_rootx() + 20
        y = self.widget.winfo_rooty() + self.widget.winfo_height() + 5

        self.tipwindow = tw = tk.Toplevel(self.widget)
        tw.wm_overrideredirect(True)
        tw.geometry(f"+{x}+{y}")

        label = tk.Label(tw, text=self.text, justify="left", background="#ffffe0", relief="solid", borderwidth=1, font=("Arial", 9))

        label.pack(ipadx=4, ipady=2)

    def hide(self, event=None):
        if self.after_id:
            self.widget.after_cancel(self.after_id)
            self.after_id = None

        if self.tipwindow:
            self.tipwindow.destroy()
            self.tipwindow = None





# Window center function
def center_window_over(parent, window, width, height):
    parent.update_idletasks()
    px = parent.winfo_rootx()
    py = parent.winfo_rooty()
    pw = parent.winfo_width()
    ph = parent.winfo_height()

    x = px + (pw // 2) - (width // 2)
    y = py + (ph // 2) - (height // 2)

    window.geometry(f"{width}x{height}+{x}+{y}")


# Resizes flag image to given size in PX
def resize_flag(img, size):
    """Resize with Pillow, exact pixels."""
    if img is None:
        return None

    # Convert Tk PhotoImage -> PIL image
    pil = Image.open(img.cget("file"))

    w, h = pil.size

    # new height = size, keep aspect ratio
    new_h = size
    new_w = int(w * (new_h / h))

    pil = pil.resize((new_w, new_h), Image.LANCZOS)

    return ImageTk.PhotoImage(pil)



def bind_jump_to_letter(combobox):
    """
    Makes the Combobox jump to the next value starting with typed letter.
    Repeated presses cycle through matching entries.
    """

    last_keypress = {"char": "", "time": 0, "index": 0}

    def on_key(event):
        char = event.char.upper()
        if not char:
            return

        values = combobox["values"]
        now = time.time()

        # Find all matches (starts with this letter)
        matches = [i for i, v in enumerate(values) if v.upper().startswith(char)]

        if not matches:
            return

        # If same key pressed within 1 second -> cycle
        if last_keypress["char"] == char and now - last_keypress["time"] < 1:
            last_keypress["index"] = (last_keypress["index"] + 1) % len(matches)
        else:
            last_keypress["index"] = 0

        match_index = matches[last_keypress["index"]]
        last_keypress["char"] = char
        last_keypress["time"] = now

        combobox.current(match_index)
        combobox.event_generate("<<ComboboxSelected>>")

    combobox.bind("<Key>", on_key)




# RST Selector Window
def open_selector(parent, target_var, options, title="Select Value", btn_width=10, btn_height=3, columns=4):

    global selector_window

    # If a window already exists → bring it to the front
    if selector_window is not None and selector_window.winfo_exists():
        selector_window.lift()
        selector_window.focus_force()
        return

    selector_window = tk.Toplevel(parent)
    selector_window.title(title)
    selector_window.resizable(False, False)

    # Bold font
    bold_font = tkFont.Font(weight="bold")

    # Function to select a value
    def select_value(v):
        global selector_window
        target_var.set(v)
        if selector_window is not None:
            selector_window.destroy()
            selector_window = None

    # Button grid
    for i, val in enumerate(options):
        btn = tk.Button(
            selector_window,
            text=val,
            width=btn_width,
            height=btn_height,
            font=bold_font,
            bg="#00FF80",     # background color
            fg="#000000",     # text color
            activebackground="#00AA80",  # background on click
            activeforeground="#000000",  # text color on click
            command=lambda v=val: select_value(v)
        )
        btn.grid(row=i // columns, column=i % columns, padx=5, pady=5)

    # Automatic window size
    selector_window.update_idletasks()
    win_w = selector_window.winfo_reqwidth()
    win_h = selector_window.winfo_reqheight()

    # Center on parent
    x = parent.winfo_rootx() + (parent.winfo_width() // 2) - (win_w // 2)
    y = parent.winfo_rooty() + (parent.winfo_height() // 2) - (win_h // 2)
    selector_window.geometry(f"{win_w}x{win_h}+{x}+{y}")

    # Close function
    def on_close():
        global selector_window
        if selector_window is not None:
            selector_window.destroy()
            selector_window = None

    selector_window.protocol("WM_DELETE_WINDOW", on_close)



# Functions used with dxspotview.py
def get_worked_calls():
    return {qso.get("Callsign", "").upper() for qso in qso_lines}

def get_worked_calls_today():
    today_str = date.today().isoformat()
    return {
        qso.get("Callsign", "").upper()
        for qso in qso_lines
        if qso.get("Date") == today_str
    }

def open_dxspotviewer():
    global dxspotviewer_window
    if dxspotviewer_window is not None and dxspotviewer_window.winfo_exists():
        dxspotviewer_window.lift()
        return

    dxspotviewer_window = tk.Toplevel()
    dxspotviewer_window.resizable(False, False)

    def get_last_qso_callsign():
        return qso_callsign_var.get().strip()

    def get_current_frequency():
        return qso_frequency_var.get().strip()

    launch_dx_spot_viewer(
        rigctl_host=hamlib_ip_var.get().strip() or "127.0.0.1",
        rigctl_port=int(hamlib_port_var.get()) if str(hamlib_port_var.get()).isdigit() else 4532,
        tracking_var=freqmode_tracking_var,

        on_callsign_selected=lambda *a, **k: handle_callsign_input(
            source="spot",
            callsign=k.get("callsign") if "callsign" in k else (a[0] if len(a) > 0 else None),
            freq=k.get("freq") if "freq" in k else (a[1] if len(a) > 1 else None),
            mode=k.get("mode") if "mode" in k else (a[2] if len(a) > 2 else None),
        ),

        get_worked_calls=get_worked_calls,
        get_worked_calls_today=get_worked_calls_today,
        get_last_qso_callsign=get_last_qso_callsign,
        get_current_frequency=get_current_frequency,
        parent_window=dxspotviewer_window
    )





def send_callsign_to_dxcluster(event=None):
    """Wordt aangeroepen wanneer TAB wordt gedrukt in callsign_entry.
       Stuurt callsign, freq en comment naar DXCluster venster indien open."""
    global dxspotviewer_window
    if dxspotviewer_window is not None and dxspotviewer_window.winfo_exists():
        if hasattr(dxspotviewer_window, "dxcluster_app"):
            app = dxspotviewer_window.dxcluster_app
            app.spot_callsign_var.set(qso_callsign_var.get())
            app.spot_freq_var.set(qso_frequency_var.get())
            app.spot_comment_var.set(qso_comment_var.get())



# Function to update Title
def update_title(root, version, filename=None, status_text=None):
    parts = [f"MiniBook {version}"]
    if filename:
        parts.append(f"[{os.path.basename(filename)}]")
    if status_text:
        parts.append(status_text)
    root.title(" - ".join(parts))



def add_qso_row(qso):
    """Insert 1 QSO at top in the Logbook Tree and rebuild index mapping."""
    if tree is None or not tree.winfo_exists():
        return
    
    children = tree.get_children()
    tag = "evenrow" if (len(children) % 2 == 0) else "oddrow"    

    # Insert one row at top
    tree.insert("", 0, values=(
        qso.get("Date",""),
        qso.get("Time",""),
        qso.get("Callsign",""),
        qso.get("Name",""),
        qso.get("Country",""),
        qso.get("CQZ",""),
        qso.get("ITUZ",""),
        qso.get("Sent",""),
        qso.get("Received",""),
        qso.get("Mode",""),
        qso.get("Submode",""),
        qso.get("Band",""),
        qso.get("Frequency",""),
        qso.get("Locator",""),
        qso.get("Comment",""),
        qso.get("Satellite",""),
        qso.get("Prop",""),
        qso.get("QS",""),
        qso.get("QR",""),
        qso.get("LWS",""),
        qso.get("LWR",""),
    ), tags=(tag,))

    print("INSERT TAGS:", tree.item(tree.get_children()[0], "tags"))

    rebuild_index_map()

    # Update counter directly
    if qso_count_label:
        qso_count_label.config(text=f"Total of {len(qso_lines)} QSO's in logbook")    



def rebuild_index_map():
    global tree_to_log_index, qso_lines

    tree_to_log_index = {}
    for idx, iid in enumerate(tree.get_children()):
        tree_to_log_index[iid] = idx


# Function to reset variables and entries when logbook file failed to load.
def no_file_loaded():
    global CURRENT_JSON_FILE

    CURRENT_JSON_FILE = None # Reset the CURRENT_JSON_FILE to None
    update_title(root, VERSION_NUMBER, "Load or create logbook first!", radio_status_var.get())
    station_locator_var.set("")
    station_callsign_var.set("")
    station_operator_var.set("")
    station_location_var.set("")
    station_bota_var.set("")
    station_cota_var.set("")
    station_iota_var.set("")
    station_pota_var.set("")
    station_sota_var.set("")
    station_wlota_var.set("")
    station_wwff_var.set("")


    station_callsign_entry.config(textvariable=station_callsign_var)
    station_operator_entry.config(textvariable=station_operator_var)
    station_locator_entry.config(textvariable=station_locator_var)
    station_location_entry.config(textvariable=station_location_var)
    station_wwff_entry.config(textvariable=station_wwff_var)
    station_pota_entry.config(textvariable=station_pota_var)
    station_bota_entry.config(textvariable=station_bota_var)
    station_cota_entry.config(textvariable=station_cota_var)
    station_wlota_entry.config(textvariable=station_wlota_var)
    station_iota_entry.config(textvariable=station_iota_var)
    station_sota_entry.config(textvariable=station_sota_var)

    file_menu.entryconfig("Station setup", state="disabled")


# Function to restore fields after logging or wiping
def reset_fields():
    qso_comment_var.set("")
    last_qso_label.config(text="")
    QRZ_status_label.config(text="")
    qso_callsign_var.set("")
    qso_name_var.set("")
    qrz_city_var.set("")
    qrz_zipcode_var.set("")
    qrz_address_var.set("")
    qrz_qsl_info_var.set("")
    qso_satellite_var.set("")
    qso_prop_mode_var.set("")
    qso_submode_var.set("")
    on_mode_change() 
    qso_locator_var.set("")    
    callsign_entry.focus_set()
    wwff_var.set("")
    wwff_park_name_var.set("")
    pota_var.set("")
    pota_park_name_var.set("")
    bota_name_var.set("")
    bota_var.set("")
    iota_var.set("")
    iota_name_var.set("")
    sota_var.set("")
    wlota_var.set("")
    bota_var.set("")
    cota_var.set("")
    qso_receive_exchange_var.set("")
    sota_name_var.set("")
    sota_matches['values'] = ()
    sota_matches.set("")





# Function to clear My Station Parameters in GUI
def clear_station_labels():
    # Clear station info StringVars
    station_callsign_var.set("")
    station_operator_var.set("")
    station_locator_var.set("")
    station_location_var.set("")
    station_wwff_var.set("")
    station_pota_var.set("")
    station_bota_var.set("")
    station_cota_var.set("")
    station_iota_var.set("")
    station_wlota_var.set("")
    station_sota_var.set("")



#Checks if a given maidenhead locator is valid for example JO22LO49
def is_valid_locator(locator):
    if locator == "":
        return True
    if len(locator) < 4 or len(locator) % 2 != 0:
        return False
    # Regex pattern:
    # - [A-R]{2} : veld
    # - \d{2} : vierkant
    # - ([A-X]{2})? : optioneel subsquare
    # - (\d{2})? : optioneel extended vierkant (cijfers)
    pattern = r'^[A-R]{2}\d{2}([A-X]{2})?(\d{2})?$'
    return bool(re.match(pattern, locator, re.IGNORECASE))



#Converts Maidenhead locator to Latttitude and Longitude coordinates
def maidenhead_to_latlon(locator):
    """
    Convert Maidenhead locator (2/4/6/8 characters) to (lat, lon) at the center of the grid square.
    Returns: (latitude, longitude) as floats rounded to 6 decimals.
    """

    loc = locator.strip().upper()
    if len(loc) < 2:
        raise ValueError("Locator must be at least 2 characters")

    # basis (fields)
    lon = (ord(loc[0]) - ord('A')) * 20.0 - 180.0
    lat = (ord(loc[1]) - ord('A')) * 10.0 - 90.0

    # squares (digits)
    if len(loc) >= 4:
        lon += int(loc[2]) * 2.0
        lat += int(loc[3]) * 1.0

    # subsquares (letters A-X)
    if len(loc) >= 6:
        # 2 degrees / 24 = 5' = 0.083333333... ; 1 degree / 24 = 2.5' = 0.041666666...
        lon += (ord(loc[4]) - ord('A')) * (2.0 / 24.0)
        lat += (ord(loc[5]) - ord('A')) * (1.0 / 24.0)

    # optional extended digits (7/8 chars)
    if len(loc) >= 8:
        lon += int(loc[6]) * (2.0 / 24.0 / 10.0)
        lat += int(loc[7]) * (1.0 / 24.0 / 10.0)

    # center-offset afhankelijk van precisie
    if len(loc) >= 8:
        lon += (2.0 / 24.0 / 10.0) / 2.0
        lat += (1.0 / 24.0 / 10.0) / 2.0
    elif len(loc) >= 6:
        lon += (2.0 / 24.0) / 2.0
        lat += (1.0 / 24.0) / 2.0
    elif len(loc) >= 4:
        lon += 1.0
        lat += 0.5
    else:
        lon += 10.0
        lat += 5.0

    return round(lat, 6), round(lon, 6)
    

# Opens OpenStreetMap and shows pins on map with reference number and name, with a line showing distance in tooltip
def open_osm_map(lat_var, long_var, station_locator_var, station_callsign_var, ref, name):
    station_locator = station_locator_var.get().strip()
    station_callsign = station_callsign_var.get().strip().upper()

    lat = lat_var.get().strip()
    lon = long_var.get().strip()

    if lat and lon:
        ref_escaped = html.escape(ref).replace("'", "\\'")
        name_escaped = html.escape(name).replace("'", "\\'")

        station_lat, station_lon = None, None
        if station_locator:  # alleen omzetten als er iets is ingevuld
            try:
                station_lat, station_lon = maidenhead_to_latlon(station_locator)
            except Exception:
                station_lat, station_lon = None, None

        # Kies startpositie van de kaart
        if station_lat and station_lon:
            map_center = f"[{station_lat}, {station_lon}]"
        else:
            map_center = f"[{lat}, {lon}]"

        html_template = f"""<!DOCTYPE html>
        <html>
        <head>
          <title>SOTA Reference Map</title>
          <meta charset="utf-8" />
          <meta name="viewport" content="width=device-width, initial-scale=1.0">

          <link rel="stylesheet" href="https://unpkg.com/leaflet/dist/leaflet.css"/>
          <script src="https://unpkg.com/leaflet/dist/leaflet.js"></script>

          <style>
            #map {{ height: 100vh; margin:0; padding:0; }}
            html, body {{ height: 100%; margin:0; padding:0; }}
            .leaflet-tooltip {{
                background: white;
                border: 1px solid black;
                border-radius: 4px;
                padding: 2px 5px;
                font-size: 12px;
                font-weight: bold;
            }}
          </style>
        </head>
        <body>
          <div id="map"></div>
          <script>
            var map = L.map('map').setView({map_center}, 10);

            var osm = L.tileLayer('https://{{s}}.tile.openstreetmap.org/{{z}}/{{x}}/{{y}}.png', {{
              maxZoom: 19,
              attribution: '© OpenStreetMap contributors'
            }}).addTo(map);

            var esriSat = L.tileLayer('https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{{z}}/{{y}}/{{x}}', {{
              maxZoom: 19,
              attribution: 'Tiles © Esri'
            }});

            var esriLabels = L.tileLayer('https://server.arcgisonline.com/ArcGIS/rest/services/Reference/World_Boundaries_and_Places/MapServer/tile/{{z}}/{{y}}/{{x}}', {{
              maxZoom: 19,
              attribution: 'Labels © Esri',
              pane: 'overlayPane'
            }});

            var esriHybrid = L.layerGroup([esriSat, esriLabels]);

            var redIcon = L.icon({{
              iconUrl: 'https://raw.githubusercontent.com/pointhi/leaflet-color-markers/master/img/marker-icon-red.png',
              shadowUrl: 'https://unpkg.com/leaflet@1.9.3/dist/images/marker-shadow.png',
              iconSize:     [25, 41],
              iconAnchor:   [12, 41],
              popupAnchor:  [1, -34],
              shadowSize:   [41, 41]
            }});

            var marker2 = L.marker([{lat}, {lon}], {{icon: redIcon}})
                .addTo(map)
                .bindPopup('<b>{ref_escaped}</b><br>{name_escaped}')
                .openPopup();
        """

        # Alleen station tonen als locator geldig is
        if station_lat and station_lon:
            html_template += f"""
            var blueIcon = L.icon({{
              iconUrl: 'https://unpkg.com/leaflet@1.9.3/dist/images/marker-icon.png',
              shadowUrl: 'https://unpkg.com/leaflet@1.9.3/dist/images/marker-shadow.png',
              iconSize:     [25, 41],
              iconAnchor:   [12, 41],
              popupAnchor:  [1, -34],
              shadowSize:   [41, 41]
            }});

            var marker1 = L.marker([{station_lat}, {station_lon}], {{icon: blueIcon}})
                .addTo(map)
                .bindPopup('Your Location:<br><b>{station_callsign}</b>');

            var latlngs = [
                [{station_lat}, {station_lon}],
                [{lat}, {lon}]
            ];
            var polyline = L.polyline(latlngs, {{color: 'green'}}).addTo(map);

            // ---- afstand in km én miles ----
            var distanceKm = map.distance(
                L.latLng({station_lat}, {station_lon}),
                L.latLng({lat}, {lon})
            ) / 1000.0;
            var distanceMi = distanceKm * 0.621371;

            var distanceKmTxt = distanceKm.toFixed(1);
            var distanceMiTxt = distanceMi.toFixed(1);

            polyline.bindTooltip(distanceKmTxt + ' km / ' + distanceMiTxt + ' mi', {{
                permanent: true,
                direction: 'center',
                className: 'distance-tooltip'
            }}).openTooltip();

            var group = new L.featureGroup([marker1, marker2, polyline]);
            map.fitBounds(group.getBounds().pad(0.5));
            """

        html_template += """
            var baseMaps = {
                "OpenStreetMap": osm,
                "Satelliet": esriSat,
                "Satelliet + Labels": esriHybrid
            };
            L.control.layers(baseMaps).addTo(map);
          </script>
        </body>
        </html>
        """

        # FIX: schrijf bestand altijd als UTF-8
        with tempfile.NamedTemporaryFile('w', delete=False, suffix='.html', encoding="utf-8") as f:
            f.write(html_template)
            temp_file_path = f.name

        webbrowser.open(f"file://{temp_file_path}")

    else:
        messagebox.showwarning("No Coordinates", "No valid coordinates found.")



  




def on_mode_change(event=None):
    selected_mode = qso_mode_var.get().upper()
    if selected_mode in ["CW", "CW-R", "RTTY", "OLIVIA", "PSK31", "PSK64", "PSK125", "MFSK", "DOMINO", "VARA", "SSTV"]:
        qso_rst_sent_var.set("599")
        qso_rst_received_var.set("599")
    else:
        qso_rst_sent_var.set(rst_options[8])
        qso_rst_received_var.set(rst_options[8])





def edit_satellite_names():
    """Open an editor window for satellites.txt (only one instance) and refresh the dropdown after saving."""
    import tkinter as tk
    from tkinter import messagebox

    global satellite_editor_window

    # Prevent multiple instances
    if 'satellite_editor_window' in globals() and satellite_editor_window is not None:
        try:
            if satellite_editor_window.winfo_exists():
                satellite_editor_window.lift()
                satellite_editor_window.focus_force()
                return
        except:
            pass

    editor = tk.Toplevel()
    satellite_editor_window = editor
    editor.title("Edit Satellite Names")
    editor.geometry("400x500")
    editor.minsize(400, 400)
    editor.grab_set()  # Modal

    # Center over main window
    main_x = root.winfo_rootx()
    main_y = root.winfo_rooty()
    main_w = root.winfo_width()
    main_h = root.winfo_height()
    win_w, win_h = 400, 500
    pos_x = main_x + (main_w // 2) - (win_w // 2)
    pos_y = main_y + (main_h // 2) - (win_h // 2)
    editor.geometry(f"{win_w}x{win_h}+{pos_x}+{pos_y}")

    text_area = tk.Text(editor, wrap='word', font=('Consolas', 11))
    text_area.pack(expand=True, fill='both', padx=10, pady=10)

    satellites_file = os.path.join(SAT_FILE)

    # Load file contents
    try:
        with open(satellites_file, "r", encoding="utf-8") as f:
            text_area.insert("1.0", f.read())
    except FileNotFoundError:
        text_area.insert("1.0", "# Add one satellite name per line\n")

    def save_satellite_names():
        content = text_area.get("1.0", "end").strip()
        try:
            with open(satellites_file, "w", encoding="utf-8") as f:
                f.write(content + "\n")
        except Exception as e:
            messagebox.showerror("Error", f"Could not save file:\n{e}")
            return

        # Refresh dropdown list
        updated_list = load_satellite_names()
        updated_list.insert(0, "")
        satellite_dropdown["values"] = updated_list
        messagebox.showinfo("Saved", "Satellite list updated successfully.")
        editor.destroy()

    def on_close():
        global satellite_editor_window
        satellite_editor_window = None
        editor.destroy()

    # Buttons frame
    btn_frame = tk.Frame(editor)
    btn_frame.pack(fill='x', padx=10, pady=5)
    tk.Button(btn_frame, text="Save", command=save_satellite_names, width=10).pack(side='right', padx=5)
    tk.Button(btn_frame, text="Cancel", command=on_close, width=10).pack(side='right')

    editor.protocol("WM_DELETE_WINDOW", on_close)




def load_satellite_names(filename=SAT_FILE):
    """Load satellite names from file, create with defaults if not found."""
    default_satellites = [
        "AO-91 (Fox-1B)","AO-92 (Fox-1D)","AO-85 (Fox-1A)","SO-50","PO-101 (Diwata-2)","CAS-4A","CAS-4B","XW-2A","XW-2B","XW-2C","XW-2D","ISS","Tevel-1","Tevel-2","Tevel-3","Tevel-4","Tevel-5","Tevel-6","Tevel-7","Tevel-8","QO-100"
    ]

    try:
        with open(filename, "r", encoding="utf-8") as file:
            return [line.strip() for line in file if line.strip()]
    except FileNotFoundError:
        # Ensure data folder exists
        data_dir = os.path.dirname(filename)
        if data_dir and not os.path.exists(data_dir):
            os.makedirs(data_dir)

        # Create new satellites.txt with defaults
        try:
            with open(filename, "w", encoding="utf-8") as f:
                f.write("\n".join(default_satellites) + "\n")
        except Exception as e:
            print(f"Error creating satellites.txt: {e}")
            return []

        # Show popup once file is created
        try:
            import tkinter.messagebox as messagebox
            messagebox.showinfo("Satellite List Created",
                                "File 'satellites.txt' was not found and has been created "
                                "with a default list of popular amateur satellites.")
        except:
            print("satellites.txt created with default satellites.")

        return default_satellites




# Function to update GUI status
# 0 = dummy
# 10 = connecting
# 11 = connected
# 12 = disconnected
# 20 = noradio
def gui_state_control(status):
    global radio_status_var, tracking_enabled, hamlib_ip_var, hamlib_port_var

    if status == 11:   # CONNECTED
        radio_status_var.set(f"Connected to {hamlib_ip_var.get()}:{hamlib_port_var.get()}")

        # enable checkbox
        tracking_checkbox.config(state="normal")

        # force tracking into ON state on connect
        tracking_enabled_var.set(True)
        tracking_enabled = True

        freq_entry.config(state='readonly', fg="red")

        band_combobox.config(state='disabled')


    elif status == 12:  # DISCONNECTED
        radio_status_var.set("Disconnected")

        # disable checkbox + turn off
        tracking_checkbox.config(state="disabled")
        tracking_enabled_var.set(False)

        # freq edit manually ok
        freq_entry.config(state='normal', fg="black")

        band_combobox.config(state='normal')

    elif status == 10:
        radio_status_var.set("Connecting...")

    else:
        radio_status_var.set("Idle")

    update_title(root, VERSION_NUMBER, CURRENT_JSON_FILE, radio_status_var.get())




def refresh_sent_serial_from_logbook():
    """
    Recalculate the next STX value based on the current logbook.
    Called after add/edit/delete operations.
    """
    if not use_serial_var.get():
        log.debug("STX refresh skipped: Use Serial is disabled")
        return

    if not CURRENT_JSON_FILE:
        log.debug("STX refresh skipped: no CURRENT_JSON_FILE")
        return

    try:
        with open(CURRENT_JSON_FILE, "r", encoding="utf-8") as file:
            data = json.load(file)
            logbook = data.get("Logbook", [])

        max_stx = 0
        for qso in logbook:
            stx_val = str(qso.get("STX", "")).strip()
            if stx_val.isdigit():
                max_stx = max(max_stx, int(stx_val))

        next_stx = max_stx + 1 if max_stx else 1
        qso_sent_exchange_var.set(str(next_stx))

        log.debug(
            f"STX refreshed from logbook: highest={max_stx}, next={next_stx}, "
            f"qsos={len(logbook)}"
        )

    except Exception as e:
        log.warning(f"STX refresh failed: {e}")




    
    
def open_backup_folder():
    backup_dir = config.get(
        "General",
        "backup_folder",
        fallback=str(BACKUP_FOLDER)
    )

    Path(backup_dir).mkdir(parents=True, exist_ok=True)

    try:
        if sys.platform == "win32":
            os.startfile(backup_dir)
        elif sys.platform == "darwin":
            subprocess.Popen(["open", backup_dir])
        else:
            subprocess.Popen(["xdg-open", backup_dir])
    except Exception as e:
        messagebox.showerror(
            "Error",
            f"Could not open the backup folder:\n{e}"
        )




def calculate_headings(lat1, lon1, lat2, lon2):
    """
    Calculate short path and long path headings between two points (in degrees).
    """
    # Convert to radians
    lat1_rad = math.radians(lat1)
    lat2_rad = math.radians(lat2)
    delta_lon_rad = math.radians(lon2 - lon1)

    x = math.sin(delta_lon_rad) * math.cos(lat2_rad)
    y = math.cos(lat1_rad) * math.sin(lat2_rad) - \
        math.sin(lat1_rad) * math.cos(lat2_rad) * math.cos(delta_lon_rad)

    initial_bearing_rad = math.atan2(x, y)
    initial_bearing_deg = (math.degrees(initial_bearing_rad) + 360) % 360  # SP

    long_path = (initial_bearing_deg + 180) % 360  # LP is SP + 180°

    return round(initial_bearing_deg), round(long_path)        
    


#########################################################################################
#  _    ___   ___ ___  ___   ___  _  __     _ ___  ___  _  _ 
# | |  / _ \ / __| _ )/ _ \ / _ \| |/ /  _ | / __|/ _ \| \| |
# | |_| (_) | (_ | _ \ (_) | (_) | ' <  | || \__ \ (_) | .` |
# |____\___/ \___|___/\___/ \___/|_|\_\  \__/|___/\___/|_|\_|
#                                                            
#########################################################################################
def create_new_json():
    global CURRENT_JSON_FILE, Logbook_Window, qso_lines, duplicate_index_map, tree_to_log_index

    if CURRENT_JSON_FILE:
        response = messagebox.askquestion(
            "Confirmation",
            "This will close the currently loaded logbook\n\nAre you sure?"
        )
        if response == 'no':
            return

    # ------------------------------------------------------------------
    # 1) Close logbook window FIRST so geometry is saved correctly
    # ------------------------------------------------------------------
    if Logbook_Window is not None and Logbook_Window.winfo_exists():
        try:
            # Use the global close_logbook helper so it calls save_window_geometry()
            close_logbook()
        except Exception:
            # Hard fallback if something goes wrong in close_logbook
            Logbook_Window.destroy()
            Logbook_Window = None

    # ------------------------------------------------------------------
    # 2) Clear loaded json file and all related data
    # ------------------------------------------------------------------
    no_file_loaded()
    qso_lines = []
    duplicate_index_map = {}
    tree_to_log_index = {}

    # Reset serial usage
    try:
        use_serial_var.set(False)
        qso_sent_exchange_var.set("1")
        toggle_serial_fields()
    except NameError:
        print("DEBUG: use_serial_var or serial entries not defined")

    # ------------------------------------------------------------------
    # 3) Ask for new file path
    # ------------------------------------------------------------------
    CURRENT_JSON_FILE = filedialog.asksaveasfilename(
        defaultextension=".mbk",
        filetypes=[("MiniBook files", "*.mbk")]
    )
    if not CURRENT_JSON_FILE:
        return

    # --- Initialize JSON structure with BookInfo + full Station skeleton ---
    data = {
        "BookInfo": {"Version": str(BOOK_VERSION)},
        "Station": {
            "Callsign": "",
            "Operator": "",
            "Locator": "",
            "Location": "",
            "Name": "",
            "Street": "",
            "Postalcode": "",
            "City": "",
            "County": "",
            "Country": "",
            "CQ Zone": "",
            "ITU Zone": "",
            "Contest": "",
            "BOTA": "",
            "COTA": "",
            "IOTA": "",
            "POTA": "",
            "SOTA": "",
            "WLOTA": "",
            "WWFF": "",
            "QRZAPI": "",
            "QRZUpload": False
        },
        "Logbook": []
    }

    # Save the new JSON (compact & fast)
    try:
        with open(CURRENT_JSON_FILE, 'w', encoding='utf-8') as file:
            json.dump(data, file, ensure_ascii=False, separators=(",", ":"))
    except Exception as e:
        messagebox.showerror("Error", f"Could not create new logbook:\n{e}")
        return

    # ------------------------------------------------------------------
    # 4) Reload INI from disk so the geometry saved by close_logbook()
    #    is merged into the global config BEFORE we write last_loaded_logbook
    # ------------------------------------------------------------------
    try:
        if CONFIG_FILE.exists():
            config_file_path = CONFIG_FILE
        else:
            config_file_path = DATA_FOLDER / "minibook.ini"

        # Reload the latest version of the INI (with new LogbookWindow geometry)
        config.read(config_file_path, encoding='utf-8')

        if not config.has_section('General'):
            config.add_section('General')
        config.set('General', 'last_loaded_logbook', CURRENT_JSON_FILE)

        with open(config_file_path, 'w', encoding='utf-8') as cfgfile:
            config.write(cfgfile)
    except Exception as e:
        print(f"WARNING: Could not update last_loaded_logbook in config: {e}")

    # ------------------------------------------------------------------
    # 5) Update GUI title and station setup
    # ------------------------------------------------------------------
    update_title(root, VERSION_NUMBER, CURRENT_JSON_FILE, radio_status_var.get())
    load_station_setup()

    # Enable Station setup menu item again
    try:
        file_menu.entryconfig("Station setup", state="normal")
    except Exception:
        # file_menu might not be defined in some contexts
        pass





def load_last_logbook_on_startup():
    global Logbook_Window, logbook_window_open, qso_lines, duplicate_index_map, tree_to_log_index, CURRENT_JSON_FILE

    # Check if reload is enabled in config
    reload_enabled = config.getboolean('General', 'reload_last_logbook', fallback=False)
    if not reload_enabled:
        return  # User chose not to reload last logbook on startup

    # Proceed if there's a file to load
    last_file = config.get('General', 'last_loaded_logbook', fallback=None)
    if last_file and os.path.exists(last_file):
        # Clear old logbook data
        qso_lines = []
        duplicate_index_map = {}
        tree_to_log_index = {}
        CURRENT_JSON_FILE = None

        # Close logbook window if open
        if Logbook_Window is not None and Logbook_Window.winfo_exists():
            Logbook_Window.destroy()
            Logbook_Window = None

        # Load the last JSON file
        load_json(last_file)

def normalize_qso_fields(qso: dict) -> dict:
    """Convert GUI names to internal names (for saving to JSON or ADIF export)."""
    qso = dict(qso)
    for gui_name, internal_name in FIELD_ALIAS.items():
        if gui_name in qso:
            qso[internal_name] = qso.pop(gui_name)
    return qso


def convert_log(old_data, station_callsign="N0CALL"):
    """Convert old logbook structure to new internal short-field format, including DXCC entity mapping."""
    global BOOK_VERSION, VERSION_NUMBER

    # --- Determine if conversion is needed ---
    book_info = old_data.get("BookInfo")
    if book_info is None:
        old_version = 0.0
        need_conversion = True
    else:
        try:
            old_version = float(book_info.get("Version", "0"))
        except ValueError:
            old_version = 0.0
        need_conversion = old_version < BOOK_VERSION

    if not need_conversion:
        return old_data

    # --- User confirmation ---
    response = messagebox.askyesno(
        "Logbook Conversion Required",
        (
            f"This logbook was created with an older Book version: {old_version}.\n\n"
            f"It must be converted to Book version: {BOOK_VERSION}.\n"
            f"After conversion it will NOT work with older MiniBook versions.\n\n"
            f"Do you want to continue?"
        )
    )
    if not response:
        messagebox.showinfo(
            "Conversion Cancelled",
            "The logbook will not be loaded until it has been converted.\n\n"
            "No data has been changed."
        )
        return None

    print(f"DEBUG: Converting logbook (v{old_version} → {BOOK_VERSION})")

    # --- Load station info ---
    station = old_data.get("Station", {})
    if not isinstance(station, dict):
        station = {}

    my_callsign = station.get("Callsign", station_callsign)
    my_operator = station.get("Operator", my_callsign)
    my_locator = station.get("Locator", "")
    my_location = station.get("Location", "")

    # --- Get old logbook entries ---
    if isinstance(old_data, dict):
        old_log = old_data.get("Logbook", [])
    elif isinstance(old_data, list):
        old_log = old_data
    else:
        raise ValueError("Invalid logbook format")

    OPTIONAL_FIELDS = [
        "WWFF", "POTA", "BOTA", "COTA", "IOTA", "SOTA", "WLOTA",
        "Satellite",
        "My WWFF", "My POTA", "My BOTA", "My COTA",
        "My IOTA", "My SOTA", "My WLOTA"
    ]

    new_logbook = []

    # --- Convert each QSO ---
    for qso in old_log:

        # Normalize names (GUI → internal)
        qso = normalize_qso_fields(qso)

        callsign = qso.get("Callsign", "").strip().upper()

        # --- DXCC LOOKUP DURING CONVERSION ---
        # Determine DXCC entity using prefix_to_dxcc.json
        dxcc_entity = None
        for i in range(len(callsign), 0, -1):
            pf = callsign[:i]
            if pf in prefix_to_dxcc:
                dxcc_entity = prefix_to_dxcc[pf]
                break

        # Build converted QSO
        new_qso = {
            "Date":      qso.get("Date", ""),
            "Time":      qso.get("Time", ""),
            "Callsign":  callsign,
            "Name":      qso.get("Name", ""),
            "Country":   qso.get("Country", ""),
            "Continent": qso.get("Continent", ""),

            # --- DXCC entity stored here ---
            "dxcc": dxcc_entity if dxcc_entity else "",
            "CQZ":  qso.get("CQZ", ""),
            "ITUZ": qso.get("ITUZ", ""),

            "Sent":      qso.get("Sent", ""),
            "Received":  qso.get("Received", ""),
            "STX":       qso.get("STX", ""),
            "SRX":       qso.get("SRX", ""),
            "Mode":      qso.get("Mode", ""),
            "Submode":   qso.get("Submode", ""),
            "Band":      qso.get("Band", ""),
            "Frequency": qso.get("Frequency", ""),
            "Locator":   qso.get("Locator", ""),
            "Comment":   qso.get("Comment", ""),

            # Station fields
            "My Callsign": my_callsign,
            "My Operator": my_operator,
            "My Locator":  my_locator,
            "My Location": my_location,

            # QSL / LoTW fields
            "QS":   qso.get("QS", "No"),
            "QSD":  qso.get("QSD", ""),
            "QSV":  qso.get("QSV", ""),

            "QR":   qso.get("QR", "No"),
            "QRD":  qso.get("QRD", ""),
            "QRV":  qso.get("QRV", ""),

            "LWS":  qso.get("LWS", "No"),
            "LWSD": qso.get("LWSD", ""),
            "LWR":  qso.get("LWR", "No"),
            "LWRD": qso.get("LWRD", ""),
        }

        # Add optional fields if present
        for field in OPTIONAL_FIELDS:
            val = qso.get(field, "")
            if val and str(val).strip():
                new_qso[field] = val

        # Fix QSL/LoTW consistency
        for yes_key, date_key in (("QS", "QSD"), ("QR", "QRD"), ("LWS", "LWSD"), ("LWR", "LWRD")):
            if new_qso.get(yes_key) == "Yes" and not new_qso.get(date_key):
                new_qso[date_key] = datetime.now().strftime("%Y-%m-%d")
            elif new_qso.get(yes_key) == "No":
                new_qso[date_key] = ""

        new_logbook.append(new_qso)

    # Fill missing station fields
    defaults = {
        "Name": "", "Street": "", "Postalcode": "", "City": "", "County": "", "Country": "",
        "CQ Zone": "", "ITU Zone": "", "Contest": "",
        "WWFF": "", "POTA": "", "BOTA": "", "IOTA": "", "SOTA": "", "WLOTA": "", "COTA": "",
        "QRZAPI": "", "QRZUpload": False
    }
    for k, v in defaults.items():
        station.setdefault(k, v)

    # Notify user
    messagebox.showinfo(
        "Logbook Converted",
        (
            f"The logbook has been successfully converted to\nBook version: {BOOK_VERSION}.\n\n"
            f"Total QSOs converted: {len(new_logbook)}.\n\n"
            f"⚠️ Not compatible with MiniBook versions older than this release."
        )
    )

    return {
        "BookInfo": {"Version": str(BOOK_VERSION)},
        "Station": station,
        "Logbook": new_logbook
    }



# --- Integrate into load_json ---
def load_json(file_to_load=None):
    global CURRENT_JSON_FILE, Logbook_Window, qso_lines, duplicate_index_map, tree_to_log_index

    log.info("LOAD_JSON called")

    # ----------------------------------------------------------
    # 1) Ask for file FIRST. Do not reset anything yet.
    # ----------------------------------------------------------
    selected_file = file_to_load or filedialog.askopenfilename(
        filetypes=[("MiniBook files", "*.mbk")]
    )
    if not selected_file:
        log.info("LOAD_JSON aborted: no file selected")
        return  # cancel → abort

    log.info(f"Selected file: {selected_file}")

    # try to get file size
    try:
        size = os.path.getsize(selected_file)
        log.info(f"File size: {size} bytes")
    except Exception as e:
        log.warning(f"Could not determine file size: {e}")

    # ----------------------------------------------------------
    # 2) CLOSE THE WINDOW BEFORE ANY UI RESET
    # ----------------------------------------------------------
    if Logbook_Window is not None and Logbook_Window.winfo_exists():
        close_logbook()

    # now safe to reset everything
    qso_lines = []
    duplicate_index_map = {}
    tree_to_log_index = {}


    reset_fields()
    clear_station_labels()
    workedb4_tree.delete(*workedb4_tree.get_children())
    CURRENT_JSON_FILE = ""

    try:
        use_serial_var.set(False)
        qso_sent_exchange_var.set("1")
        toggle_serial_fields()
    except NameError:
        pass

    CURRENT_JSON_FILE = selected_file

    # After opening a logbook, update last_loaded_logbook in config
    try:
        if not config.has_section('General'):
            config.add_section('General')

        config.set('General', 'last_loaded_logbook', CURRENT_JSON_FILE)

        config_file_path = CONFIG_FILE if CONFIG_FILE.exists() else DATA_FOLDER / "minibook.ini"
        with open(config_file_path, 'w', encoding='utf-8') as cfgfile:
            config.write(cfgfile)

    except Exception as e:
        log.exception(f"Failed to update last_loaded_logbook in config: {e}")


    # ----------------------------------------------------------
    # Validate backup folder (unchanged)
    # ----------------------------------------------------------
    try:
        if CONFIG_FILE.exists():
            config_file_path = CONFIG_FILE
        else:
            config_file_path = DATA_FOLDER / "minibook.ini"

        backup_dir = config.get("General", "backup_folder", fallback=str(BACKUP_FOLDER)).strip()

        if not backup_dir or not os.path.isdir(backup_dir):
            log.warning("Backup folder invalid, using default backup folder")
            backup_dir = str(BACKUP_FOLDER)
            os.makedirs(backup_dir, exist_ok=True)

            show_auto_close_messagebox(
                "MiniBook",
                f"The configured backup folder could not be found.\n\n"
                f"MiniBook will now use the default folder:\n\n{backup_dir}",
                duration=4000
            )

            if not config.has_section('General'):
                config.add_section('General')

            try:
                config.set('General', 'backup_folder', backup_dir)
                with open(config_file_path, 'w', encoding='utf-8') as cfgfile:
                    config.write(cfgfile)
            except Exception as e:
                log.exception(f"Failed saving updated backup folder setting: {e}")
        else:
            os.makedirs(backup_dir, exist_ok=True)

    except Exception as e:
        log.exception(f"Backup folder initialization failed: {e}")
        backup_dir = str(BACKUP_FOLDER)
        os.makedirs(backup_dir, exist_ok=True)

    # ----------------------------------------------------------
    # Backup
    # ----------------------------------------------------------
    try:
        from datetime import datetime
        base_name = os.path.basename(CURRENT_JSON_FILE)
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        backup_path = os.path.join(backup_dir, f"{timestamp}_{base_name}")

        log.info(f"Creating backup: {backup_path}")

        with open(CURRENT_JSON_FILE, "r", encoding="utf-8") as f1, \
             open(backup_path, "w", encoding="utf-8") as f2:
            f2.write(f1.read())

        log.info("Backup OK")

    except Exception as e:
        log.exception(f"Backup failed: {e}")

    # ----------------------------------------------------------
    # LOAD JSON
    # ----------------------------------------------------------
    try:
        log.info("Reading JSON...")
        with open(CURRENT_JSON_FILE, 'r', encoding='utf-8') as file:
            data = json.load(file)

        log.info("JSON parsed successfully")

        qso_lines = data.get("Logbook", [])

        # ------------------------------------------------------
        # Convert if needed
        # ------------------------------------------------------
        converted_data = convert_log(data)

        if converted_data is None:
            log.error("Conversion failed: convert_log returned None")
            no_file_loaded()
            return

        if converted_data is not data:
            log.info("Converted logbook structure detected, rewriting file...")
            data = converted_data
            try:
                atomic_write_json(CURRENT_JSON_FILE, data)
                log.info("Converted log saved OK")
            except Exception as e:
                log.exception(f"Failed to save converted logbook: {e}")
                messagebox.showerror("Save Error", f"Failed to save converted logbook:\n{e}")
                no_file_loaded()
                return

            qso_lines = data.get("Logbook", [])

        # ------------------------------------------------------
        # Validate structure
        # ------------------------------------------------------
        if not (isinstance(data, dict) and "Station" in data and "Logbook" in data):
            log.error("Invalid logbook structure")
            messagebox.showerror("Invalid Format", "The file does not contain a valid logbook structure.")
            no_file_loaded()
            return

        log.info(f"Loaded {len(qso_lines)} QSO entries")

        update_title(root, VERSION_NUMBER, CURRENT_JSON_FILE, radio_status_var.get())
        load_station_setup()
        file_menu.entryconfig("Station setup", state="normal")
        
        # Load into TreeView
        load_json_content_to_tree()
        log.info("LOAD_JSON completed successfully")

    except Exception as e:
        log.exception(f"LOAD_JSON failed: {e}")
        traceback.print_exc()
        messagebox.showerror("Error", f"Could not read the file:\n{str(e)}")
        no_file_loaded()


def load_json_content_to_tree():
    """Load qso_lines into the TreeView using ALL fields defined in FIELD_LAYOUT."""
    global tree, qso_count_label, qso_lines, tree_to_log_index

    tree_to_log_index = {}

    if tree is None or not tree.winfo_exists():
        return

    # Clear TreeView
    try:
        tree.delete(*tree.get_children())
    except Exception as e:
        print(f"Error clearing TreeView: {e}")
        return

    if not qso_lines:
        if qso_count_label and qso_count_label.winfo_exists():
            qso_count_label.config(text="Total of 0 QSO's in logbook")
        return

    # ------------------------------------------------------------
    # Ensure internal QSL / LoTW fields exist (short internal form)
    # ------------------------------------------------------------
    for qso in qso_lines:
        defaults = {
            "QS": "No", "QSD": "", "QSV": "",
            "QR": "No", "QRD": "", "QRV": "",
            "LWS": "No", "LWSD": "",
            "LWR": "No", "LWRD": ""
        }
        for k, v in defaults.items():
            qso.setdefault(k, v)

    # ------------------------------------------------------------
    # DateTime field for sorting
    # ------------------------------------------------------------
    for qso in qso_lines:
        date_str = str(qso.get("Date", "")).strip()
        time_str = str(qso.get("Time", "")).strip()
        try:
            if date_str and time_str:
                qso["DateTime"] = datetime.strptime(
                    f"{date_str} {time_str}", "%Y-%m-%d %H:%M:%S"
                )
            elif date_str:
                qso["DateTime"] = datetime.strptime(date_str, "%Y-%m-%d")
            else:
                qso["DateTime"] = datetime.min
        except Exception:
            qso["DateTime"] = datetime.min

    # Newest first
    qso_lines.sort(key=lambda x: x.get("DateTime", datetime.min), reverse=True)

    # ------------------------------------------------------------
    # FIELD_ALIAS–aware value resolver (CENTRAL)
    # ------------------------------------------------------------
    def get_field_value(qso, gui_field):
        """
        Resolve GUI field name via FIELD_ALIAS to internal field name.
        Falls back to direct access if no alias exists.
        """
        internal = FIELD_ALIAS.get(gui_field, gui_field)
        return qso.get(internal, "")

    # ------------------------------------------------------------
    # Use *current visible TreeView columns*
    # ------------------------------------------------------------
    visible_columns = list(tree["columns"])

    qso_counter = 0

    for idx, qso in enumerate(qso_lines):
        tag = "oddrow" if idx % 2 else "evenrow"
        qso_counter += 1

        values = [
            get_field_value(qso, col)
            for col in visible_columns
        ]

        iid = f"qso_{idx}"
        tree.insert(
            "",
            "end",
            iid=iid,
            values=values,
            tags=(tag,)
        )

        tree_to_log_index[iid] = idx

    if qso_count_label and qso_count_label.winfo_exists():
        qso_count_label.config(text=f"Total of {qso_counter} QSO's in logbook")

    tree.tag_configure("oddrow", background="#f0f0f0")
    tree.tag_configure("evenrow", background="white")

    print(f"✅ Loaded {len(qso_lines)} QSO's using FIELD_ALIAS")

def open_column_selector():
    global tree

    if tree is None or not tree.winfo_exists():
        return

    REQUIRED = ("Date", "Time", "Callsign")

    DEFAULT_VISIBLE = [
        "Date", "Time", "Callsign", "Name", "Country",
        "Band", "Mode", "Frequency",
        "Sent", "Received",
        "QSL Sent", "QSL Received",
        "LoTW Sent", "LoTW Received",
    ]

    win = tk.Toplevel(Logbook_Window)
    win.title("Select visible columns")
    win.transient(Logbook_Window)
    win.grab_set()

    # -------------------------------------------------
    # Center selector on Logbook window
    # -------------------------------------------------
    Logbook_Window.update_idletasks()
    w, h = 680, 430
    x = Logbook_Window.winfo_rootx() + (Logbook_Window.winfo_width() // 2) - (w // 2)
    y = Logbook_Window.winfo_rooty() + (Logbook_Window.winfo_height() // 2) - (h // 2)
    win.geometry(f"{w}x{h}+{x}+{y}")

    # -------------------------------------------------
    # Layout
    # -------------------------------------------------
    main = tk.Frame(win, padx=10, pady=10)
    main.pack(fill="both", expand=True)

    left_frame = tk.Frame(main)
    left_frame.pack(side="left", fill="both", expand=True)

    mid_frame = tk.Frame(main)
    mid_frame.pack(side="left", padx=10)

    right_frame = tk.Frame(main)
    right_frame.pack(side="left", fill="both", expand=True)

    tk.Label(left_frame, text="Available columns").pack()
    tk.Label(right_frame, text="Visible columns").pack()

    lb_available = tk.Listbox(left_frame, selectmode="extended")
    lb_available.pack(fill="both", expand=True)

    lb_visible = tk.Listbox(right_frame, selectmode="extended")
    lb_visible.pack(fill="both", expand=True)

    # -------------------------------------------------
    # Initial content
    # -------------------------------------------------
    current_visible = list(tree["columns"])
    all_fields = list(FIELD_LAYOUT.keys())

    visible = current_visible[:]
    available = sorted(f for f in all_fields if f not in visible)

    for f in available:
        lb_available.insert("end", f)
    for f in visible:
        lb_visible.insert("end", f)

    # -------------------------------------------------
    # Helpers
    # -------------------------------------------------
    def sort_available():
        items = sorted(lb_available.get(0, "end"))
        lb_available.delete(0, "end")
        for v in items:
            lb_available.insert("end", v)

    def selected_values(lb):
        return [lb.get(i) for i in lb.curselection()]

    def move_selected(src, dst, sort_dst=False):
        values = selected_values(src)
        if not values:
            return

        for v in values:
            if src is lb_visible and v in REQUIRED:
                continue
            idxs = list(src.curselection())
            for i in reversed(idxs):
                if src.get(i) == v:
                    src.delete(i)

            dst.insert("end", v)

        if sort_dst:
            sort_available()

    def move_all(src, dst, sort_dst=False):
        values = list(src.get(0, "end"))
        src.delete(0, "end")

        for v in values:
            if src is lb_visible and v in REQUIRED:
                lb_visible.insert("end", v)
            else:
                dst.insert("end", v)

        if sort_dst:
            sort_available()

    def move_up():
        sel = lb_visible.curselection()
        for i in sel:
            if i == 0:
                continue
            v = lb_visible.get(i)
            if v in REQUIRED:
                continue
            above = lb_visible.get(i - 1)
            if above in REQUIRED:
                continue
            lb_visible.delete(i)
            lb_visible.insert(i - 1, v)
            lb_visible.selection_set(i - 1)

    def move_down():
        sel = list(lb_visible.curselection())
        size = lb_visible.size()
        for i in reversed(sel):
            if i >= size - 1:
                continue
            v = lb_visible.get(i)
            if v in REQUIRED:
                continue
            below = lb_visible.get(i + 1)
            if below in REQUIRED:
                continue
            lb_visible.delete(i)
            lb_visible.insert(i + 1, v)
            lb_visible.selection_set(i + 1)

    # -------------------------------------------------
    # Buttons (middle)
    # -------------------------------------------------
    tk.Button(mid_frame, text=">", width=4,
              command=lambda: move_selected(lb_available, lb_visible)).pack(pady=2)

    tk.Button(mid_frame, text="<", width=4,
              command=lambda: move_selected(lb_visible, lb_available, sort_dst=True)).pack(pady=2)

    tk.Button(mid_frame, text=">>", width=4,
              command=lambda: move_all(lb_available, lb_visible)).pack(pady=10)

    tk.Button(mid_frame, text="<<", width=4,
              command=lambda: move_all(lb_visible, lb_available, sort_dst=True)).pack(pady=2)

    tk.Button(mid_frame, text="↑", width=4, command=move_up).pack(pady=10)
    tk.Button(mid_frame, text="↓", width=4, command=move_down).pack(pady=2)

    # -------------------------------------------------
    # Bottom buttons
    # -------------------------------------------------
    bottom = tk.Frame(win)
    bottom.pack(fill="x", pady=10)

    def reset_defaults():
        lb_available.delete(0, "end")
        lb_visible.delete(0, "end")

        for f in sorted(f for f in FIELD_LAYOUT if f not in DEFAULT_VISIBLE):
            lb_available.insert("end", f)

        for f in DEFAULT_VISIBLE:
            if f in FIELD_LAYOUT:
                lb_visible.insert("end", f)

    def apply_columns():
        new_cols = list(lb_visible.get(0, "end"))

        # Enforce required columns
        for r in REQUIRED:
            if r not in new_cols and r in FIELD_LAYOUT:
                new_cols.insert(0, r)

        apply_tree_columns(new_cols)

        if not config.has_section("LogbookColumns"):
            config.add_section("LogbookColumns")

        config.set("LogbookColumns", "visible", ",".join(new_cols))
        cfg_path = CONFIG_FILE if CONFIG_FILE.exists() else DATA_FOLDER / "minibook.ini"
        with open(cfg_path, "w", encoding="utf-8") as f:
            config.write(f)

        win.destroy()

    tk.Button(bottom, text="Reset defaults", width=14, command=reset_defaults).pack(side="left", padx=5)
    tk.Button(bottom, text="Apply", width=10, command=apply_columns).pack(side="right", padx=5)
    tk.Button(bottom, text="Cancel", width=10, command=win.destroy).pack(side="right")



def apply_tree_columns(col_list):
    """
    Apply a column list to the logbook TreeView.
    Rebuild headings, widths and reload content.
    """
    global tree, shared_column_widths

    if tree is None or not tree.winfo_exists():
        return

    tree.config(columns=col_list)

    for col in col_list:
        # --- determine column width safely ---
        if isinstance(shared_column_widths, dict):
            width = shared_column_widths.get(col, 100)
            minwidth = shared_column_widths.get(col, 60)
        else:
            # shared_column_widths is a list or something else
            width = 100
            minwidth = 60

        tree.heading(
            col,
            text=col,
            anchor="center",
            command=lambda c=col: sort_treeview(c)
        )
        tree.column(
            col,
            anchor="center",
            width=width,
            minwidth=minwidth,
            stretch=True
        )

    load_json_content_to_tree()


# ------------------------------------------------------------------
# Function to handle column header clicks for sorting (dynamic columns)
# ------------------------------------------------------------------
def sort_treeview(column):
    global sort_column, sort_reverse, tree

    if tree is None or not tree.winfo_exists():
        return

    cols = list(tree["columns"])
    if column not in cols:
        return

    # Toggle sort direction
    if sort_column == column:
        sort_reverse = not sort_reverse
    else:
        sort_column = column
        sort_reverse = False

    col_index = cols.index(column)

    def parse_float_freq(v):
        """
        Convert frequency strings to a numeric value in MHz when possible.
        Accepts '14.080', '14080', '14080000', etc.
        """
        s = str(v).strip().replace(",", ".")
        if not s:
            return float("-inf")

        # Keep only digits and one dot
        cleaned = []
        dot_used = False
        for ch in s:
            if ch.isdigit():
                cleaned.append(ch)
            elif ch == "." and not dot_used:
                cleaned.append(ch)
                dot_used = True
        s = "".join(cleaned)
        if not s or s == ".":
            return float("-inf")

        try:
            x = float(s)
        except Exception:
            return float("-inf")

        # Heuristic scaling:
        # - If it has a dot -> assume MHz already (14.080)
        # - If it's large -> assume Hz (14080000) or kHz (14080)
        if "." in s:
            return x  # MHz
        if x >= 1_000_000:
            return x / 1_000_000.0  # Hz -> MHz
        if x >= 1_000:
            return x / 1_000.0      # kHz -> MHz
        return x  # already MHz-ish

    def parse_int(v):
        s = str(v).strip()
        if not s:
            return -10**12
        digits = "".join(ch for ch in s if ch.isdigit())
        if not digits:
            return -10**12
        try:
            return int(digits)
        except Exception:
            return -10**12

    def parse_datetime_for_row(item_id):
        """
        Build a datetime from the row's Date and Time columns if present.
        Falls back safely.
        """
        try:
            date_val = ""
            time_val = ""

            if "Date" in cols:
                date_val = str(tree.item(item_id, "values")[cols.index("Date")]).strip()
            if "Time" in cols:
                time_val = str(tree.item(item_id, "values")[cols.index("Time")]).strip()

            if date_val and time_val:
                return datetime.strptime(f"{date_val} {time_val}", "%Y-%m-%d %H:%M:%S")
            if date_val:
                return datetime.strptime(date_val, "%Y-%m-%d")
        except Exception:
            pass
        return datetime.min

    def sort_key(item_id):
        v = tree.item(item_id, "values")[col_index]

        # Column-specific typing
        if column == "Frequency":
            return parse_float_freq(v)

        if column in ("Sent Exchange", "Receive Exchange", "DXCC"):
            return parse_int(v)

        if column in ("Date", "Time"):
            # Sort Date or Time based on combined DateTime when possible
            return parse_datetime_for_row(item_id)

        # Default: case-insensitive text sort
        return str(v).strip().lower()

    try:
        items = list(tree.get_children())
        items_sorted = sorted(items, key=sort_key, reverse=sort_reverse)
    except Exception as e:
        print(f"Sort error on column '{column}': {e}")
        return

    for i, item_id in enumerate(items_sorted):
        tree.move(item_id, "", i)
        tree.item(item_id, tags=("oddrow" if i % 2 == 0 else "evenrow"))

    tree.tag_configure("oddrow", background="#f0f0f0")
    tree.tag_configure("evenrow", background="white")



#########################################################################################
#  _    ___   ___ ___  ___   ___  _  __ __      _____ _  _ ___   _____      __
# | |  / _ \ / __| _ )/ _ \ / _ \| |/ / \ \    / /_ _| \| |   \ / _ \ \    / /
# | |_| (_) | (_ | _ \ (_) | (_) | ' <   \ \/\/ / | || .` | |) | (_) \ \/\/ / 
# |____\___/ \___|___/\___/ \___/|_|\_\   \_/\_/ |___|_|\_|___/ \___/ \_/\_/  
#                                                                             
#########################################################################################

# Global variables for Logbook Viewer
qso_lines = []  # This will hold QSO entries
sort_column = None  # Column currently being sorted
sort_reverse = False  # Flag for sort order

def close_logbook():
    global Logbook_Window, tree
    if Logbook_Window is None:
        return

    if Logbook_Window.winfo_exists():
        save_window_geometry(Logbook_Window, "LogbookWindow")

    try:
        Logbook_Window.destroy()
    except:
        pass

    Logbook_Window = None
    tree = None

# Function to open and display the logbook in a new window
def view_logbook():
    global tree, qso_count_label, search_entry, qso_lines, column_checkboxes, Logbook_Window

    if not CURRENT_JSON_FILE:
        messagebox.showwarning("Warning", "Please first load logbook!")
        return

       # Check if the logbook window is already open
    if Logbook_Window is not None and Logbook_Window.winfo_exists():
        Logbook_Window.lift()  # Bring the existing window to the front
        return

    # Create a new window to display the logbook
    Logbook_Window = tk.Toplevel(root)
    Logbook_Window.title(f"MiniBook Logbook - " + os.path.basename(CURRENT_JSON_FILE))

    info_frame = tk.Frame(Logbook_Window)
    info_frame.pack(fill="x", padx=10, pady=2)

    # Label to show the number of QSO's logged
    qso_count_label = tk.Label(info_frame, text="Total QSO's: 0", font=('Arial', 14))
    qso_count_label.grid(pady=2)

    separator = ttk.Separator(Logbook_Window, orient='horizontal')
    separator.pack(fill='x', pady=2)

    # Adding Menu to logbook window
    menu_bar = tk.Menu(Logbook_Window)
    file_menu = tk.Menu(menu_bar, tearoff=0)
    file_menu.add_command(label="Import ADIF", command=import_adif)
    file_menu.add_command(label="Export to ADIF", command=export_to_adif)
    file_menu.add_separator()
    file_menu.add_command(label="Exit", command=close_logbook)
    menu_bar.add_cascade(label="File", menu=file_menu)
    Logbook_Window.config(menu=menu_bar)

    style = ttk.Style(Logbook_Window)

    bg = "SystemButtonFace"
    labelframe_font = ('Arial', 10, 'bold')

    style.configure("Grooved.TLabelframe", relief="groove", borderwidth=2, background=bg)
    style.configure("Grooved.TLabelframe.Label", background=bg, foreground=style.lookup("TLabel", "foreground"))

    # =========================================================
    # Search options (entry + buttons)
    # =========================================================
    search_labelframe = tk.LabelFrame(Logbook_Window, text="Search options", font=labelframe_font, relief="groove", bd=2, bg=bg)
    search_labelframe.pack(fill="x", padx=10, pady=(5, 2))

    search_frame = tk.Frame(search_labelframe, bg=bg)
    search_frame.pack(fill="x", padx=5, pady=5)

    search_label = tk.Label(search_frame, text="Search in log:", font=('Arial', 10), bg=bg)
    search_label.pack(side='left', padx=(0, 5))

    search_entry = tk.Entry(search_frame, font=('Arial', 10))
    search_entry.pack(side='left', padx=(0, 10))

    # =========================================================
    # Select Search Filter (ONLY checkboxes, dynamic layout)
    # =========================================================
    filter_labelframe = tk.LabelFrame(Logbook_Window, text="Select Fields to search in", font=labelframe_font, relief="groove", bd=2, bg=bg)
    filter_labelframe.pack(fill="x", padx=10, pady=(2, 5))

    checkbox_container = tk.Frame(filter_labelframe, bg=bg)
    checkbox_container.pack(fill="x", expand=True, padx=5, pady=5)

    # ------------------------------------------------------------------
    # Define columns for the logbook (USE ALL AVAILABLE FIELDS)
    # ------------------------------------------------------------------
    columns = tuple(FIELD_LAYOUT.keys())

    column_checkboxes = {}
    checkbox_widgets = []

    # Scrollbars for the Treeview
    tree_frame = tk.Frame(Logbook_Window)
    tree_frame.pack(fill='both', expand=True)

    # Scrollbars
    x_scrollbar = tk.Scrollbar(tree_frame, orient='horizontal')
    x_scrollbar.pack(side='bottom', fill='x')

    y_scrollbar = tk.Scrollbar(tree_frame, orient='vertical')
    y_scrollbar.pack(side='right', fill='y')

    # Create a style and configure the Treeview heading font
    style = ttk.Style()
    style.configure("Treeview.Heading", font=("Arial", 10, "bold"))

    # ------------------------------------------------------------------
    # Create the Treeview (initially with a default subset visible)
    # ------------------------------------------------------------------
    default_visible = {"Callsign"} # Default checkboxes checked
    '''
    default_visible = {
        "Date", "Time", "Callsign", "Name", "Country",
        "Band", "Mode", "Frequency",
        "Sent", "Received",
        "QSL Sent", "QSL Received",
        "LoTW Sent", "LoTW Received",
    }
    '''

    # Pre-create checkbox vars (search depends on them)
    for col in columns:
        var = tk.BooleanVar()
        column_checkboxes[col] = var

    # Default SEARCH fields (can be different from visible columns)
    # Keep sensible defaults, but do NOT use these to show/hide TreeView columns.
    for col in columns:
        column_checkboxes[col].set(col in default_visible)
    if "Callsign" in column_checkboxes:
        column_checkboxes["Callsign"].set(True)

    # Initial TreeView columns are just a placeholder; we will apply saved columns right after creation.
    visible_initial = [c for c in columns if c in default_visible]
    if not visible_initial:
        visible_initial = ["Callsign"]

    tree = ttk.Treeview(
        tree_frame,
        columns=visible_initial,
        show='headings',
        selectmode='extended',
        xscrollcommand=x_scrollbar.set,
        yscrollcommand=y_scrollbar.set
    )
    tree.pack(fill='both', expand=True)


    # -------------------------------------------------
    # Restore visible columns from config (if present)
    # -------------------------------------------------
    saved_columns = None

    if config.has_section("LogbookColumns"):
        saved = config.get("LogbookColumns", "visible", fallback="").strip()
        if saved:
            saved_columns = [c.strip() for c in saved.split(",") if c.strip() in FIELD_LAYOUT]

    # Always require these for matching logic (delete/QRZ/Quick QSL)
    required_cols = ["Date", "Time", "Callsign"]
    if saved_columns:
        for r in required_cols:
            if r in FIELD_LAYOUT and r not in saved_columns:
                saved_columns.insert(0, r)
        visible_cols_to_apply = saved_columns
    else:
        visible_cols_to_apply = visible_initial
        for r in required_cols:
            if r in FIELD_LAYOUT and r not in visible_cols_to_apply:
                visible_cols_to_apply.insert(0, r)

    apply_tree_columns(visible_cols_to_apply)

    # Build checkbox widgets (search filters only)
    for col in columns:
        var = column_checkboxes[col]
        cb = tk.Checkbutton(
            checkbox_container,
            text=col,
            variable=var,
            anchor="w",
            bg=bg,
            activebackground=bg,
            highlightthickness=0
        )
        checkbox_widgets.append(cb)

    # --------- Dynamic layout on resize ----------
    def layout_checkboxes(event=None):
        width = checkbox_container.winfo_width()
        if width <= 1:
            return

        checkbox_width = 135
        cols = max(1, width // checkbox_width)

        for i, cb in enumerate(checkbox_widgets):
            cb.grid(row=i // cols, column=i % cols, sticky="w", padx=5, pady=2)

    checkbox_container.bind("<Configure>", layout_checkboxes)

    # Right-click context menu for editing QSO
    def show_context_menu(event):
        item = tree.identify_row(event.y)
        if item:
            # Add to selection if not already selected
            if item not in tree.selection():
                tree.selection_add(item)

            selected_items = tree.selection()

            # Only show "Edit QSO" if exactly 1 line is selected
            if len(selected_items) == 1:
                context_menu.entryconfig("Edit QSO", state="normal")
            else:
                context_menu.entryconfig("Edit QSO", state="disabled")

            # Only show "Bulk Edit" when 2 or more lines are selected
            if len(selected_items) >= 2:
                context_menu.entryconfig("Bulk Edit", state="normal")
            else:
                context_menu.entryconfig("Bulk Edit", state="disabled")

            context_menu.post(event.x_root, event.y_root)

#    _  _ ___    _                 _     _  _  _
#   / \|_) _/   |_)| ||  |/    | ||_)|  / \|_|| \
#   \_X| \/__   |_)|_||__|\    |_||  |__\_/| ||_/

    def qrz_bulk_log(msg):
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        with open(QRZ_UPLOAD_BULK_LOG_FILE, "a", encoding="utf-8") as f:
            f.write(f"{timestamp}  {msg}\n")

    def upload_qsos_to_qrz():
        selected_items = tree.selection()
        if not selected_items:
            messagebox.showinfo("Info", "Select one or more QSOs to upload to QRZ.", parent=Logbook_Window)
            return

        confirm = messagebox.askyesno(
            "Upload to QRZ",
            f"Upload {len(selected_items)} selected QSO(s) to QRZ?",
            parent=Logbook_Window
        )
        if not confirm:
            return

        # --- bulk log reset ---
        with open(QRZ_UPLOAD_BULK_LOG_FILE, "w", encoding="utf-8"):
            pass

        qrz_bulk_log(f"=== BULK UPLOAD START ({len(selected_items)} QSOs) ===")

        progress_win = tk.Toplevel(Logbook_Window)
        progress_win.title("Uploading to QRZ")

        # Center window
        logbook_x = Logbook_Window.winfo_rootx()
        logbook_y = Logbook_Window.winfo_rooty()
        center_x = logbook_x + Logbook_Window.winfo_width() // 2
        center_y = logbook_y + Logbook_Window.winfo_height() // 2
        progress_win.geometry(f"+{center_x - 100}+{center_y - 50}")

        progress_label = tk.Label(progress_win, text="Uploading QSOs...")
        progress_label.pack(pady=5)

        progress_bar = ttk.Progressbar(progress_win, length=250, mode='determinate', maximum=len(selected_items))
        progress_bar.pack(pady=5)

        progress_counter = tk.Label(progress_win, text=f"0 / {len(selected_items)} QSOs geüpload")
        progress_counter.pack(pady=5)

        progress_win.update()

        result_counts = {
            "✅ Uploaded successfully": 0,
            "⚠️ Duplicate QSO": 0,
            "❌ Wrong station_callsign": 0,
            "❌ Other errors": 0,
            "🔍 Not found in memory (skipped)": 0
        }

        def upload_thread():
            for idx, item in enumerate(selected_items, start=1):
                values = tree.item(item)['values']

                # IMPORTANT: selection matching assumes Date/Time/Callsign are visible (kept required)
                matched_qso = next((qso for qso in qso_lines if
                                    qso.get("Date") == values[0] and
                                    qso.get("Time") == values[1] and
                                    qso.get("Callsign") == values[2]), None)

                if matched_qso:
                    response = upload_to_qrz(matched_qso, True)

                    callsign = matched_qso.get('Callsign')

                    if response and hasattr(response, "text"):
                        text = response.text.strip()

                        if "RESULT=OK" in text:
                            result_counts["✅ Uploaded successfully"] += 1
                            qrz_bulk_log(f"OK: {callsign}")

                        elif "RESULT=FAIL" in text:
                            reason = ""
                            if "REASON=" in text:
                                reason = text.split("REASON=")[-1].split("&")[0].strip().lower()

                            if "duplicate" in reason:
                                result_counts["⚠️ Duplicate QSO"] += 1
                                qrz_bulk_log(f"DUPLICATE: {callsign}")

                            elif "wrong station_callsign" in reason:
                                result_counts["❌ Wrong station_callsign"] += 1
                                qrz_bulk_log(f"WRONG_STATION_CALLSIGN: {callsign}")

                            else:
                                result_counts["❌ Other errors"] += 1
                                qrz_bulk_log(f"ERROR: {callsign} ({reason})")

                        elif "RESULT=AUTH" in text:
                            result_counts["❌ Other errors"] += 1
                            qrz_bulk_log(f"INVALID API KEY: {callsign}")
                        else:
                            result_counts["❌ Other errors"] += 1
                            qrz_bulk_log(f"UNKNOWN RESPONSE: {callsign}")

                    else:
                        result_counts["❌ Other errors"] += 1
                        qrz_bulk_log(f"INVALID RESPONSE: {callsign}")

                else:
                    result_counts["🔍 Not found in memory (skipped)"] += 1
                    try:
                        qrz_bulk_log(f"SKIPPED (NOT FOUND): {values[2]}")
                    except Exception:
                        qrz_bulk_log("SKIPPED (NOT FOUND): <unknown>")

                # update progress
                progress_bar['value'] = idx
                progress_counter.config(text=f"{idx} / {len(selected_items)} QSOs uploaded")
                progress_win.update()

            progress_win.destroy()

            summary_lines = [f"{k}: {v}" for k, v in result_counts.items() if v > 0]
            summary = "\n".join(summary_lines)

            messagebox.showinfo("QRZ Upload Result", summary)

            timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            qrz_bulk_log(f"=== BULK UPLOAD DONE @ {timestamp} ===")
            qrz_bulk_log(summary)

        threading.Thread(target=upload_thread, daemon=True).start()

#    _  __    _____ __    _  __ _  /  __
#   | \|_ |  |_  | |_    / \(_ / \   (_
#   |_/|__|__|__ | |__   \_X__)\_/   __)

    # Function to delete QSO from the menu
    def delete_qso_from_menu():
        global qso_lines

        selected_items = tree.selection()
        if not selected_items:
            return

        if not messagebox.askyesno(
            "Delete QSO(s)",
            f"Are you sure you want to delete {len(selected_items)} selected QSO(s)?",
            parent=Logbook_Window
        ):
            return

        # --------------------------------------------------------
        # ONE SINGLE WINDOW (appears immediately)
        # --------------------------------------------------------
        progress_win = tk.Toplevel(Logbook_Window)
        progress_win.title("Deleting QSOs…")
        progress_win.resizable(False, False)
        progress_win.transient(Logbook_Window)

        x = Logbook_Window.winfo_rootx() + Logbook_Window.winfo_width() // 2
        y = Logbook_Window.winfo_rooty() + Logbook_Window.winfo_height() // 2
        progress_win.geometry(f"+{x-150}+{y-70}")

        # This label NEVER changes anymore
        tk.Label(
            progress_win,
            text="Preparing deletion. This may take a moment...",
            font=("Segoe UI", 10)
        ).pack(pady=(10, 5))

        progress_bar = ttk.Progressbar(progress_win, length=280, mode="determinate")
        progress_bar.pack(pady=5)

        progress_txt = tk.Label(progress_win, text="")
        progress_txt.pack(pady=(0, 10))

        progress_win.update_idletasks()

        # --------------------------------------------------------
        # PHASE 1: Collect identifiers
        # --------------------------------------------------------
        def worker_prepare():
            nonlocal selected_items

            identifiers = []
            for iid in selected_items:
                values = tree.item(iid)['values']
                identifiers.append((
                    str(values[0]).strip(),
                    str(values[1]).strip(),
                    str(values[2]).strip().upper()
                ))

            root.after(0, lambda: do_delete(identifiers))

        threading.Thread(target=worker_prepare, daemon=True).start()

        # --------------------------------------------------------
        # PHASE 2: Deletion + progress
        # --------------------------------------------------------
        def do_delete(identifiers):
            total = len(identifiers)
            progress_bar["maximum"] = total

            def worker_delete():
                new_data = [
                    qso for qso in qso_lines
                    if (
                        str(qso.get("Date", "")).strip(),
                        str(qso.get("Time", "")).strip(),
                        str(qso.get("Callsign", "")).strip().upper()
                    ) not in identifiers
                ]

                # Progress loop
                for i in range(total):
                    progress_bar["value"] = i + 1
                    progress_txt.config(text=f"{i+1} / {total}")
                    progress_win.update()

                qso_lines[:] = new_data
                save_async()

                progress_win.destroy()

                def finish():
                    load_json_content_to_tree()
                    update_worked_before_tree()
                    messagebox.showinfo(
                        "Delete Complete",
                        f"{total} QSO(s) deleted successfully.",
                        parent=Logbook_Window
                    )

                root.after(0, finish)

            threading.Thread(target=worker_delete, daemon=True).start()

    def quick_qsl_update(mode, via):
        """
        mode: 'sent' or 'recv'
        via: 'Bureau', 'Direct', 'Electronic', 'Manager'
        """

        selected = tree.selection()
        if not selected:
            return

        now = datetime.now().strftime("%Y-%m-%d")

        # Intern veld mapping
        if mode == "sent":
            flag = "QS"
            datefield = "QSD"
            viafield = "QSV"
        else:
            flag = "QR"
            datefield = "QRD"
            viafield = "QRV"

        updated = 0

        for item in selected:
            qso_index = tree_to_log_index.get(item)
            if qso_index is None:
                continue

            qso = qso_lines[qso_index]

            qso[flag] = "Yes"
            qso[datefield] = now
            qso[viafield] = via
            updated += 1

        if updated:
            save_async()
            load_json_content_to_tree()
            update_worked_before_tree()


    # Create a context menu
    context_menu = tk.Menu(Logbook_Window, tearoff=0)

    context_menu.add_command(label="Edit QSO", command=edit_qso)
    context_menu.add_command(label="Bulk Edit", command=open_bulk_edit_window)
    context_menu.add_separator()

    menu_sent = tk.Menu(context_menu, tearoff=0)
    menu_recv = tk.Menu(context_menu, tearoff=0)

    for opt in ("Bureau", "Direct", "Electronic", "Manager"):
        menu_sent.add_command(label=opt, command=lambda o=opt: quick_qsl_update("sent", o))
        menu_recv.add_command(label=opt, command=lambda o=opt: quick_qsl_update("recv", o))

    context_menu.add_cascade(label="Quick Confirm Received", menu=menu_recv)
    context_menu.add_cascade(label="Quick Confirm Sent", menu=menu_sent)
    context_menu.add_separator()
    context_menu.add_command(label="Update DXCC Information", command=update_dxcc_selected)
    context_menu.add_command(label="Update Name with QRZ Lookup", command=update_name_from_qrz)
    context_menu.add_separator()
    context_menu.add_command(label="Upload to QRZ", command=upload_qsos_to_qrz)
    context_menu.add_command(label="Export to ADIF (selected)", command=export_selected_to_adif)
    context_menu.add_separator()
    context_menu.add_command(label="Delete QSO('s)", command=delete_qso_from_menu)

    # Bind the right-click event to show the context menu
    tree.bind("<Button-2>" if platform.system() == "Darwin" else "<Button-3>", show_context_menu)

    # Bind double-left click event to show edit window
    tree.bind("<Double-1>", edit_qso)

    if tree is None:
        print("Treeview is not initialized immediately after creation.")

    x_scrollbar.config(command=tree.xview)
    y_scrollbar.config(command=tree.yview)

    # Initial load of the JSON content
    load_json_content_to_tree()

#       _  __    __ __ _  _  __
#   |  / \/__   (_ |_ |_||_)/  |_|
#   |__\_/\_|   __)|__| || \\__| |

    def search_log():
        global tree_to_log_index

        search_term = search_entry.get().lower().strip()
        matching_entries = []

        # Clear old tree + mapping
        tree.delete(*tree.get_children())
        tree_to_log_index = {}

        if not search_term:
            qso_count_label.config(text="Total QSO's: 0")
            return

        # Helper to get correct search value (handles internal QSL / LoTW fields)
        def get_search_value(qso, field):
            """
            Resolve TreeView column name → internal QSO field using FIELD_ALIAS
            """
            # GUI → internal field mapping (e.g. "Receive Exchange" → "SRX")
            internal = FIELD_ALIAS.get(field, field)
            return qso.get(internal, "")


        # Search through original qso_lines
        for original_index, qso in enumerate(qso_lines):
            for col, var in column_checkboxes.items():
                if not var.get():
                    continue

                value = get_search_value(qso, col)
                if value and search_term in str(value).lower():
                    matching_entries.append((original_index, qso))
                    break

        # Show results
        if matching_entries:
            row_color = True
            visible_cols = list(tree["columns"])

            for original_index, qso in matching_entries:
                iid = f"search_{original_index}"

                # Build row values matching current visible columns
                values = []
                for c in visible_cols:
                    values.append(get_search_value(qso, c))

                tree.insert(
                    "",
                    "end",
                    iid=iid,
                    values=tuple(values),
                    tags=('oddrow' if row_color else 'evenrow',)
                )

                # Link TreeView IID back to original line for EDIT / BULK EDIT
                tree_to_log_index[iid] = original_index
                row_color = not row_color

            qso_count_label.config(text=f"Total QSO's: {len(matching_entries)}")
        else:
            qso_count_label.config(text="Total QSO's: 0")


    # Search button
    search_button = tk.Button(search_frame, text="Search", command=search_log)
    search_button.pack(pady=5, side='left', padx=(0,20))

    find_duplicates_btn = tk.Button(search_frame,  text="Find Duplicates", command=find_duplicates)
    find_duplicates_btn.pack(side='left', padx=20)

    # Function to reset the log to show all entries
    def reset_view():
        load_json_content_to_tree()  # Load all entries again
        search_entry.delete(0, tk.END)  # Clear the search entry

    # Reset button
    reset_button = tk.Button(search_frame, text="Reset View", command=reset_view)
    reset_button.pack(pady=5, padx=20, side='left')

    # --- Statistics Button ---
    stats_button = tk.Button(search_frame, text="Statistics", command=open_statistics_window)
    stats_button.pack(side='left', padx=30)

    columns_button = tk.Button(
        search_frame,
        text="Customize Layout",
        command=open_column_selector
    )
    columns_button.pack(side='left', padx=30)

    # Add a button to close the logbook window
    tk.Button(search_frame, text="Close Window", command=close_logbook).pack(pady=5, padx=30, side="right")

    # Function to update the logbook when new QSO is logged
    def update_logbook():
        load_json_content_to_tree()

    # Save a reference to the update function so we can call it later when logging a new QSO
    Logbook_Window.update_logbook = update_logbook

    # Load saved geometry
    load_window_geometry(Logbook_Window, "LogbookWindow")

    # Update config AFTER window geometry has been applied
    if not config.has_section('General'):
        config.add_section('General')

    config.set('General', 'last_loaded_logbook', CURRENT_JSON_FILE)

    config_file_path = CONFIG_FILE if CONFIG_FILE.exists() else DATA_FOLDER / "minibook.ini"
    with open(config_file_path, 'w', encoding='utf-8') as cfgfile:
        config.write(cfgfile)

    # Set the Logbook_Window to None when it is closed
    Logbook_Window.protocol("WM_DELETE_WINDOW", close_logbook)









def load_flag_images():
    global FLAG_IMAGES

    flags_dir = DATA_FOLDER / "flags"
    fallback = flags_dir / "0.png"

    for file in os.listdir(flags_dir):
        if not file.lower().endswith(".png"):
            continue

        name = os.path.splitext(file)[0]
        try:
            dxcc_num = int(name)
        except:
            continue

        FLAG_IMAGES[dxcc_num] = tk.PhotoImage(file=str(flags_dir / file))

    # fallback
    if fallback.exists():
        FLAG_IMAGES[-1] = tk.PhotoImage(file=str(fallback))
    else:
        FLAG_IMAGES[-1] = None




class CanvasTable(tk.Frame):

    FLAG_SIZE = 24
    FLAG_MODE = "fit"

    def __init__(self, parent, columns, on_header_click=None):
        super().__init__(parent)

        self.columns = columns
        self.row_height = 26
        self.header_height = 28
        self.on_header_click = on_header_click

        # Per-column widths
        self.col_widths = []
        for col in columns:
            if col == "Flag":
                width = 60
            elif col == "Entity":
                width = 200
            elif col == "DXCC":
                width = 60
            else:
                width = 60
            self.col_widths.append(width)

        # ================================
        # NEW: SEPARATE HEADER CANVAS
        # ================================
        self.header_canvas = tk.Canvas(
            self,
            height=self.header_height,
            bg="white",
            highlightthickness=0
        )
        self.header_canvas.grid(row=0, column=0, sticky="ew")

        # ================================
        # DATA CANVAS (scrollable)
        # ================================
        self.canvas = tk.Canvas(self, bg="white", highlightthickness=0)
        self.v_scroll = ttk.Scrollbar(self, orient="vertical", command=self.canvas.yview)
        self.h_scroll = ttk.Scrollbar(self, orient="horizontal", command=self.canvas.xview)

        self.canvas.configure(
            yscrollcommand=self.v_scroll.set,
            xscrollcommand=self.h_scroll.set
        )

        self.canvas.grid(row=1, column=0, sticky="nsew")
        self.v_scroll.grid(row=1, column=1, sticky="ns")
        self.h_scroll.grid(row=2, column=0, sticky="ew")

        self.grid_rowconfigure(1, weight=1)
        self.grid_columnconfigure(0, weight=1)

        self.rows = []
        self.header_ids = []
        self.sort_state = {}

        self.draw_header()

        # Mouse wheel binding
        self.canvas.bind("<MouseWheel>", self._on_mousewheel)
        self.canvas.bind("<Button-4>", self._on_mousewheel_linux)
        self.canvas.bind("<Button-5>", self._on_mousewheel_linux)

        # sync header scroll with table scroll
        self.canvas.bind("<Configure>", self._sync_header)

    # ---------------------------------
    # FIXED HEADER DRAW
    # ---------------------------------
    def draw_header(self):
        self.header_canvas.delete("all")
        self.header_ids.clear()

        for col_i, col_name in enumerate(self.columns):

            x0 = sum(self.col_widths[:col_i])
            width = self.col_widths[col_i]
            x1 = x0 + width

            rect_id = self.header_canvas.create_rectangle(
                x0, 0, x1, self.header_height,
                fill="#e8e8e8", outline="black"
            )

            label = col_name
            state = self.sort_state.get(col_name)
            if state == "asc":
                label += " ▲"
            elif state == "desc":
                label += " ▼"

            text_id = self.header_canvas.create_text(
                (x0 + x1) / 2,
                self.header_height / 2,
                text=label,
                anchor="center",
                font=("Arial", 10, "bold")
            )

            self.header_ids.append((rect_id, text_id, col_name))
            self.header_canvas.tag_bind(rect_id, "<Button-1>",
                                        lambda e, c=col_name: self._header_clicked(c))
            self.header_canvas.tag_bind(text_id, "<Button-1>",
                                        lambda e, c=col_name: self._header_clicked(c))

        total_width = sum(self.col_widths)
        self.header_canvas.configure(scrollregion=(0, 0, total_width, self.header_height))

    def _header_clicked(self, col):
        if self.on_header_click:
            self.on_header_click(col)

    
    def _sync_header(self, *_):
        x = self.canvas.xview()
        # same scroll offset applied to header
        self.header_canvas.xview_moveto(x[0])

    # ---------------------------------
    # CLEAR
    # ---------------------------------
    def clear(self):
        self.canvas.delete("all")
        self.rows = []
        self.draw_header()

    # ---------------------------------
    # ROW ADDING (unchanged except Y offset)
    # ---------------------------------
    def add_row(self, values, colors):
        row_index = len(self.rows)
        y0 = row_index * self.row_height
        y1 = y0 + self.row_height

        row_ids = []

        for col_i, text in enumerate(values):
            x0 = sum(self.col_widths[:col_i])
            width = self.col_widths[col_i]
            x1 = x0 + width

            bg = colors.get(col_i, "#ffffff")

            rect_id = self.canvas.create_rectangle(
                x0, y0, x1, y1,
                fill=bg, outline="black"
            )

            if isinstance(text, tk.PhotoImage):
                target = self.row_height - 4
                w, h = text.width(), text.height()
                scale = max(1, min(w // target, h // target))
                resized = text.subsample(scale, scale)
                if not hasattr(self, "_img_refs"):
                    self._img_refs = []
                self._img_refs.append(resized)

                text_id = self.canvas.create_image(
                    x0 + width / 2,
                    (y0 + y1) / 2,
                    image=resized,
                    anchor="center"
                )
            else:
                text_id = self.canvas.create_text(
                    x0 + width / 2,
                    (y0 + y1) / 2,
                    text=text,
                    font=("Arial", 9),
                    anchor="center"
                )

            row_ids.append((rect_id, text_id))

        self.rows.append((values, colors, row_ids))

        total_h = len(self.rows) * self.row_height
        total_w = sum(self.col_widths)
        self.canvas.configure(scrollregion=(0, 0, total_w, total_h))
    # -----------------------------------------------------------
    # Used by external table code: keep arrow state
    # -----------------------------------------------------------
    def set_sort_state(self, sort_state):
        """
        sort_state = { "col": "DXCC", "dir": "asc" }
        """
        self.sort_state.clear()

        col = sort_state.get("col")
        direction = sort_state.get("dir")

        if col and direction in ("asc", "desc"):
            self.sort_state[col] = direction

        # redraw to update arrows
        self.draw_header()


    def _on_mousewheel(self, event):
        # Windows / Linux
        self.canvas.yview_scroll(int(-1 * (event.delta / 120)), "units")

    def _on_mousewheel_linux(self, event):
        # Linux wheel emulates Button-4 (up) en Button-5 (down)
        if event.num == 4:
            self.canvas.yview_scroll(-1, "units")
        elif event.num == 5:
            self.canvas.yview_scroll(1, "units")

    def _on_mousewheel_mac(self, event):
        # macOS heeft kleinere delta’s
        self.canvas.yview_scroll(int(-1 * event.delta), "units")








def open_statistics_window():
    global Statistics_Window

    # already open?
    if Statistics_Window is not None and Statistics_Window.winfo_exists():
        Statistics_Window.lift()
        Statistics_Window.focus_force()
        return

    Statistics_Window = tk.Toplevel(Logbook_Window)
    Statistics_Window.title("Logbook Statistics")
    Statistics_Window.resizable(True, True)
    center_window_over(Logbook_Window, Statistics_Window, 1250, 800)

    # close handler
    def on_close():
        global Statistics_Window
        try:
            Statistics_Window.destroy()
        except:
            pass
        Statistics_Window = None

    Statistics_Window.protocol("WM_DELETE_WINDOW", on_close)

    main = tk.Frame(Statistics_Window)
    main.pack(fill="both", expand=True, padx=10, pady=10)

    # =====================================================
    # CONFIG (change here)
    # =====================================================
    BAND_WIDTH   = 140
    BAND_HEIGHT  = 240

    MODE_WIDTH   = 140
    MODE_HEIGHT  = 240

    CONF_WIDTH   = 140
    CONF_HEIGHT  = 240

    # =====================================================
    # LEFT PANEL
    # =====================================================
    left_panel = tk.Frame(main)
    left_panel.pack(side="left", fill="both", padx=(0,15), expand=False)

    # ---------------- Summary ----------------
    lf_summary = tk.LabelFrame(left_panel, text="Summary",
                               font=("Arial", 11, "bold"),
                               relief="groove", borderwidth=2, labelanchor="n")
    lf_summary.grid(row=0, column=0, columnspan=2, sticky="ew", pady=(0,10))
    lf_summary.grid_columnconfigure(0, weight=1)

    dxcc_set = {str(q.get("dxcc")) for q in qso_lines if str(q.get("dxcc","")).strip()}
    total_unique_dxcc = len(dxcc_set)

    band_count = {}
    confirmed_count = {}
    mode_count = {}

    # single loop calculates everything
    for q in qso_lines:
        band = q.get("Band", "")
        if band:
            band_count[band] = band_count.get(band, 0) + 1

            # confirmed via QSL or LoTW
            if q.get("QR") == "Yes" or q.get("LWR") == "Yes":
                confirmed_count[band] = confirmed_count.get(band, 0) + 1

        mode = q.get("Mode","")
        if mode:
            mode_count[mode] = mode_count.get(mode, 0) + 1

    qsl_sent = sum(1 for q in qso_lines if q.get("QS") == "Yes")
    qsl_recv = sum(1 for q in qso_lines if q.get("QR") == "Yes")
    lotw_sent = sum(1 for q in qso_lines if q.get("LWS") == "Yes")
    lotw_recv = sum(1 for q in qso_lines if q.get("LWR") == "Yes")

    def add_summary(label, value):
        row = tk.Frame(lf_summary)
        row.pack(anchor="center", pady=1)
        tk.Label(row, text=f"{label}:", width=20, anchor="e").pack(side="left")
        tk.Label(row, text=value, width=20, anchor="w").pack(side="left")

    logname = os.path.basename(CURRENT_JSON_FILE) if CURRENT_JSON_FILE else "Unknown"

    add_summary("Logbook", logname)
    add_summary("Total QSOs", len(qso_lines))
    add_summary("Unique DXCC", total_unique_dxcc)
    add_summary("QSL Sent", qsl_sent)
    add_summary("QSL Received", qsl_recv)
    add_summary("LoTW Sent", lotw_sent)
    add_summary("LoTW Received", lotw_recv)

    # =====================================================
    # Bands used
    # =====================================================
    used_bands = {q.get("Band","") for q in qso_lines if q.get("Band","")}
    used_bands.discard("")
    band_order = list(band_to_frequency.keys())
    known = [b for b in band_order if b in used_bands]
    unknown = sorted(b for b in used_bands if b not in band_order)
    all_bands = known + unknown

    # =====================================================
    # QSOs per Band
    # =====================================================
    lf_band = tk.LabelFrame(left_panel, text="QSOs per Band",
                            font=("Arial", 11, "bold"),
                            relief="groove", borderwidth=2, labelanchor="n")
    lf_band.grid(row=1, column=0, sticky="nsew", padx=(0,10))
    lf_band.grid_columnconfigure(0, weight=1)

    band_canvas = tk.Canvas(lf_band, width=BAND_WIDTH, height=BAND_HEIGHT)
    band_scroll = ttk.Scrollbar(lf_band, orient="vertical", command=band_canvas.yview)
    band_canvas.configure(yscrollcommand=band_scroll.set)
    band_frame = tk.Frame(band_canvas)
    band_canvas.create_window((0,0), window=band_frame, anchor="nw")

    band_canvas.grid(row=0, column=0, sticky="nsew")
    band_scroll.grid(row=0, column=1, sticky="ns")

    def resize_band(_):
        band_canvas.configure(scrollregion=band_canvas.bbox("all"))
    band_frame.bind("<Configure>", resize_band)

    for b, c in sorted(band_count.items()):
        row = tk.Frame(band_frame)
        row.pack(anchor="w")
        tk.Label(row, text=f"{b}:", width=10, anchor="e").pack(side="left")
        tk.Label(row, text=c, width=10, anchor="w").pack(side="left")

    # =====================================================
    # Confirmed QSOs
    # =====================================================
    lf_conf = tk.LabelFrame(left_panel, text="Confirmed QSOs",
                            font=("Arial", 11, "bold"),
                            relief="groove", borderwidth=2, labelanchor="n")
    lf_conf.grid(row=2, column=0, sticky="nsew")
    lf_conf.grid_columnconfigure(0, weight=1)

    conf_canvas = tk.Canvas(lf_conf, width=CONF_WIDTH, height=CONF_HEIGHT)
    conf_scroll = ttk.Scrollbar(lf_conf, orient="vertical", command=conf_canvas.yview)
    conf_canvas.configure(yscrollcommand=conf_scroll.set)
    conf_frame = tk.Frame(conf_canvas)
    conf_canvas.create_window((0,0), window=conf_frame, anchor="nw")

    conf_canvas.grid(row=0, column=0, sticky="nsew")
    conf_scroll.grid(row=0, column=1, sticky="ns")

    def resize_conf(_):
        conf_canvas.configure(scrollregion=conf_canvas.bbox("all"))
    conf_frame.bind("<Configure>", resize_conf)

    for b in all_bands:
        row = tk.Frame(conf_frame)
        row.pack(anchor="w")
        count = confirmed_count.get(b, 0)
        tk.Label(row, text=f"{b}:", width=10, anchor="e").pack(side="left")
        tk.Label(row, text=count, width=10, anchor="w").pack(side="left")

    # =====================================================
    # From here: your existing MODE + Right panel code unchanged
    # =====================================================

    # (keep the rest of your code after here)

    # MODE TABLE
    lf_mode = tk.LabelFrame(
        left_panel,
        text="QSOs per Mode",
        font=("Arial", 11, "bold"),
        relief="groove",
        borderwidth=2,
        labelanchor="n"
    )
    lf_mode.grid(row=1, column=1, sticky="nsew")
    lf_mode.grid_columnconfigure(0, weight=1)

    mode_canvas = tk.Canvas(lf_mode, width=MODE_WIDTH, height=MODE_HEIGHT)
    mode_scroll = ttk.Scrollbar(lf_mode, orient="vertical", command=mode_canvas.yview)
    mode_canvas.configure(yscrollcommand=mode_scroll.set)

    mode_frame = tk.Frame(mode_canvas)
    mode_canvas.create_window((0, 0), window=mode_frame, anchor="nw")

    mode_canvas.grid(row=0, column=0, sticky="nsew")
    mode_scroll.grid(row=0, column=1, sticky="ns")

    def resize_mode(_):
        mode_canvas.configure(scrollregion=mode_canvas.bbox("all"))

    mode_frame.bind("<Configure>", resize_mode)

    for m, c in sorted(mode_count.items()):
        row = tk.Frame(mode_frame)
        row.pack(anchor="w")
        tk.Label(row, text=f"{m}:", width=10, anchor="e").pack(side="left")
        tk.Label(row, text=c, width=10, anchor="w").pack(side="left")

    # ===============================================================
    # RIGHT PANEL (filters + canvas table)
    # ===============================================================
    right_panel = tk.Frame(main)
    right_panel.pack(side="right", fill="both", expand=True)

    # bands in your order
    used_bands = {q.get("Band", "") for q in qso_lines if q.get("Band", "")}
    used_bands.discard("")
    band_order = list(band_to_frequency.keys())
    known = [b for b in band_order if b in used_bands]
    unknown = sorted(b for b in used_bands if b not in band_order)
    all_bands = known + unknown

    # filter header
    filter_frame = tk.Frame(right_panel)
    filter_frame.pack(fill="x")

    tk.Label(filter_frame, text="Bands:", font=("Arial", 10, "bold")).pack(side="left")

    band_vars = {}
    for b in all_bands:
        var = tk.BooleanVar(value=True)
        band_vars[b] = var
        tk.Checkbutton(filter_frame, text=b, variable=var).pack(side="left", padx=2)

    # container for tabel
    table_container = tk.Frame(right_panel)
    table_container.pack(fill="both", expand=True)

    # Sort status (3-way): None → asc → desc → None
    sort_state = {"col": None, "dir": None}  # dir is "asc"/"desc"/None

    # ---------------------------------------------------------------
    # DX stats helper
    # ---------------------------------------------------------------
    def build_dx_stats(selected_bands):
        dx_stats = {}

        for q in qso_lines:
            dxcc = str(q.get("dxcc", "")).strip()
            if not dxcc:
                continue

            if dxcc not in dx_stats:
                dx_stats[dxcc] = {
                    "entity": q.get("Country", ""),
                    "total": 0,
                    "CW": 0,
                    "Phone": 0,
                    "Digital": 0,
                    "bands": {b: {"W": False, "C": False} for b in selected_bands}
                }

            dx_stats[dxcc]["total"] += 1

            mode = q.get("Mode", "").upper()
            band = q.get("Band", "")

            if mode in ("CW", "CWR", "CWL"):
                dx_stats[dxcc]["CW"] += 1
            elif mode in ("SSB", "USB", "LSB", "FM", "AM"):
                dx_stats[dxcc]["Phone"] += 1
            else:
                dx_stats[dxcc]["Digital"] += 1

            if band in selected_bands:
                dx_stats[dxcc]["bands"][band]["W"] = True

            if q.get("LWR") == "Yes" or q.get("QR") == "Yes":
                if band in selected_bands:
                    dx_stats[dxcc]["bands"][band]["C"] = True

        return dx_stats

    # ---------------------------------------------------------------
    # rebuild_table: uses sort_state["col"] / ["dir"]
    # ---------------------------------------------------------------
    def rebuild_table():
        selected = [b for b, v in band_vars.items() if v.get()]
        if not selected:
            messagebox.showwarning("No bands", "Select at least one band.")
            return

        dx_stats = build_dx_stats(selected)

        base_cols = ["DXCC", "Flag", "Entity", "Total", "CW", "Phone", "Digital"]
        columns = base_cols + selected

        # remove old table in container
        for child in table_container.winfo_children():
            child.destroy()

        # helper: veilige sort key voor DXCC (string/int mix voorkomen)
        def dxcc_sort_key(item):
            dxcc_key = str(item[0])
            if dxcc_key.isdigit():
                # numerieke DXCC eerst, oplopend
                return (0, int(dxcc_key))
            # niet-numerieke DXCC daarna, alfabetisch
            return (1, dxcc_key)

        # callback voor header click
        def header_clicked(col):
            prev_col = sort_state["col"]
            prev_dir = sort_state["dir"]

            if prev_col != col:
                # New column → start with ascending
                sort_state["col"] = col
                sort_state["dir"] = "asc"
            else:
                # Same column → toggle 3-steps
                if prev_dir == "asc":
                    sort_state["dir"] = "desc"
                elif prev_dir == "desc":
                    sort_state["dir"] = None
                    sort_state["col"] = None
                else:
                    sort_state["dir"] = "asc"

            rebuild_table()

        table = CanvasTable(table_container, columns, on_header_click=header_clicked)

        # Show arrows
        table.set_sort_state(sort_state)
        table.draw_header()

        table.pack(fill="both", expand=True)

        table.canvas.bind(
            "<Enter>",
            lambda e: table.canvas.bind_all("<MouseWheel>", table._on_mousewheel)
        )
        table.canvas.bind(
            "<Leave>",
            lambda e: table.canvas.unbind_all("<MouseWheel>")
        )

        COLOR_W = "#b8ffb8"
        COLOR_C = "#ffd5a6"
        COLOR_N = "#f0f0f0"

        items = list(dx_stats.items())  # (dxcc, info)
        col = sort_state["col"]
        direction = sort_state["dir"]

        # --------------------
        # Sort
        # --------------------
        if col is not None and direction in ("asc", "desc"):
            reverse = (direction == "desc")

            def key_func(item):
                dxcc, info = item

                if col == "DXCC":
                    # zelfde sleutel als default, maar nu op DXCC-kolom
                    return dxcc_sort_key((dxcc, info))
                elif col == "Entity":
                    return info["entity"].lower()
                elif col in ("Total", "CW", "Phone", "Digital"):
                    field_map = {
                        "Total": "total",
                        "CW": "CW",
                        "Phone": "Phone",
                        "Digital": "Digital",
                    }
                    return info[field_map[col]]
                elif col in selected:
                    f = info["bands"][col]
                    if f["C"]:
                        return 2
                    elif f["W"]:
                        return 1
                    else:
                        return 0
                else:
                    return dxcc

            items.sort(key=key_func, reverse=reverse)
        else:
            # default sort: DXCC ascending (numeric waar mogelijk)
            items.sort(key=dxcc_sort_key)

        # --------------------
        # Fill
        # --------------------
        for dxcc, info in items:
            # Select Flag Image
            try:
                dxcc_int = int(dxcc)
                flag_img = FLAG_IMAGES.get(dxcc_int, FLAG_IMAGES.get(-1))
            except Exception:
                flag_img = FLAG_IMAGES.get(-1)

            base_values = [
                dxcc,
                flag_img,
                info["entity"],
                info["total"],
                info["CW"],
                info["Phone"],
                info["Digital"]
            ]

            values = base_values[:]
            colors = {}
            offset = len(base_values)

            for i, b in enumerate(selected):
                cell = info["bands"][b]
                if cell["C"]:
                    values.append("CFM")
                    colors[offset + i] = COLOR_C
                elif cell["W"]:
                    values.append("WKD")
                    colors[offset + i] = COLOR_W
                else:
                    values.append("")
                    colors[offset + i] = COLOR_N

            table.add_row(values, colors)

    # Update-button
    tk.Button(filter_frame, text="Update Table", command=rebuild_table).pack(
        side="left", padx=6, pady=6
    )
    # Exit-button
    tk.Button(filter_frame, text="Exit", width=10, command=on_close).pack(
        side="right", padx=6, pady=6
    )

    # First Build
    rebuild_table()






def update_name_from_qrz():
    """
    Bulk-update the 'Name' field of selected QSOs using QRZ XML lookup.

    - Uses QRZ credentials from config.ini
    - Runs in background thread
    - Progress window centered over logbook
    - Works with search_iids and sorted tree via tree_to_log_index
    """
    global qso_lines

    selected_items = tree.selection()
    if not selected_items:
        messagebox.showinfo("QRZ Name Update", "Select one or more QSOs first.", parent=Logbook_Window)
        return

    if not messagebox.askyesno(
        "Confirm QRZ Name Update",
        f"Update the Name field via QRZ lookup for {len(selected_items)} QSO(s)?\n\n"
        f"This will overwrite existing Name values.",
        parent=Logbook_Window
    ):
        return

    # ---------------------------------------------------------
    # Get QRZ credentials
    # ---------------------------------------------------------
    try:
        username = config.get("QRZ", "username", fallback="").strip()
        password = config.get("QRZ", "password", fallback="").strip()
    except Exception as e:
        print("[DEBUG] QRZ credentials error:", e)
        messagebox.showerror("QRZ Error", "Failed to read QRZ credentials from config.ini")
        return

    if not username or not password:
        messagebox.showerror("QRZ Error",
                             "QRZ username or password is missing in config.ini")
        return

    # ---------------------------------------------------------
    # Create progress window centered over Logbook
    # ---------------------------------------------------------
    progress_window = tk.Toplevel(Logbook_Window)
    progress_window.title("Updating Names via QRZ")
    progress_window.resizable(False, False)
    progress_window.transient(Logbook_Window)
    progress_window.grab_set()

    center_window_over(Logbook_Window, progress_window, 380, 150)

    tk.Label(progress_window, text="Updating Names from QRZ…").pack(pady=(10, 5))
    progress_label = tk.Label(progress_window, text="0 / 0")
    progress_label.pack(pady=3)
    progress_bar = ttk.Progressbar(progress_window, orient="horizontal",
                                   length=320, mode="determinate")
    progress_bar.pack(pady=5)

    # ---------------------------------------------------------
    # Safe XML lookup
    # ---------------------------------------------------------
    def safe_qrz_lookup(callsign, session_key):
        ns = {"qrz": "http://xmldata.qrz.com"}

        def do_query(cs):
            try:
                url = f"https://xmldata.qrz.com/xml/current/?s={session_key}&callsign={cs}"
                r = requests.get(url, timeout=5)
                root = ET.fromstring(r.content)
                return root.find(".//qrz:Callsign", ns)
            except Exception as e:
                print(f"[DEBUG] QRZ lookup error for {cs}: {e}")
                return None

        node = do_query(callsign)

        # fallback PD5DJ/P → PD5DJ
        if node is None:
            core = extract_core(callsign)
            if core != callsign:
                node = do_query(core)

        if node is None:
            return None

        fname = node.findtext("qrz:fname", "", ns)
        lname = node.findtext("qrz:name", "", ns)
        full = f"{fname} {lname}".strip()
        return full if full else None

    # ---------------------------------------------------------
    # Worker thread
    # ---------------------------------------------------------
    def worker():
        session_key, err = get_session_key(username, password)
        if not session_key:
            def fail():
                messagebox.showerror("QRZ Login Failed", err or "Login failed.")
                progress_window.destroy()
            root.after(0, fail)
            return

        total = len(selected_items)
        updated_count = 0
        progress_bar["maximum"] = total

        for i, item in enumerate(selected_items, 1):

            # -------------------------------------------
            # RESOLVE TREE IID → QSO INDEX
            # -------------------------------------------
            try:
                q_index = tree_to_log_index[item]
            except Exception as e:
                print(f"[DEBUG] Worker error on {item}: {e}")
                continue

            qso = qso_lines[q_index]
            call = qso.get("Callsign", "").strip().upper()
            if not call:
                continue

            new_name = safe_qrz_lookup(call, session_key)
            if new_name:
                qso["Name"] = new_name
                updated_count += 1

            # progress UI update
            def update(i=i, total=total):
                progress_label.config(text=f"{i} / {total}")
                progress_bar["value"] = i
            root.after(0, update)

        save_to_json()

        def finish():
            load_json_content_to_tree()
            update_worked_before_tree()
            progress_window.destroy()
            messagebox.showinfo(
                "QRZ Name Update",
                f"Finished.\nUpdated {updated_count} QSO(s)."
            )
        root.after(0, finish)

    threading.Thread(target=worker, daemon=True).start()





def update_dxcc_selected():
    """
    Threaded DXCC updater with confirmation dialog and progress window.
    Updates: Country, Continent, DXCC, CQZ, ITUZ.
    Window is centered using center_window_over().
    Fully supports search_iids and sorted tree via tree_to_log_index.
    """

    selected = tree.selection()
    if not selected:
        messagebox.showinfo("Update DXCC", "No QSOs selected.")
        return

    # ------------------------------------------------------------------
    # Confirmation dialog
    # ------------------------------------------------------------------
    if not messagebox.askyesno(
        "Confirm DXCC Update",
        f"You are about to update DXCC, Country and Continent\n"
        f"for {len(selected)} QSO(s).\n\n"
        f"This will overwrite existing country information.\n\n"
        f"Proceed?",
        parent=Logbook_Window
    ):
        return

    # ------------------------------------------------------------------
    # Progress window
    # ------------------------------------------------------------------
    progress_window = tk.Toplevel(Logbook_Window)
    progress_window.title("Updating DXCC...")
    progress_window.resizable(False, False)

    center_window_over(Logbook_Window, progress_window, 320, 80)

    tk.Label(progress_window, text="Updating DXCC information...").pack(pady=5)

    progress_bar = ttk.Progressbar(progress_window, length=260, mode="determinate")
    progress_bar.pack(pady=5)

    progress_bar["maximum"] = len(selected)
    progress_bar["value"] = 0

    progress_window.grab_set()

    # ------------------------------------------------------------------
    # Worker thread
    # ------------------------------------------------------------------
    def worker():
        updated = 0
        count = 0

        for item in selected:
            try:
                # -------------------------------------------
                # FIX: support search_iids and sorted tree
                # -------------------------------------------
                try:
                    qso_index = tree_to_log_index[item]
                except KeyError:
                    print(f"[DXCC] Could not map tree item '{item}' to qso index.")
                    continue

                qso = qso_lines[qso_index]
                callsign = qso.get("Callsign", "").strip().upper()
                if not callsign:
                    continue

                # ----------------------------------------------------------
                # DXCC prefix matching (longest prefix wins)
                # ----------------------------------------------------------
                best_match = None
                best_len = 0
                best_prefix = ""

                for entry in dxcc_data:
                    for raw_prefix in entry.prefixes:
                        prefix = re.sub(r'[=;()\[\]*]', '', raw_prefix).strip().upper()
                        if callsign.startswith(prefix) and len(prefix) > best_len:
                            best_match = entry
                            best_len = len(prefix)
                            best_prefix = prefix

                # ----------------------------------------------------------
                # Save DXCC info  (UITGEBREID: CQZ / ITUZ)
                # ----------------------------------------------------------
                if best_match:
                    qso["Country"] = best_match.name
                    qso["Continent"] = best_match.continent

                    # 👉 NIEUW: zones mee updaten
                    qso["CQZ"] = best_match.cq_zone
                    qso["ITUZ"] = best_match.itu_zone

                    dxcc_code = prefix_to_dxcc.get(best_prefix)
                    if dxcc_code is not None:
                        qso["dxcc"] = int(dxcc_code)

                updated += 1

            except Exception as e:
                print(f"DXCC update error for row {item}: {e}")

            count += 1
            root.after(0, lambda c=count: progress_bar.config(value=c))

        # --------------------------------------------------------------
        # Final UI update
        # --------------------------------------------------------------
        def finish():
            save_to_json()
            load_json_content_to_tree()
            update_worked_before_tree()

            progress_window.destroy()

            messagebox.showinfo(
                "DXCC Update Complete",
                f"DXCC information updated for {updated} QSO(s).",
                parent=Logbook_Window
            )

        root.after(0, finish)

    threading.Thread(target=worker, daemon=True).start()





#    _____    _     _     _    ___ __ _ ___ __ __
#   |_  | |\|| \   | \| ||_)|   | /  |_| | |_ (_ 
#   |  _|_| ||_/   |_/|_||  |___|_\__| | | |____)
def find_duplicates(check_hours=True, check_minutes=True, check_seconds=False):
    """
    Detect duplicate QSOs and display them in a centered popup window.
    Window is always centered over the Logbook_Window using center_window_over().
    """

    if not qso_lines:
        messagebox.showinfo("Duplicates", "No logbook loaded or empty.")
        return

    from datetime import datetime

    # ------------------------------------------------------------------
    # Helper: Build a time key with user-selected time precision
    # ------------------------------------------------------------------
    def make_time_key(qso):
        call = qso.get("Callsign", "").strip().upper()
        date_str = qso.get("Date", "").strip()
        time_str = qso.get("Time", "").strip()
        if not date_str or not time_str:
            return None
        try:
            t = datetime.strptime(f"{date_str} {time_str}", "%Y-%m-%d %H:%M:%S")
        except ValueError:
            return None

        # Reduce precision depending on settings
        if not check_seconds:
            t = t.replace(second=0)
        if not check_minutes:
            t = t.replace(minute=0, second=0)
        if not check_hours:
            t = t.replace(hour=0, minute=0, second=0)

        return (call, t.date(), t.time())

    seen = {}
    duplicate_index_map = {}

    # -----------------------------------------------------------
    # Scan all QSOs
    # -----------------------------------------------------------
    for idx, qso in enumerate(qso_lines):
        key = make_time_key(qso)
        if not key:
            continue

        if key in seen:
            if key not in duplicate_index_map:
                duplicate_index_map[key] = [seen[key]]
            duplicate_index_map[key].append(idx)
        else:
            seen[key] = idx

    all_duplicate_indices = []
    for indices in duplicate_index_map.values():
        all_duplicate_indices.extend(indices)

    #if not all_duplicate_indices:
        #messagebox.showinfo("Duplicates", "No duplicate QSOs found.")
        #return

    total_groups = len(duplicate_index_map)
    total_dupes = len(all_duplicate_indices)

    # ========================================================
    # Create result window
    # ========================================================
    dup_window = tk.Toplevel(Logbook_Window)
    dup_window.title("Duplicate Finder")
    win_w, win_h = 720, 520
    dup_window.geometry(f"{win_w}x{win_h}")

    # Center over Logbook window AFTER initialization
    dup_window.after(
        10,
        lambda: center_window_over(Logbook_Window, dup_window, win_w, win_h)
    )

    # ========================================================
    # Time precision options
    # ========================================================
    frame_time = ttk.LabelFrame(dup_window, text="Time compare precision")
    frame_time.pack(fill="x", padx=10, pady=5)

    var_hours = tk.BooleanVar(value=check_hours)
    var_minutes = tk.BooleanVar(value=check_minutes)
    var_seconds = tk.BooleanVar(value=check_seconds)

    ttk.Checkbutton(frame_time, text="Hours", variable=var_hours).pack(side="left", padx=5)
    ttk.Checkbutton(frame_time, text="Minutes", variable=var_minutes).pack(side="left", padx=5)
    ttk.Checkbutton(frame_time, text="Seconds", variable=var_seconds).pack(side="left", padx=5)

    # Re-run using selected precision
    def find_again():
        dup_window.destroy()
        find_duplicates(
            check_hours=var_hours.get(),
            check_minutes=var_minutes.get(),
            check_seconds=var_seconds.get()
        )

    ttk.Button(frame_time, text="Find Duplicates", command=find_again).pack(side="right", padx=10)

    # ========================================================
    # Summary label
    # ========================================================
    info_text = f"{total_groups} duplicate groups found, {total_dupes} total duplicate QSOs."
    lbl_info = ttk.Label(dup_window, text=info_text, font=("Segoe UI", 10, "bold"))
    lbl_info.pack(pady=(5, 0))

    # ========================================================
    # Results Treeview
    # ========================================================
    frame = tk.Frame(dup_window)
    frame.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)

    tree_dup = ttk.Treeview(
        frame,
        columns=("Index", "Callsign", "Date", "Time", "Mode", "Frequency"),
        show="headings",
        selectmode="extended"
    )

    for col in ("Index", "Callsign", "Date", "Time", "Mode", "Frequency"):
        tree_dup.heading(col, text=col)
        tree_dup.column(col, anchor="center")

    scrollbar = ttk.Scrollbar(frame, orient="vertical", command=tree_dup.yview)
    tree_dup.configure(yscrollcommand=scrollbar.set)

    scrollbar.pack(side="right", fill="y")
    tree_dup.pack(side="left", fill=tk.BOTH, expand=True)

    # Apply alternating colors per group
    tree_to_log_index = {}
    color_styles = ["lightblue", "white", "lightgreen", "lightyellow", "lightpink", "lavender", "lightgray"]

    for i, color in enumerate(color_styles):
        tree_dup.tag_configure(f"tag_{i}", background=color)

    for group_index, (key, indices) in enumerate(duplicate_index_map.items()):
        tag_name = f"tag_{group_index % len(color_styles)}"
        for idx in indices:
            qso = qso_lines[idx]
            iid = f"dup_{idx}"
            tree_dup.insert(
                "", "end", iid=iid,
                values=(
                    idx,
                    qso.get("Callsign"),
                    qso.get("Date"),
                    qso.get("Time"),
                    qso.get("Mode"),
                    qso.get("Frequency")
                ),
                tags=(tag_name,)
            )
            tree_to_log_index[iid] = idx

    # ========================================================
    # Action buttons
    # ========================================================
    def delete_selected_duplicates():
        selected_iids = tree_dup.selection()
        if not selected_iids:
            messagebox.showwarning("No Selection", "Select records to delete.")
            return

        if not messagebox.askyesno("Confirm Delete", f"Delete {len(selected_iids)} selected duplicates?"):
            return

        indices_to_delete = sorted([tree_to_log_index[iid] for iid in selected_iids], reverse=True)
        for i in indices_to_delete:
            try:
                del qso_lines[i]
            except IndexError:
                continue

        save_to_json()
        load_json_content_to_tree()
        dup_window.destroy()
        messagebox.showinfo("Done", f"{len(indices_to_delete)} duplicates removed and saved.")

    def delete_all_duplicates_keep_best():
        fields_to_consider = ["Callsign", "Date", "Time", "Mode", "Frequency", "Name", "My Locator", "My Location"]
        to_delete_all = []

        for key, indices in duplicate_index_map.items():
            best_idx = max(indices, key=lambda i: sum(1 for f in fields_to_consider if qso_lines[i].get(f)))
            to_delete_all.extend(i for i in indices if i != best_idx)

        for i in sorted(to_delete_all, reverse=True):
            try:
                del qso_lines[i]
            except IndexError:
                continue

        save_to_json()
        load_json_content_to_tree()
        dup_window.destroy()
        messagebox.showinfo("Done", f"{len(to_delete_all)} duplicate QSOs removed, best entries kept.")

    tk.Button(dup_window, text="Delete Selected", command=delete_selected_duplicates).pack(pady=5)
    tk.Button(dup_window, text="Delete Duplicates / Best kept", command=delete_all_duplicates_keep_best).pack(pady=5)
    tk.Button(dup_window, text="Close", command=dup_window.destroy).pack(pady=5)




#########################################################################################
#  ___ _   _ _    _  __  ___ ___ ___ _____  __      _____ _  _ ___   _____      __
# | _ ) | | | |  | |/ / | __|   \_ _|_   _| \ \    / /_ _| \| |   \ / _ \ \    / /
# | _ \ |_| | |__| ' <  | _|| |) | |  | |    \ \/\/ / | || .` | |) | (_) \ \/\/ / 
# |___/\___/|____|_|\_\ |___|___/___| |_|     \_/\_/ |___|_|\_|___/ \___/ \_/\_/  
#                                                                                 
#########################################################################################

def open_bulk_edit_window():
    """Bulk edit multiple QSO records with FIELD_ALIAS-based field mapping."""

    PROPAGATION_MODES = [
        "AS", "AUE", "AUR", "BS", "ECH", "EME", "ES", "F2", "FAI",
        "GWAVE", "INTERNET", "ION", "IRL", "LOS", "MS", "RPT", "RS",
        "SAT", "TEP", "TR"
    ]

    selected_items = tree.selection()
    if not selected_items:
        messagebox.showinfo("No Selection", "Select one or more QSO records first.")
        return

    ALL_FIELDS = sorted(
        set(FIELD_ALIAS.keys()) | {
            "Date", "Time", "Callsign", "Name", "Country", "DXCC",
            "Sent", "Sent Exchange", "Received", "Receive Exchange",
            "Mode", "Submode", "Propagation",
            "Band", "Frequency", "Locator", "Comment",
            "Satellite",
            "WWFF", "POTA", "BOTA", "COTA", "IOTA", "SOTA", "WLOTA",
            "My Callsign", "My Operator", "My Locator", "My Location",
            "My WWFF", "My POTA", "My BOTA", "My COTA", "My IOTA",
            "My SOTA", "My WLOTA"
        }
    )


    # --- Create the window ---
    edit_window = tk.Toplevel(Logbook_Window)
    edit_window.title("Bulk Edit")
    edit_window.resizable(False, False)
    edit_window.transient(Logbook_Window)
    edit_window.grab_set()

    # --- Center using helper ---
    center_window_over(Logbook_Window, edit_window, 360, 260)


    tk.Label(edit_window, text="Field to edit:").pack(pady=5)
    field_var = tk.StringVar()
    field_combo = ttk.Combobox(
        edit_window, textvariable=field_var, state="readonly",
        font=('Arial', 10), width=25
    )
    field_combo['values'] = ALL_FIELDS
    field_combo.pack(pady=5)

    tk.Label(edit_window, text="New value:").pack(pady=5)
    value_frame = tk.Frame(edit_window)
    value_frame.pack(pady=5)
    value_widget = None

    def update_value_widget(*args):
        nonlocal value_widget
        if value_widget:
            value_widget.destroy()
            value_widget = None

        field = field_var.get()

        # QSL Yes/No
        if field in ["QSL Sent", "QSL Received", "LoTW Sent", "LoTW Received"]:
            cb = ttk.Combobox(value_frame, values=["Yes", "No"], state="readonly", width=20)
            cb.pack()
            value_widget = cb

        # QSL Via
        elif field in ["QSL Sent Via", "QSL Received Via"]:
            cb = ttk.Combobox(
                value_frame,
                values=["Bureau", "Direct", "Electronic", "Manager"],
                state="readonly", width=20
            )
            cb.pack()
            value_widget = cb

        # NEW: Propagation mode
        elif field == "Propagation":
            cb = ttk.Combobox(
                value_frame,
                values=[""] + PROPAGATION_MODES,
                state="readonly",
                width=20
            )
            cb.pack()
            value_widget = cb

        # Date picker
        elif "date" in field.lower():
            de = DateEntry(value_frame, date_pattern="yyyy-mm-dd", width=18)
            de.pack()
            value_widget = de

        # Default text entry
        else:
            ent = tk.Entry(value_frame, font=('Arial', 10), width=22)
            ent.pack()
            value_widget = ent

    field_var.trace_add("write", update_value_widget)

    uppercase_fields = [
        "Callsign", "Locator", "My Callsign", "My Operator", "My Locator",
        "My WWFF", "My POTA", "My BOTA", "My COTA", "My IOTA",
        "My SOTA", "My WLOTA", "Continent", "Mode", "Submode", "Prop",
        "WWFF", "POTA", "BOTA", "COTA", "IOTA", "SOTA", "WLOTA",
        "DXCC"
    ]





    def apply_bulk_edit():
        field = field_var.get()
        if not field:
            messagebox.showwarning("Missing Field", "Please select a field.", parent=Logbook_Window)
            return

        new_value = value_widget.get().strip() if value_widget else ""
        internal_field = FIELD_ALIAS.get(field, field)

        # DXCC must be numeric → convert safely
        if internal_field == "DXCC":
            if new_value == "":
                new_value = ""
            else:
                if not new_value.isdigit():
                    messagebox.showerror("Invalid DXCC", "DXCC must be a number (entity code).", parent=Logbook_Window)
                    return
                new_value = int(new_value)


        # Validate locator
        if field.lower() == "locator" and new_value and not is_valid_locator(new_value):
            messagebox.showerror("Invalid Locator", "Locator must be valid (≥4 chars). Example: FN31 or JN58TD.")
            return

        # Validate date
        if "date" in field.lower() and new_value:
            try:
                datetime.strptime(new_value, "%Y-%m-%d")
            except ValueError:
                messagebox.showerror("Invalid Date", "Date must be YYYY-MM-DD.", parent=Logbook_Window)
                return

        # Validate time
        if field.lower() == "time" and new_value:
            try:
                datetime.strptime(new_value, "%H:%M:%S")
            except ValueError:
                messagebox.showerror("Invalid Time", "Time must be HH:MM:SS", parent=Logbook_Window)
                return

        # Uppercase fields
        if field in uppercase_fields:
            new_value = new_value.upper()

        confirm = messagebox.askyesno(
            "Confirm Bulk Edit",
            f"Apply value '{new_value}' to field '{field}' for {len(selected_items)} QSO(s)?", parent=Logbook_Window
        )
        if not confirm:
            return

        updated_count = 0
        now_str = datetime.now().strftime("%Y-%m-%d")

        for item in selected_items:
            qso_index = tree_to_log_index.get(item)
            if qso_index is None:
                continue

            if 0 <= qso_index < len(qso_lines):
                qso = qso_lines[qso_index]

                OPTIONAL_REF_FIELDS = [
                    "WWFF", "POTA", "BOTA", "COTA", "IOTA", "SOTA", "WLOTA", "Satellite",
                    "My WWFF", "My POTA", "My BOTA", "My COTA",
                    "My IOTA", "My SOTA", "My WLOTA"
                ]

                # Optional field remove
                if internal_field in OPTIONAL_REF_FIELDS and not new_value:
                    qso.pop(internal_field, None)
                else:
                    qso[internal_field] = new_value

                # QSL logic
                if internal_field == "QS":
                    qso["QSD"] = now_str if new_value == "Yes" else ""
                    if new_value == "No":
                        qso["QSV"] = ""

                elif internal_field == "QR":
                    qso["QRD"] = now_str if new_value == "Yes" else ""
                    if new_value == "No":
                        qso["QRV"] = ""

                # LoTW logic
                elif internal_field == "LWS":
                    qso["LWSD"] = now_str if new_value == "Yes" else ""

                elif internal_field == "LWR":
                    qso["LWRD"] = now_str if new_value == "Yes" else ""

                updated_count += 1

        save_to_json()
        load_json_content_to_tree()
        update_worked_before_tree()
        edit_window.destroy()
        messagebox.showinfo("Success", f"{updated_count} QSO(s) updated.", parent=Logbook_Window)

    btn_frame = tk.Frame(edit_window)
    btn_frame.pack(pady=10)
    tk.Button(btn_frame, text="Save", command=apply_bulk_edit, width=10).pack(side="left", padx=10)
    tk.Button(btn_frame, text="Cancel", command=edit_window.destroy, width=10).pack(side="right", padx=10)




def export_selected_to_adif():
    log.info("EXPORT_SELECTED_ADIF called")

    if not tree:
        log.error("EXPORT_SELECTED_ADIF aborted: Treeview not available")
        messagebox.showerror("Error", "Treeview not available.", parent=Logbook_Window)
        return

    selected_items = tree.selection()
    if not selected_items:
        log.info("EXPORT_SELECTED_ADIF aborted: no QSOs selected")
        messagebox.showinfo("Export to ADIF", "No QSOs selected.", parent=Logbook_Window)
        return

    log.info(f"EXPORT_SELECTED_ADIF selected {len(selected_items)} QSOs")

    adif_lines = []
    for item in selected_items:
        values = tree.item(item)['values']
        qso = next(
            (q for q in qso_lines
             if q.get("Date") == values[0]
             and q.get("Time") == values[1]
             and q.get("Callsign") == values[2]),
            None
        )

        if not qso:
            log.warning(f"EXPORT_SELECTED_ADIF could not match QSO for item {values}")
            continue

        adif_line_list = []

        def add_field(tag, value):
            if value is not None and str(value).strip() != "":
                val = str(value).strip()
                adif_line_list.append(f"<{tag}:{len(val)}>{val}")

        # Base fields
        add_field("qso_date", qso.get("Date", "").replace("-", ""))
        add_field("time_on", qso.get("Time", "").replace(":", ""))
        add_field("call", qso.get("Callsign", ""))
        add_field("name", qso.get("Name", ""))
        add_field("rst_sent", qso.get("Sent", ""))
        add_field("rst_rcvd", qso.get("Received", ""))
        add_field("stx", qso.get("STX", ""))
        add_field("srx", qso.get("SRX", ""))
        add_field("mode", qso.get("Mode", ""))
        add_field("submode", qso.get("Submode", ""))
        add_field("prop_mode", qso.get("Prop", ""))
        add_field("band", qso.get("Band", ""))
        add_field("freq", qso.get("Frequency", ""))
        add_field("gridsquare", qso.get("Locator", ""))
        add_field("comment", qso.get("Comment", ""))
        add_field("station_callsign", qso.get("My Callsign", ""))
        add_field("operator", qso.get("My Operator", ""))
        add_field("country", qso.get("Country", ""))
        add_field("cont", qso.get("Continent", ""))
        add_field("CQZ", qso.get("CQZ", ""))
        add_field("ITUZ", qso.get("ITUZ", ""))
        add_field("sat_name", qso.get("Satellite", ""))

        # Special SIG fields
        for tag in ["WWFF", "POTA", "BOTA", "COTA", "SOTA", "IOTA", "WLOTA"]:
            if qso.get(tag):
                add_field("sig", tag)
                add_field("sig_info", qso.get(tag, ""))

        # QSL fields
        add_field("QSL_SENT",
                  "Y" if qso.get("QSL Sent", "").lower() == "yes"
                  else "N" if qso.get("QSL Sent") else "")
        add_field("QSLSDATE", qso.get("QSL Sent Date", "").replace("-", ""))
        add_field("QSL_RCVD",
                  "Y" if qso.get("QSL Received", "").lower() == "yes"
                  else "N" if qso.get("QSL Received") else "")
        add_field("QSLRDATE", qso.get("QSL Received Date", "").replace("-", ""))
        add_field("QSL_SENT_VIA", qso.get("QSL Sent Via", ""))
        add_field("QSL_RCVD_VIA", qso.get("QSL Received Via", ""))

        # LoTW
        lotw_sent = qso.get("LWS", "")
        lotw_sent_date = qso.get("LWSD", "")
        lotw_rcvd = qso.get("LWR", "")
        lotw_rcvd_date = qso.get("LWRD", "")

        if lotw_sent:
            add_field("LOTW_QSL_SENT", "Y" if lotw_sent.lower() == "yes" else "N")
        if lotw_sent_date:
            add_field("LOTW_QSLSDATE", lotw_sent_date.replace("-", ""))
        if lotw_rcvd:
            add_field("LOTW_QSL_RCVD", "Y" if lotw_rcvd.lower() == "yes" else "N")
        if lotw_rcvd_date:
            add_field("LOTW_QSLRDATE", lotw_rcvd_date.replace("-", ""))

        adif_lines.append(" ".join(adif_line_list) + " <EOR>\n")

    if not adif_lines:
        log.warning("EXPORT_SELECTED_ADIF aborted: no valid QSOs found")
        messagebox.showinfo("Export to ADIF", "No valid QSOs found for export.", parent=Logbook_Window)
        return

    export_file = filedialog.asksaveasfilename(
        defaultextension=".adi",
        filetypes=[("ADIF files", "*.adi")]
    )

    if export_file:
        log.info(f"EXPORT_SELECTED_ADIF writing to {export_file}")
        try:
            with open(export_file, "w", encoding="utf-8") as f:
                f.write("Generated by MiniBook\n")
                f.write("<ADIF_VER:5>3.1.0 <PROGRAMID:8>MiniBook <EOH>\n")
                f.writelines(adif_lines)
            log.info(f"EXPORT_SELECTED_ADIF success: exported {len(adif_lines)} QSOs")
            messagebox.showinfo("Export to ADIF",
                                f"{len(adif_lines)} QSOs exported successfully.",
                                parent=Logbook_Window)
        except Exception as e:
            log.exception("EXPORT_SELECTED_ADIF failed saving file")
            messagebox.showerror("Export Error", f"Failed to save ADIF file:\n{e}", parent=Logbook_Window)
    else:
        log.info("EXPORT_SELECTED_ADIF cancelled by user (no filename)")




def atomic_write_json(target_path, data, retries=10, delay=0.2):
    import time

    log.debug(f"atomic_write_json called → target={target_path}")

    folder = os.path.dirname(target_path)
    fd, tmp_path = tempfile.mkstemp(prefix="mbk_", dir=folder)

    log.debug(f"Temporary file created: {tmp_path}")

    try:
        with os.fdopen(fd, "w", encoding="utf-8") as f:
            json.dump(data, f, ensure_ascii=False, separators=(',', ':'))
            f.flush()
            os.fsync(f.fileno())
            log.debug("Write + flush + fsync OK")

        # attempt replacing target file
        for attempt in range(retries):
            try:
                log.debug(f"atomic replace attempt {attempt+1}/{retries}")
                os.replace(tmp_path, target_path)
                log.info(f"atomic_write_json SUCCESS → {target_path}")
                return
            except OSError as e:
                if getattr(e, "winerror", None) == 32:
                    log.warning(f"atomic_write_json: file locked, retrying...")
                    time.sleep(delay)
                else:
                    log.exception("atomic_write_json replace error")
                    raise

        # retried too long
        raise Exception("Atomic save failed: target locked too long")

    except Exception as e:
        log.exception(f"atomic_write_json FAILED → {e}")
        try:
            os.remove(tmp_path)
            log.debug(f"Temporary file removed: {tmp_path}")
        except Exception as e2:
            log.warning(f"Could not remove temp file {tmp_path}: {e2}")
        raise e



#########################################################################################
#    _   _____   ___  _  ___   _    ___   _   ___   _____   ___   _____ 
#   /_\ / __\ \ / / \| |/ __| | |  / _ \ /_\ |   \ / / __| /_\ \ / / __|
#  / _ \\__ \\ V /| .` | (__  | |_| (_) / _ \| |) / /\__ \/ _ \ V /| _| 
# /_/ \_\___/ |_| |_|\_|\___| |____\___/_/ \_\___/_/ |___/_/ \_\_/ |___|
#                                                                       
#########################################################################################                                                                       
# Async save throttle
save_pending = False
save_lock = threading.Lock()

# Function to save the QSO lines back to the JSON file
def save_to_json(updated_data=None):
    log.info("SAVE_JSON called")

    global qso_lines, CURRENT_JSON_FILE

    if not CURRENT_JSON_FILE:
        log.warning("SAVE_JSON aborted: no CURRENT_JSON_FILE")
        return

    try:
        # --------- Fast path for import & station setup ----------
        if updated_data is not None:
            log.debug("SAVE_JSON using updated_data atomic write")
            atomic_write_json(CURRENT_JSON_FILE, updated_data)
            log.info("SAVE_JSON updated_data written successfully")
            return

        # --------- Normal path: rebuild logbook ----------
        try:
            log.debug("Reading existing JSON to preserve Station/BookInfo")
            with open(CURRENT_JSON_FILE, "r", encoding="utf-8") as f:
                existing = json.load(f)
                bookinfo = existing.get("BookInfo", {"Version": "3"})
                station = existing.get("Station", {})
        except Exception as e:
            log.warning(f"Failed to read existing JSON for BookInfo/Station: {e}")
            bookinfo = {"Version": "3"}
            station = {}

        data_to_save = []
        for qso in qso_lines:
            q = dict(qso)
            q.pop("DateTime", None)
            data_to_save.append(q)

        log.info(f"Attempting atomic save: {len(data_to_save)} QSOs")

        atomic_write_json(CURRENT_JSON_FILE, {
            "BookInfo": bookinfo,
            "Station": station,
            "Logbook": data_to_save
        })

        log.info(f"SAVE_JSON succeeded ({len(data_to_save)} QSOs stored)")

    except Exception as e:
        log.exception(f"SAVE_JSON failed: {e}")
        messagebox.showerror(
            "Save Error",
            f"Could not save logbook:\n{e}",
            parent=Logbook_Window
        )




def save_async(updated_data=None):
    global save_pending

    if save_pending:
        return

    save_pending = True

    def worker():
        global save_pending   # <-- BELANGRIJK
        try:
            if updated_data is None:
                save_to_json()
            else:
                save_to_json(updated_data)
        finally:
            save_pending = False

            # Update STX in GUI thread after any add/edit/delete
            try:
                root.after(0, refresh_sent_serial_from_logbook)
            except Exception:
                pass            

    threading.Thread(target=worker, daemon=True).start()





#########################################################################################
#  ___ ___ ___ _____    ___  ___  ___  
# | __|   \_ _|_   _|  / _ \/ __|/ _ \ 
# | _|| |) | |  | |   | (_) \__ \ (_) |
# |___|___/___| |_|    \__\_\___/\___/ 
#                
#########################################################################################
# ---------- Edit window (uses global FIELD_ALIAS) ----------

def build_edit_window():
    """Build the Edit QSO window and widgets."""
    global Edit_Window, entries

    Edit_Window = tk.Toplevel(root)
    Edit_Window.title("Edit QSO Entry")
    Edit_Window.withdraw()

    parent = Logbook_Window if Logbook_Window and Logbook_Window.winfo_exists() else root
    center_window_over(parent, Edit_Window, 765, 750)

    Edit_Window.deiconify()

    form_frame = tk.Frame(Edit_Window)
    form_frame.pack(padx=10, pady=10, fill="both", expand=True)

    FIELD_LAYOUT = {
        'Date':              (0, 0, 'date'),
        'Time':              (1, 0, 'entry'),
        'Callsign':          (2, 0, 'entry'),
        'Name':              (3, 0, 'entry'),

        'Country':           (4, 0, 'entry'),
        'DXCC':              (5, 0, 'entry'),
        'CQZ':               (6, 0, 'readonly'),
        'ITUZ':              (7, 0, 'readonly'),

        'Band':              (8, 0, 'band'),
        'Mode':              (9, 0, 'mode'),
        'Submode':           (10, 0, 'submode'),
        'Sent':              (11, 0, 'entry'),
        'Received':          (12, 0, 'entry'),
        'Sent Exchange':     (13, 0, 'entry'),
        'Receive Exchange':  (14, 0, 'entry'),
        'Frequency':         (15, 0, 'entry'),
        'Locator':           (16, 0, 'entry'),
        'Comment':           (17, 0, 'entry'),
        'Satellite':         (18, 0, 'entry'),
        'Propagation':       (19, 0, 'prop'),

        'WWFF':              (0, 2, 'entry'),
        'POTA':              (1, 2, 'entry'),
        'BOTA':              (2, 2, 'entry'),
        'COTA':              (3, 2, 'entry'),
        'IOTA':              (4, 2, 'entry'),
        'SOTA':              (5, 2, 'entry'),
        'WLOTA':             (6, 2, 'entry'),

        'My Callsign':       (7, 2, 'entry'),
        'My Operator':       (8, 2, 'entry'),
        'My Locator':        (9, 2, 'entry'),
        'My Location':       (10, 2, 'entry'),
        'My WWFF':           (11, 2, 'entry'),
        'My POTA':           (12, 2, 'entry'),
        'My BOTA':           (13, 2, 'entry'),
        'My COTA':           (14, 2, 'entry'),
        'My IOTA':           (15, 2, 'entry'),
        'My SOTA':           (16, 2, 'entry'),
        'My WLOTA':          (17, 2, 'entry'),

        'QSL Sent':          (20, 0, 'qsl'),
        'QSL Sent Date':     (20, 2, 'date'),
        'QSL Sent Via':      (20, 4, 'qslvia'),
        'QSL Received':      (21, 0, 'qsl'),
        'QSL Received Date': (21, 2, 'date'),
        'QSL Received Via':  (21, 4, 'qslvia'),

        'LoTW Sent':          (22, 0, 'qsl'),
        'LoTW Sent Date':     (22, 2, 'date'),
        'LoTW Received':      (23, 0, 'qsl'),
        'LoTW Received Date': (23, 2, 'date'),
    }


    entries = {}

    # =====================================
    # QSL + LoTW automatic date + first VIA
    # =====================================
    def qsl_changed(event=None):
        today = datetime.now().strftime("%Y-%m-%d")

        def handle(prefix):
            yes = entries[f"{prefix}"].get()
            date = entries[f"{prefix} Date"]
            via = entries.get(f"{prefix} Via")  # LoTW does not have Via

            if yes == "Yes":
                date.set_date(today)
                if via and via['values']:
                    via.current(0)
            elif yes == "No":
                date.delete(0, tk.END)
                if via:
                    via.set("")

        handle("QSL Sent")
        handle("QSL Received")
        handle("LoTW Sent")
        handle("LoTW Received")

    def make_widget(field, wtype):
        if wtype == 'date':
            widget = DateEntry(form_frame, date_pattern='yyyy-mm-dd')

            def on_pick(event=None, f=field):
                if "Sent Date" in f or "Received Date" in f:
                    entries[f.replace(" Date", "")].set("Yes")

            widget.bind("<<DateEntrySelected>>", on_pick)

        elif wtype == 'band':
            widget = ttk.Combobox(form_frame,
                                  values=list(band_to_frequency.keys()),
                                  state="readonly", width=14)

        elif wtype == 'mode':
            widget = ttk.Combobox(form_frame,
                                  values=mode_options,
                                  state="readonly", width=14)

        elif wtype == 'submode':
            widget = ttk.Combobox(form_frame,
                                  values=submode_options,
                                  state="readonly", width=14)

        elif wtype == 'prop':
            widget = ttk.Combobox(
                form_frame,
                values=[
                    "", "AS", "AUE", "AUR", "BS", "ECH", "EME", "ES", "F2", "FAI",
                    "GWAVE", "INTERNET", "ION", "IRL", "LOS", "MS", "RPT", "RS",
                    "SAT", "TEP", "TR"
                ],
                state="readonly", width=14
            )

        elif wtype == 'qsl':
            widget = ttk.Combobox(form_frame,
                                  values=["Yes", "No"],
                                  state="readonly", width=8)

        elif wtype == 'qslvia':
            widget = ttk.Combobox(form_frame,
                                  values=["Bureau", "Direct", "Electronic", "Manager"],
                                  state="readonly", width=12)

        else:
            var = tk.StringVar()
            widget = tk.Entry(form_frame, textvariable=var, width=20)

        return widget

    # Build GUI
    for field, (row, col, wtype) in FIELD_LAYOUT.items():
        tk.Label(form_frame, text=field).grid(row=row, column=col,
                                              sticky="w", padx=5, pady=3)
        widget = make_widget(field, wtype)
        widget.grid(row=row, column=col + 1, sticky="w", padx=5, pady=3)
        entries[field] = widget

        if ("QSL" in field or "LoTW" in field) and "Date" not in field:
            widget.bind("<<ComboboxSelected>>", qsl_changed)

    # Buttons
    button_frame = tk.Frame(Edit_Window)
    button_frame.pack(pady=10)

    tk.Button(button_frame, text="Save", width=10,
              command=save_changes).pack(side="left", padx=10)

    tk.Button(button_frame, text="Cancel", width=10,
              command=lambda: Edit_Window.withdraw()).pack(side="right", padx=10)

    Edit_Window.resizable(False, False)
    Edit_Window.protocol("WM_DELETE_WINDOW", Edit_Window.withdraw)




def fill_edit_window(qso):
    """Fill the Edit window widgets using FIELD_ALIAS mapping (internal → GUI)."""

    # Ensure defaults for all QSL and LoTW internal fields
    defaults = {
        "QS": "No", "QSD": "", "QSV": "",
        "QR": "No", "QRD": "", "QRV": "",
        "LWS": "No", "LWSD": "",
        "LWR": "No", "LWRD": ""
    }
    for k, v in defaults.items():
        if k not in qso:
            qso[k] = v

    # --- Propagatie normaliseren ---
    # Alles moet Prop zijn. Nooit Propagation.
    if "Propagation" in qso:
        qso["Prop"] = qso["Propagation"]
        del qso["Propagation"]

    # Als Prop nog niet bestaat → leeg veld
    if "Prop" not in qso:
        qso["Prop"] = ""

    # --- Veldvulling ---
    for gui_field, widget in entries.items():
        internal_field = FIELD_ALIAS.get(gui_field, gui_field)
        value = qso.get(internal_field, "")

        # DateEntry
        if isinstance(widget, DateEntry):
            widget.delete(0, tk.END)
            if value:
                try:
                    widget.set_date(value)
                except Exception:
                    pass

        # Combobox (mode, band, submode, qsl, prop, ...)
        elif isinstance(widget, ttk.Combobox):
            try:
                if value in widget["values"] or value == "":
                    widget.set(value)
                else:
                    widget.set(value)     # Onbekende waarde toch tonen
            except:
                widget.set(value)

        # Entry (textvak)
        else:
            widget.delete(0, tk.END)
            widget.insert(0, value)



def save_changes():
    """Save QSO edits using FIELD_ALIAS mapping (GUI → internal)."""
    global current_qso, qso_index, entries

    # --- Validate date/time ---
    try:
        datetime.strptime(entries['Date'].get().strip(), '%Y-%m-%d')
    except ValueError:
        messagebox.showerror("Invalid Date", "Date must be in format YYYY-MM-DD.", parent=Logbook_Window)
        return

    try:
        datetime.strptime(entries['Time'].get().strip(), '%H:%M:%S')
    except ValueError:
        messagebox.showerror("Invalid Time", "Time must be in format HH:MM:SS", parent=Logbook_Window)
        return

    # --- Copy GUI values to internal dictionary ---
    for gui_field, widget in entries.items():
        internal_key = FIELD_ALIAS.get(gui_field, gui_field)
        try:
            val = widget.get().strip()
        except Exception:
            val = ""

        # -- DXCC special handling (must be integer or empty) --
        if internal_key == "dxcc":
            if val == "":
                # Empty DXCC → remove if present
                current_qso.pop("dxcc", None)
            else:
                if not val.isdigit():
                    messagebox.showerror("Invalid DXCC", "DXCC must be a numeric entity code.", parent=Logbook_Window)
                    return
                current_qso["dxcc"] = int(val)
            continue

        current_qso[internal_key] = val

    # --- Remove empty optional reference fields ---
    OPTIONAL_REF_FIELDS = [
        "WWFF", "POTA", "BOTA", "COTA", "IOTA", "SOTA", "WLOTA", "Satellite",
        "My WWFF", "My POTA", "My BOTA", "My COTA",
        "My IOTA", "My SOTA", "My WLOTA"
    ]

    for key in OPTIONAL_REF_FIELDS:
        if key in current_qso and not current_qso[key].strip():
            current_qso.pop(key, None)

    # --- Normalize QSL & LoTW consistency ---
    now_str = datetime.now().strftime("%Y-%m-%d")

    for yes_key, date_key, via_key in (
        ("QS", "QSD", "QSV"),
        ("QR", "QRD", "QRV")
    ):
        if current_qso.get(yes_key) == "Yes":
            if not current_qso.get(date_key):
                current_qso[date_key] = now_str
        elif current_qso.get(yes_key) == "No":
            current_qso[date_key] = ""
            current_qso[via_key] = ""

    for yes_key, date_key in (
        ("LWS", "LWSD"),
        ("LWR", "LWRD")
    ):
        if current_qso.get(yes_key) == "Yes" and not current_qso.get(date_key):
            current_qso[date_key] = now_str
        elif current_qso.get(yes_key) == "No":
            current_qso[date_key] = ""

    # --- Normalize text fields ---
    for f in ("Callsign", "My Callsign", "My Operator"):
        if f in current_qso:
            current_qso[f] = current_qso[f].upper()

    for f in ("Locator", "My Locator"):
        if f in current_qso:
            current_qso[f] = current_qso[f].upper()

    if "Band" in current_qso:
        current_qso["Band"] = current_qso["Band"].lower()

    if "Mode" in current_qso:
        current_qso["Mode"] = current_qso["Mode"].upper()

    if "Submode" in current_qso:
        current_qso["Submode"] = current_qso["Submode"].upper()

    # --- Rebuild DateTime ---
    current_qso["Date"] = entries["Date"].get().strip()
    current_qso["Time"] = entries["Time"].get().strip()

    try:
        current_qso["DateTime"] = datetime.strptime(
            f"{current_qso['Date']} {current_qso['Time']}",
            "%Y-%m-%d %H:%M:%S"
        )
    except Exception:
        current_qso.pop("DateTime", None)

    # --- Save ---
    qso_lines[qso_index] = current_qso
    save_to_json()
    load_json_content_to_tree()
    update_worked_before_tree()
    Edit_Window.withdraw()




def edit_qso(event=None):
    """Open the edit window and load QSO data using FIELD_ALIAS."""
    global current_qso, qso_index

    selected_item = tree.selection()
    if not selected_item:
        messagebox.showwarning("Edit QSO", "No QSO selected.", parent=Logbook_Window)
        return
    
    item_id = selected_item[0]
    qso_index = tree_to_log_index.get(item_id)

    if qso_index is None:
        messagebox.showerror("Error", "Could not find matching QSO index.", parent=Logbook_Window)
        return

    if qso_index < 0 or qso_index >= len(qso_lines):
        messagebox.showerror("Error", "Invalid QSO-index.", parent=Logbook_Window)
        return

    current_qso = qso_lines[qso_index]
    if Edit_Window is None or not Edit_Window.winfo_exists():
        build_edit_window()
    fill_edit_window(current_qso)
    Edit_Window.deiconify()
    Edit_Window.lift()


#########################################################################################
#    _   ___ ___ ___   ___ __  __ ___  ___  ___ _____ 
#   /_\ |   \_ _| __| |_ _|  \/  | _ \/ _ \| _ \_   _|
#  / _ \| |) | || _|   | || |\/| |  _/ (_) |   / | |  
# /_/ \_\___/___|_|   |___|_|  |_|_|  \___/|_|_\ |_|  
#                                                     
#########################################################################################

def extract_field(record, field_name):
    """Extract a field's value from the ADIF record in the format <FIELD_NAME:length>value."""
    pattern = rf"<{field_name}:(\d+)>([^<]*)"
    match = re.search(pattern, record, re.IGNORECASE)
    if match:
        return match.group(2).strip()
    return ""


def strip_leading_zeros(val):
    """
    Remove leading zeros from numeric strings.
    Keeps '0' as '0', empty stays empty.
    """
    s = str(val).strip()
    if not s:
        return ""
    if s.isdigit():
        return str(int(s))
    return s


def import_format_date(date_str):
    """Return YYYY-MM-DD or '' for invalid/empty ADIF dates."""
    if not date_str:
        return ""

    date_str = date_str.strip()

    # ADIF unknown/empty date values → ignore
    if date_str in ("00000000", "0000000", "000000", "0000"):
        return ""

    # Must be numeric
    if not date_str.isdigit():
        return ""

    # Must be exactly 8 chars for YYYYMMDD
    if len(date_str) != 8:
        return ""

    # Safe parse
    try:
        return datetime.strptime(date_str, "%Y%m%d").strftime("%Y-%m-%d")
    except Exception:
        return ""



def import_format_time(time_str):
    """Return HH:MM:SS or '' for invalid ADIF time."""
    if not time_str:
        return ""

    time_str = time_str.strip()

    # ADIF uses 0000 or 000000 for empty times
    if time_str in ("0000", "000000"):
        return ""

    # Keep only digits
    if not time_str.isdigit():
        return ""

    # ADIF times can be 4 (HHMM) or 6 (HHMMSS)
    if len(time_str) == 4:
        time_str = time_str + "00"   # Append SS = 00

    # Now it must be 6 digits
    if len(time_str) != 6:
        return ""

    # Safe parse
    try:
        return datetime.strptime(time_str, "%H%M%S").strftime("%H:%M:%S")
    except Exception:
        return ""



def validate_adif_date(raw_date: str) -> bool:
    raw_date = (raw_date or "").strip()
    if len(raw_date) != 8 or not raw_date.isdigit():
        return False
    try:
        datetime.strptime(raw_date, "%Y%m%d")
        return True
    except ValueError:
        return False




def find_best_dxcc(call):
    cs = call.upper()

    for n in range(len(cs), 0, -1):
        pf = cs[:n]
        if pf in dxcc_prefix_map:
            entry = dxcc_prefix_map[pf]
            dxcc_code = prefix_to_dxcc.get(pf)
            return entry, dxcc_code, pf

    return None, None, ""



def extract_n1mm_exchange(rec):
    matches = re.findall(
        r"<APP_N1MM_EXCHANGE\d*:\d+>([^<]+)",
        rec,
        flags=re.IGNORECASE
    )
    return matches[0].strip() if matches else ""


def ask_for_value(title, text, default=""):
    dlg = tk.Toplevel(Logbook_Window)
    dlg.title(title)
    dlg.transient(Logbook_Window)
    dlg.grab_set()
    center_window_over(Logbook_Window, dlg, 380, 150)

    tk.Label(dlg, text=text, wraplength=350).pack(padx=10, pady=(10, 5))
    entry = tk.Entry(dlg)
    entry.pack(padx=10)
    entry.insert(0, default)
    entry.focus_set()

    result = {"value": None}

    def ok():
        result["value"] = entry.get().strip()
        dlg.destroy()

    tk.Button(dlg, text="OK", width=10, command=ok).pack(pady=10)
    dlg.wait_window()
    return result["value"]



SRX_FALLBACK_FIELDS = [
    {
        "key": "srx",
        "label": "SRX (Receive Exchange)",
        "detect": lambda recs: any(
            re.search(r"<SRX\s*:\s*\d+>", r, re.IGNORECASE)
            for r in recs
        ),
        "extract": lambda rec: extract_field(rec, "srx"),
    },
    {
        "key": "n1mm",
        "label": "APP_N1MM_EXCHANGE",
        "detect": lambda recs: any(
            re.search(r"<APP_N1MM_EXCHANGE\d*:", r, re.IGNORECASE)
            for r in recs
        ),
        "extract": lambda rec: extract_n1mm_exchange(rec),
    },
    {
        "key": "cqz",
        "label": "CQ Zone (CQZ)",
        "detect": lambda recs: any(
            re.search(r"<CQZ\s*:\s*\d+>", r, re.IGNORECASE)
            for r in recs
        ),
        "extract": lambda rec: extract_field(rec, "cqz"),
    },
    {
        "key": "ituz",
        "label": "ITU Zone (ITUZ)",
        "detect": lambda recs: any(
            re.search(r"<ITUZ\s*:\s*\d+>", r, re.IGNORECASE)
            for r in recs
        ),
        "extract": lambda rec: extract_field(rec, "ituz"),
    },
]


def ask_stx_source():
    dlg = tk.Toplevel(Logbook_Window)
    dlg.title("Send Exchange (STX)")
    dlg.transient(Logbook_Window)
    dlg.grab_set()

    tk.Label(
        dlg,
        text=(
            "This is a contest log.\n\n"
            "Select how the Send Exchange (STX) should be determined:"
        ),
        justify="left",
        wraplength=420
    ).pack(padx=12, pady=(12, 8))

    choice = tk.StringVar(value="stx")

    options = [
        ("Use STX from ADIF", "stx"),
        ("Use fixed / manual value (e.g. CQ Zone, ITU Zone, serial, section)", "manual"),
    ]

    for text, val in options:
        tk.Radiobutton(dlg, text=text, variable=choice, value=val).pack(anchor="w", padx=20)

    manual_frame = tk.Frame(dlg)
    manual_frame.pack(padx=20, pady=(6, 0), anchor="w")

    tk.Label(manual_frame, text="Manual STX value:").pack(side="left")
    manual_entry = tk.Entry(manual_frame, width=12)
    manual_entry.pack(side="left", padx=6)

    hint = tk.Label(
        dlg,
        text="Example values: 14, 27, 001, PA, NH, DX",
        font=("Segoe UI", 8),
        fg="gray"
    )
    hint.pack(anchor="w", padx=42, pady=(2, 0))

    result = {"mode": None, "value": ""}

    def ok():
        result["mode"] = choice.get()
        result["value"] = manual_entry.get().strip()
        dlg.destroy()

    btns = tk.Frame(dlg)
    btns.pack(pady=12)

    tk.Button(btns, text="OK", width=10, command=ok).pack(side="left", padx=5)
    tk.Button(btns, text="Cancel", width=10, command=dlg.destroy).pack(side="left", padx=5)

    dlg.update_idletasks()
    center_window_over(Logbook_Window, dlg, dlg.winfo_width(), dlg.winfo_height())
    dlg.wait_window()

    return result


def ask_srx_source(available_sources, default_choice="none"):

    dlg = tk.Toplevel(Logbook_Window)
    dlg.title("Receive Exchange (SRX) selection")
    dlg.transient(Logbook_Window)
    dlg.grab_set()

    # ---------- CONTENT ----------
    tk.Label(
        dlg,
        text=(
            "This ADIF file contains one or more fields that can be used as Receive Exchange (SRX).\n\n"
            "SRX from the ADIF file will always be used when it is present.\n"
            "The selection below is only used when SRX is empty."
        ),
        justify="left",
        wraplength=420
    ).pack(padx=12, pady=(12, 8))

    choice = tk.StringVar(value=default_choice)

    for src in available_sources:
        tk.Radiobutton(dlg, text=src["label"], variable=choice, value=src["key"]).pack(anchor="w", padx=20)

    tk.Radiobutton(dlg, text="Do not use a fallback (leave SRX empty when missing)", variable=choice, value="none").pack(anchor="w", padx=20, pady=(6, 0))

    btns = tk.Frame(dlg)
    btns.pack(pady=12)

    result = {"value": None}

    def ok():
        result["value"] = choice.get()
        dlg.destroy()

    tk.Button(btns, text="OK", width=10, command=ok).pack(side="left", padx=5)
    tk.Button(btns, text="Cancel", width=10, command=dlg.destroy).pack(side="left", padx=5)

    # ---------- CENTER AFTER CONTENT ----------
    dlg.update_idletasks()
    center_window_over(Logbook_Window, dlg, dlg.winfo_width(), dlg.winfo_height())

    dlg.wait_window()
    return result["value"]

def import_adif():
    log.info("IMPORT_ADIF called")

    global CURRENT_JSON_FILE, qso_lines

    if not CURRENT_JSON_FILE:
        messagebox.showerror("Error", "No logbook loaded.", parent=Logbook_Window)
        return

    adif_file = filedialog.askopenfilename(
        title="Select ADIF File",
        filetypes=[("ADIF files", "*.adi"), ("All files", "*.*")]
    )
    if not adif_file:
        return

    # -------- LOG TYPE SELECTION --------
    is_contest_log = messagebox.askyesno(
        "Import type",
        "Is this a CONTEST log?\n\n"
        "Yes  = Contest log (all contest checks will be applied)\n"
        "No   = Normal log (import without contest-specific checks)",
        parent=Logbook_Window
    )

    # ---------------- UI ----------------
    wnd = tk.Toplevel(Logbook_Window)
    wnd.title("Importing ADIF")
    center_window_over(Logbook_Window, wnd, 460, 180)
    wnd.transient(Logbook_Window)

    tk.Label(wnd, text="Importing QSOs...").pack(pady=(6, 2))
    progress_counter = tk.Label(wnd, text="")
    progress_counter.pack()

    progress = ttk.Progressbar(wnd, orient="horizontal", length=380, mode="determinate")
    progress.pack(pady=4)

    dup_mode = tk.StringVar(value="seconds")

    dup_label = tk.Label(
        wnd,
        text="Duplicate check based on Callsign + Date + Time:",
        font=("Segoe UI", 9)
    )
    dup_label.pack(pady=(6, 2))

    mode_frame = tk.Frame(wnd)
    mode_frame.pack(pady=(0, 6))

    tk.Radiobutton(
        mode_frame,
        text="Compare time by Minutes",
        variable=dup_mode,
        value="minutes"
    ).pack(side="left", padx=5)

    tk.Radiobutton(
        mode_frame,
        text="Compare time by Seconds",
        variable=dup_mode,
        value="seconds"
    ).pack(side="left", padx=5)


    # -------- LOAD ADIF --------
    try:
        with open(adif_file, "r", encoding="utf-8", errors="ignore") as f:
            content = f.read()
    except Exception as e:
        messagebox.showerror("Error", f"Failed loading ADIF:\n{e}")
        wnd.destroy()
        return

    records = [r for r in re.split(r"<eor>", content, flags=re.IGNORECASE) if r.strip()]

    # -------- CONTEST_ID detectie --------
    use_contest_id_in_comment = False
    contest_id_value = None

    if is_contest_log:
        contest_ids = {
            extract_field(r, "contest_id")
            for r in records
            if extract_field(r, "contest_id")
        }
        contest_ids.discard("")

        if contest_ids:
            contest_id_value = ", ".join(sorted(contest_ids))
            use_contest_id_in_comment = messagebox.askyesno(
                "Contest detected",
                f"CONTEST_ID found in ADIF:\n\n{contest_id_value}\n\n"
                "Do you want to add this value to the Comment field?",
                parent=Logbook_Window
            )

    # -------- SRX fallback (contest only) --------
    selected_srx_source = None

    if is_contest_log:
        available_srx_sources = [
            fb for fb in SRX_FALLBACK_FIELDS
            if fb["detect"](records)
        ]

        if available_srx_sources:
            default_choice = "srx" if any(s["key"] == "srx" for s in available_srx_sources) else "none"
            selected_srx_source = ask_srx_source(available_srx_sources, default_choice=default_choice)
            if selected_srx_source is None:
                wnd.destroy()
                return

    # -------- STX selection (contest only) --------
    stx_mode = None
    stx_manual_value = ""

    if is_contest_log:
        stx_result = ask_stx_source()
        if not stx_result or not stx_result.get("mode"):
            wnd.destroy()
            return

        stx_mode = stx_result["mode"]
        stx_manual_value = stx_result.get("value", "")

    # -------- Duplicate dialog --------
    def ask_duplicates_action(count):
        if count == 0:
            return "add"

        dlg = tk.Toplevel(Logbook_Window)
        dlg.title("Duplicates found")
        dlg.transient(Logbook_Window)
        dlg.grab_set()
        center_window_over(Logbook_Window, dlg, 320, 130)

        tk.Label(
            dlg,
            text=f"{count} duplicates found.\nWhat do you want to do?",
            font=("Segoe UI", 10),
            justify="center"
        ).pack(padx=12, pady=(12, 10))

        choice = {"value": None}

        def set_choice(val):
            choice["value"] = val
            dlg.destroy()

        btns = tk.Frame(dlg)
        btns.pack(pady=6)

        tk.Button(btns, text="Add", width=10, command=lambda: set_choice("add")).pack(side="left", padx=4)
        tk.Button(btns, text="Overwrite", width=10, command=lambda: set_choice("overwrite")).pack(side="left", padx=4)
        tk.Button(btns, text="Ignore", width=10, command=lambda: set_choice("ignore")).pack(side="left", padx=4)

        dlg.wait_window()
        return choice["value"]

    # ================= IMPORT THREAD =================
    def do_import():
        with open(CURRENT_JSON_FILE, "r", encoding="utf-8") as f:
            logbook_data = json.load(f)

        station = logbook_data.get("Station", {})
        station_callsign = station.get("Callsign", "").upper()
        station_operator = station.get("Operator", "")

        # -------- MY CALLSIGN --------
        adif_callsigns = {
            extract_field(r, "station_callsign").upper()
            for r in records
            if extract_field(r, "station_callsign")
        }
        adif_callsigns.discard("")

        if adif_callsigns:
            adif_call = next(iter(adif_callsigns))
            if adif_call != station_callsign:
                choice = messagebox.askyesnocancel(
                    "Station Callsign mismatch",
                    f"ADIF: {adif_call}\nLogbook: {station_callsign}\n\nUse ADIF callsign?",
                    parent=Logbook_Window
                )
                if choice is None:
                    wnd.destroy()
                    return
                my_callsign = adif_call if choice else ask_for_value("My Callsign", "Enter My Callsign:", station_callsign)
            else:
                my_callsign = adif_call
        else:
            my_callsign = ask_for_value("My Callsign missing", "Enter My Callsign(aka Station Callsign):", station_callsign)

        if not my_callsign:
            wnd.destroy()
            return

        # -------- MY OPERATOR --------
        adif_ops = {
            extract_field(r, "operator")
            for r in records
            if extract_field(r, "operator")
        }
        adif_ops.discard("")
        my_operator = next(iter(adif_ops)) if adif_ops else ask_for_value(
            "My Operator",
            "Enter My Operator(aka Station Operator):",
            station_operator
        )

        if not my_operator:
            wnd.destroy()
            return

        existing = logbook_data.setdefault("Logbook", [])
        index = {}

        for q in existing:
            t = q.get("Time", "")
            t = t[:5] if dup_mode.get() == "minutes" else t
            index[f"{q.get('Callsign','').upper()}_{q.get('Date','')}_{t}"] = q

        added = []
        duplicates = []

        count_new = count_added = count_overwritten = count_ignored = 0
        progress["maximum"] = len(records)

        # ================= MAIN LOOP =================
        for i, rec in enumerate(records, 1):
            call = extract_field(rec, "call").upper()
            if not call:
                continue

            raw_date = extract_field(rec, "qso_date").strip()
            if not validate_adif_date(raw_date):
                messagebox.showerror(
                    "Invalid QSO_DATE",
                    f"Invalid QSO_DATE detected:\n\n{raw_date}",
                    parent=Logbook_Window
                )
                wnd.destroy()
                return

            date = import_format_date(raw_date)
            time = import_format_time(extract_field(rec, "time_off"))
            t = time[:5] if dup_mode.get() == "minutes" else time

            srx = strip_leading_zeros(extract_field(rec, "srx"))
            stx_adif = strip_leading_zeros(extract_field(rec, "stx"))
            stx = stx_adif

            if not srx and selected_srx_source and selected_srx_source != "none":
                for fb in SRX_FALLBACK_FIELDS:
                    if fb["key"] == selected_srx_source:
                        val = strip_leading_zeros(fb["extract"](rec))
                        if val:
                            srx = val
                        break

            comment = extract_field(rec, "comment").strip()
            if use_contest_id_in_comment and contest_id_value:
                comment = f"{comment} | {contest_id_value}" if comment else contest_id_value

            rec_operator = extract_field(rec, "operator")
            my_operator_qso = rec_operator if rec_operator else my_operator

            adif_cqz = strip_leading_zeros(extract_field(rec, "cqz"))
            adif_ituz = strip_leading_zeros(extract_field(rec, "ituz"))

            entry, dxcc, prefix = find_best_dxcc(call)

            if is_contest_log:
                if stx_mode == "manual":
                    stx = stx_manual_value
                else:
                    stx = stx_adif


            q = {
                "Date": date,
                "Time": time,
                "Callsign": call,
                "Band": extract_field(rec, "band").lower(),
                "Mode": extract_field(rec, "mode"),
                "Frequency": extract_field(rec, "freq"),
                "Sent": extract_field(rec, "rst_sent"),
                "Received": extract_field(rec, "rst_rcvd"),
                "STX": stx,
                "SRX": srx,
                "Comment": comment,
                "My Callsign": my_callsign,
                "My Operator": my_operator_qso,
            }

            if entry:
                q["Country"] = entry.name
                q["CQZ"] = adif_cqz or entry.cq_zone
                q["ITUZ"] = adif_ituz or entry.itu_zone
            if dxcc:
                q["DXCC"] = dxcc

            q = normalize_qso_fields(q)

            key = f"{call}_{date}_{t}"
            if key in index:
                duplicates.append((key, q))
            else:
                added.append(q)
                count_new += 1

            progress["value"] = i
            progress_counter.config(text=f"{i}/{len(records)}")
            wnd.update_idletasks()

        action = ask_duplicates_action(len(duplicates))

        if action == "overwrite":
            for k, new_qso in duplicates:
                old_qso = index[k]
                preserved = {"DateTime": old_qso.get("DateTime")}
                old_qso.clear()
                old_qso.update(new_qso)
                old_qso.update(preserved)
                count_overwritten += 1
        elif action == "add":
            for _, q in duplicates:
                existing.append(q)
                count_added += 1
        elif action == "ignore":
            count_ignored = len(duplicates)

        existing.extend(added)

        for i, q in enumerate(existing):
            existing[i] = normalize_qso_fields(q)

        atomic_write_json(CURRENT_JSON_FILE, logbook_data)
        qso_lines[:] = existing

        root.after(0, lambda: (
            load_json_content_to_tree(),
            update_worked_before_tree(),
            messagebox.showinfo(
                "Import ADIF",
                f"QSO Import completed\n\n"
                f"New QSOs: {count_new}\n"
                f"Added (duplicates): {count_added}\n"
                f"Overwritten: {count_overwritten}\n"
                f"Ignored: {count_ignored}",
                parent=Logbook_Window
            ),
            wnd.destroy()
        ))

    def start_import():
        start_btn.config(state="disabled")
        threading.Thread(target=do_import, daemon=True).start()

    start_btn = tk.Button(wnd, text="Start Import", width=20, command=start_import)
    start_btn.pack(pady=(8, 10))




#########################################################################################
#    _   ___ ___ ___   _____  _____  ___  ___ _____ 
#   /_\ |   \_ _| __| | __\ \/ / _ \/ _ \| _ \_   _|
#  / _ \| |) | || _|  | _| >  <|  _/ (_) |   / | |  
# /_/ \_\___/___|_|   |___/_/\_\_|  \___/|_|_\ |_|  
#                                                   
#########################################################################################

# Function to escape invalid characters
def escape_invalid_characters(text):
    # Replace all non-ASCII characters with '?'
    return re.sub(r'[\u0080-\uFFFF]', '?', str(text))


# Function to ask the user to select the log type (POTA, WWFF, General)
def get_log_type():
    log_type_window = tk.Toplevel(Logbook_Window)
    log_type_window.title("Select Log Type")

    selected_log_type = tk.StringVar(value="")

    def set_log_type(log_type):
        selected_log_type.set(log_type)
        log_type_window.destroy()

    tk.Label(log_type_window, text="How would you like to export?\nSIG is used for BOTA/COTA/WLOTA").pack(pady=10)
    tk.Button(log_type_window, text="BOTA", command=lambda: set_log_type("BOTA")).pack(pady=5)
    tk.Button(log_type_window, text="WLOTA", command=lambda: set_log_type("WLOTA")).pack(pady=5)
    tk.Button(log_type_window, text="Normal", command=lambda: set_log_type("Normal")).pack(pady=5)

    parent_x = Logbook_Window.winfo_rootx()
    parent_y = Logbook_Window.winfo_rooty()
    parent_width = Logbook_Window.winfo_width()

    window_width = 200
    window_height = 300
    new_x = parent_x + (parent_width // 2) - (window_width // 2)
    new_y = parent_y

    log_type_window.geometry(f"{window_width}x{window_height}+{new_x}+{new_y}")
    log_type_window.grab_set()
    log_type_window.wait_window()
    return selected_log_type.get()


#########################################################################################
#   Export logbook (internal short names only) to ADIF 3.1.0
#########################################################################################
def export_to_adif():
    log.info("EXPORT_ADIF called")

    global CURRENT_JSON_FILE, qso_lines

    if not CURRENT_JSON_FILE or not qso_lines:
        log.warning("EXPORT_ADIF aborted: no logbook or empty log")
        messagebox.showwarning(
            "Warning",
            "No logbook file loaded or no QSO entries to export!",
            parent=Logbook_Window
        )
        return

    log.info("EXPORT_ADIF requesting log type selection")
    log_type = get_log_type()
    if not log_type:
        log.info("EXPORT_ADIF cancelled by user (no log type selected)")
        return

    try:
        with open(CURRENT_JSON_FILE, "r", encoding="utf-8") as f:
            log.debug("Reading Station info for ADIF export")
            json_data = json.load(f)
            station_info = json_data.get("Station", {})
    except Exception as e:
        log.exception("EXPORT_ADIF failed reading Station info")
        messagebox.showerror("Error", f"Failed to read Station info: {e}", parent=Logbook_Window)
        return

    adif_file = filedialog.asksaveasfilename(
        defaultextension=".adi",
        filetypes=[("ADIF files", "*.adi")]
    )
    if not adif_file:
        log.info("EXPORT_ADIF cancelled by user (no filename selected)")
        return

    log.info(f"EXPORT_ADIF exporting to: {adif_file}")

    try:
        with open(adif_file, "w", encoding="utf-8") as file:

            file.write("Generated-by: MiniBook\n")
            file.write("<ADIF_VER:5>3.1.0\n")
            file.write("<EOH>\n")

            via_map = {"Bureau": "B", "Direct": "D", "Electronic": "E", "Manager": "M"}

            count = 0

            for qso in qso_lines:
                try:
                    count += 1

                    callsign  = escape_invalid_characters(qso.get("Callsign", ""))
                    mycall    = escape_invalid_characters(qso.get("My Callsign", ""))
                    operator  = escape_invalid_characters(qso.get("My Operator", ""))
                    mode      = escape_invalid_characters(qso.get("Mode", ""))
                    submode   = escape_invalid_characters(qso.get("Submode", ""))
                    prop      = escape_invalid_characters(qso.get("Prop", ""))
                    band      = escape_invalid_characters(qso.get("Band", ""))
                    freq      = escape_invalid_characters(qso.get("Frequency", ""))
                    date      = export_format_date(qso.get("Date", ""))
                    time      = export_format_time(qso.get("Time", ""))
                    sent      = escape_invalid_characters(qso.get("Sent", ""))
                    received  = escape_invalid_characters(qso.get("Received", ""))
                    stx       = escape_invalid_characters(qso.get("STX", ""))
                    srx       = escape_invalid_characters(qso.get("SRX", ""))
                    locator   = escape_invalid_characters(qso.get("Locator", ""))
                    comment   = escape_invalid_characters(qso.get("Comment", ""))
                    satellite = escape_invalid_characters(qso.get("Satellite", ""))
                    name      = escape_invalid_characters(qso.get("Name", ""))
                    country   = escape_invalid_characters(qso.get("Country", ""))

                    dxcc = str(qso.get("dxcc", "")).strip()
                    cqz  = str(qso.get("CQZ", "")).strip()
                    ituz = str(qso.get("ITUZ", "")).strip()


                    qsl_sent_flag = qso.get("QS", "No")
                    qsl_rcvd_flag = qso.get("QR", "No")
                    qsl_sent_adif = "Y" if qsl_sent_flag.lower() == "yes" else "N"
                    qsl_rcvd_adif = "Y" if qsl_rcvd_flag.lower() == "yes" else "N"
                    qsl_sent_date = qso.get("QSD", "").replace("-", "")
                    qsl_rcvd_date = qso.get("QRD", "").replace("-", "")
                    qsl_sent_via  = via_map.get(qso.get("QSV", ""), "")
                    qsl_rcvd_via  = via_map.get(qso.get("QRV", ""), "")

                    lotw_sent_flag = qso.get("LWS", "No")
                    lotw_rcvd_flag = qso.get("LWR", "No")
                    lotw_sent_adif = "Y" if lotw_sent_flag.lower() == "yes" else "N"
                    lotw_rcvd_adif = "Y" if lotw_rcvd_flag.lower() == "yes" else "N"
                    lotw_sent_date = qso.get("LWSD", "").replace("-", "")
                    lotw_rcvd_date = qso.get("LWRD", "").replace("-", "")

                    adif = []

                    # --- Basic QSO data ---
                    adif.append(f"<call:{len(callsign)}>{callsign}")
                    adif.append(f"<qso_date:{len(date)}>{date}")
                    adif.append(f"<time_on:{len(time)}>{time}")
                    adif.append(f"<band:{len(band)}>{band}")
                    if freq:
                        adif.append(f"<freq:{len(freq)}>{freq}")
                    adif.append(f"<mode:{len(mode)}>{mode}")

                    if submode:
                        adif.append(f"<submode:{len(submode)}>{submode}")
                    if prop:
                        adif.append(f"<prop_mode:{len(prop)}>{prop}")

                    adif.append(f"<rst_sent:{len(sent)}>{sent}")
                    adif.append(f"<rst_rcvd:{len(received)}>{received}")

                    if stx:
                        adif.append(f"<stx:{len(stx)}>{stx}")
                    if srx:
                        adif.append(f"<srx:{len(srx)}>{srx}")
                    if locator:
                        adif.append(f"<gridsquare:{len(locator)}>{locator}")
                    if name:
                        adif.append(f"<name:{len(name)}>{name}")
                    if comment:
                        adif.append(f"<comment:{len(comment)}>{comment}")
                    if satellite:
                        adif.append(f"<sat_name:{len(satellite)}>{satellite}")
                    if operator:
                        adif.append(f"<operator:{len(operator)}>{operator}")
                    if mycall:
                        adif.append(f"<station_callsign:{len(mycall)}>{mycall}")
                    if country:
                        adif.append(f"<country:{len(country)}>{country}")

                    if dxcc:
                        adif.append(f"<dxcc:{len(dxcc)}>{dxcc}")

                    if cqz:
                        adif.append(f"<cqz:{len(cqz)}>{cqz}")

                    if ituz:
                        adif.append(f"<ituz:{len(ituz)}>{ituz}")


                    adif.append(f"<qsl_sent:1>{qsl_sent_adif}")
                    adif.append(f"<qsl_rcvd:1>{qsl_rcvd_adif}")

                    if qsl_sent_date:
                        adif.append(f"<qslsdate:{len(qsl_sent_date)}>{qsl_sent_date}")
                    if qsl_rcvd_date:
                        adif.append(f"<qslrdate:{len(qsl_rcvd_date)}>{qsl_rcvd_date}")
                    if qsl_sent_via:
                        adif.append(f"<qsl_sent_via:1>{qsl_sent_via}")
                    if qsl_rcvd_via:
                        adif.append(f"<qsl_rcvd_via:1>{qsl_rcvd_via}")

                    adif.append(f"<lotw_qsl_sent:1>{lotw_sent_adif}")
                    adif.append(f"<lotw_qsl_rcvd:1>{lotw_rcvd_adif}")
                    if lotw_sent_date:
                        adif.append(f"<lotw_qslsdate:{len(lotw_sent_date)}>{lotw_sent_date}")
                    if lotw_rcvd_date:
                        adif.append(f"<lotw_qslrdate:{len(lotw_rcvd_date)}>{lotw_rcvd_date}")

                    file.write(" ".join(adif) + " <EOR>\n")

                except Exception as rec_e:
                    log.exception(f"EXPORT_ADIF failed for QSO {qso.get('Callsign','?')}: {rec_e}")
                    continue

            log.info(f"EXPORT_ADIF success: exported {count} QSOs")

        messagebox.showinfo("Success", "Exported to ADIF format successfully!", parent=Logbook_Window)

    except Exception as e:
        log.exception("EXPORT_ADIF failed")
        messagebox.showerror("Error", f"Failed to export to ADIF: {e}", parent=Logbook_Window)



# Function to open the export folder
def open_export_folder(folder_path):
    try:
        if platform.system() == "Windows":
            subprocess.Popen(f'explorer "{folder_path}"')
        elif platform.system() == "Darwin":  # macOS
            subprocess.Popen(["open", folder_path])
        else:  # Linux
            subprocess.Popen(["xdg-open", folder_path])
    except Exception as e:
        messagebox.showerror("Error", f"Failed to open folder: {e}", parent=Logbook_Window)


# Function to convert date and time to ADIF format
def export_format_date(date_str):
    try:
        date_obj = datetime.strptime(date_str, '%Y-%m-%d')  # Adjust format as necessary
        return date_obj.strftime('%Y%m%d')  # Convert to 'YYYYMMDD'
    except ValueError:
        return ''  # Return empty string if parsing fails

def export_format_time(time_str):
    try:
        time_obj = datetime.strptime(time_str, '%H:%M:%S')  # Adjust format as necessary
        return time_obj.strftime('%H%M%S')  # Convert to 'HHMMSS'
    except ValueError:
        return ''  # Return empty string if parsing fails



def log_qso():
    """
    Validate input, save the QSO to the local logbook,
    update the GUI, and optionally upload to QRZ
    in a background thread.
    """
    global qso_lines, entityCode

    if not CURRENT_JSON_FILE:
        messagebox.showwarning("Warning", "No logbook file loaded!")
        return

    # --- Validate date/time format ---
    date_str = qso_date_var.get().strip()
    time_str = qso_time_var.get().strip()

    if not re.fullmatch(r"\d{4}-\d{2}-\d{2}", date_str):
        messagebox.showerror("Date Format Error", "Date must be YYYY-MM-DD")
        return
    if not re.fullmatch(r"\d{2}:\d{2}:\d{2}", time_str):
        messagebox.showerror("Time Format Error", "Time must be HH:MM:SS")
        return

    try:
        datetime.strptime(date_str, "%Y-%m-%d")
        datetime.strptime(time_str, "%H:%M:%S")
    except ValueError:
        messagebox.showerror("Invalid Date/Time", "Invalid date or time.")
        return

    # --- Basic callsign validation ---
    callsign = qso_callsign_var.get().strip().upper()
    if not callsign:
        messagebox.showwarning("Warning", "Callsign cannot be empty!")
        reset_fields()
        return

    # --- Build base QSO record ---
    qso_entry = {
        "Date": date_str,
        "Time": time_str,
        "Callsign": callsign,
        "Name": qso_name_var.get().strip(),

        "My Callsign": station_callsign_var.get().strip().upper(),
        "My Operator": station_operator_var.get().strip().upper(),
        "My Locator": station_locator_var.get().strip().upper(),
        "My Location": station_location_var.get().strip(),

        "Country": qso_country_var.get().strip(),
        "Continent": qso_continent_var.get().strip(),
        "dxcc": str(entityCode).strip() if entityCode else "",
        "CQZ":  qso_cq_zone_var.get(),
        "ITUZ": qso_itu_zone_var.get(),

        "Sent": qso_rst_sent_var.get().strip(),
        "Received": qso_rst_received_var.get().strip(),

        "Mode": qso_mode_var.get().strip(),
        "Submode": qso_submode_var.get().strip(),
        "Prop": qso_prop_mode_var.get().strip(),
        "Band": qso_band_var.get().strip(),
        "Frequency": "",
        "Locator": qso_locator_var.get().strip().upper(),
        "Comment": qso_comment_var.get().strip(),

        "QS": "No",
        "QSD": "",
        "QR": "No",
        "QRD": "",
        "LWS": "No",
        "LWSD": "",
        "LWR": "No",
        "LWRD": "",

        "STX": "",
        "SRX": "",
    }

    # --- Add reference fields only when filled ---
    ref_fields = {
        "WWFF": wwff_var.get().strip(),
        "POTA": pota_var.get().strip(),
        "BOTA": bota_var.get().strip(),
        "COTA": cota_var.get().strip(),
        "IOTA": iota_var.get().strip(),
        "SOTA": sota_var.get().strip(),
        "WLOTA": wlota_var.get().strip(),
        "Satellite": qso_satellite_var.get().strip(),

        "My WWFF": station_wwff_var.get().strip(),
        "My POTA": station_pota_var.get().strip(),
        "My BOTA": station_bota_var.get().strip(),
        "My COTA": station_cota_var.get().strip(),
        "My IOTA": station_iota_var.get().strip(),
        "My SOTA": station_sota_var.get().strip(),
        "My WLOTA": station_wlota_var.get().strip(),
    }

    for key, value in ref_fields.items():
        if value:
            qso_entry[key] = value

    # --- Contest serials ---
    if use_serial_var.get():
        qso_entry["STX"] = qso_sent_exchange_var.get().strip()
        qso_entry["SRX"] = qso_receive_exchange_var.get().strip()

    # --- Validate frequency ---
    freq_input = qso_frequency_var.get().strip()
    if freq_input:
        try:
            qso_entry["Frequency"] = f"{float(freq_input):.6f}"
        except ValueError:
            messagebox.showerror("Invalid input", "Frequency must be numeric.")
            return

    # --- Validate locator ---
    if not is_valid_locator(qso_entry["Locator"]):
        messagebox.showerror(
            "Invalid Locator",
            "Locator must be ≥4 valid characters (e.g., JO22 or FN31tk)."
        )
        return

    # --- NEW method: update cache + Tree instantly ---
    qso_lines.insert(0, qso_entry)
    add_qso_row(qso_entry)

    # save async
    save_async()

    # Reset Track Time state after logging a QSO
    datetime_tracking_enabled.set(True)
    datetime_tracking_checkbox.config(state="normal")
    time_entry.config(fg="red")
    update_datetime()


    reset_fields()
    last_qso_label.config(text=f"Last QSO: {callsign} at {time_str} "f"on {qso_entry['Frequency']} MHz ({qso_entry['Mode']})")

    # Serial increment
    if use_serial_var.get():
        try:
            current_val = int(qso_sent_exchange_var.get())
            qso_sent_exchange_var.set(str(current_val + 1))
        except ValueError:
            qso_sent_exchange_var.set("1")

    # DXCluster spot update
    comment_out = qso_entry["Comment"]
    for refkey in ["WWFF", "POTA", "BOTA", "COTA", "IOTA", "SOTA", "WLOTA"]:
        if qso_entry.get(refkey):
            comment_out += f" {refkey}:{qso_entry[refkey]}"

    if dxspotviewer_window is not None and dxspotviewer_window.winfo_exists():
        if hasattr(dxspotviewer_window, "dxcluster_app"):
            app = dxspotviewer_window.dxcluster_app
            app.spot_callsign_var.set(qso_entry["Callsign"])
            app.spot_freq_var.set(qso_entry["Frequency"])
            app.spot_comment_var.set(comment_out.strip())

    # QRZ upload
    if upload_qrz_var.get():
        threading.Thread(target=upload_to_qrz, args=(qso_entry, False), daemon=True).start()






#########################################################################################
#   ___  ___  ____  _   _ ___ _    ___   _   ___  
#  / _ \| _ \|_  / | | | | _ \ |  / _ \ /_\ |   \ 
# | (_) |   / / /  | |_| |  _/ |_| (_) / _ \| |) |
#  \__\_\_|_\/___|  \___/|_| |____\___/_/ \_\___/ 
#
#########################################################################################

def qrz_upload_log(msg):
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    with open(QRZ_UPLOAD_LOG_FILE, "a", encoding="utf-8") as f:
        f.write(f"{timestamp}  {msg}\n")

# Function to build QRZ ADIF Compatible string
def build_adif(qso):
    # Use .get() for every key to avoid KeyError and provide default values if any are missing
    callsign = qso.get('Callsign', '')
    date = qso.get('Date', '')
    time = qso.get('Time', '')
    band = qso.get('Band', '')
    mode = qso.get('Mode', '')
    sent = qso.get('Sent', '')
    received = qso.get('Received', '')
    stx = qso.get('STX', '')
    srx = qso.get('SRX', '')
    station_callsign = qso.get('My Callsign', '')
    operator = qso.get('My Operator', '')
    locator = qso.get('Locator', '')
    name = qso.get('Name', '')
    comment = qso.get('Comment', '')
    frequency = qso.get('Frequency', '')

    # Construct the ADIF formatted QSO entry string with required fields
    adif = (
        f"<CALL:{len(callsign)}>{callsign}"
        f"<QSO_DATE:8>{date.replace('-', '')}"  # Date formatted as YYYYMMDD
        f"<TIME_ON:6>{time.replace(':', '')}"   # Time formatted as HHMMSS
        f"<BAND:{len(band)}>{band}"
        f"<MODE:{len(mode)}>{mode}"
        f"<RST_SENT:{len(sent)}>{sent}"
        f"<RST_RCVD:{len(received)}>{received}"
        f"<STX:{len(stx)}>{stx}"
        f"<SRX:{len(srx)}>{srx}"
        f"<STATION_CALLSIGN:{len(station_callsign)}>{station_callsign}"
        f"<OPERATOR:{len(operator)}>{operator}"
        f"<GRIDSQUARE:{len(locator)}>{locator}"
        f"<NAME:{len(name)}>{name}"
        f"<COMMENT:{len(comment)}>{comment}"
        f"<FREQ:{len(frequency)}>{frequency}"
    )

    # Append the <EOR> tag at the end of the ADIF string to indicate the end of the QSO record
    adif += "<EOR>"

    return adif

## Function to upload QSO to QRZ
def upload_to_qrz(qso, showstatus):

    # reset file at start of upload
    with open(QRZ_UPLOAD_LOG_FILE, "w", encoding="utf-8"):
        pass

    """
    Upload a QSO to the QRZ logbook.
    Returns the requests.Response object on success, or None on failure.
    """
    global QRZ_status_label
    response = None

    callsign = qso.get('CALL', '') or qso.get('Callsign', '')
    adif_str = build_adif(qso)

    qrz_upload_log(f"START QRZ UPLOAD for {callsign}")
    qrz_upload_log(f"ADIF={adif_str}")

    post_data = {
        "KEY": station_qrzapi_var.get(),
        "ACTION": "INSERT",
        "ADIF": adif_str,
    }

    qrz_upload_log(f"USING API KEY: {station_qrzapi_var.get()}")

    try:
        response = requests.post("https://logbook.qrz.com/api", data=post_data, timeout=10)
        qrz_upload_log(f"HTTP status={response.status_code}")

        if response.status_code == 200:
            text = response.text
            qrz_upload_log(f"SERVER RESPONSE={text}")

            if "RESULT=OK" in text:
                msg = f"✅ QSO {callsign} successfully uploaded to QRZ."
                qrz_upload_log(msg)

            elif "RESULT=FAIL" in text:
                reason = text.split("REASON=")[-1].split("&")[0]
                msg = f"⚠️ QRZ Upload failed: {reason}"
                qrz_upload_log(msg)

            elif "RESULT=AUTH" in text:
                msg = "❌ Invalid QRZ API key."
                qrz_upload_log(msg)

            else:
                msg = "⚠️ QRZ returned an unknown response."
                qrz_upload_log(msg)

        else:
            msg = f"❌ HTTP error while uploading to QRZ: {response.status_code}"
            qrz_upload_log(msg)

    except Exception as e:
        msg = "❌ QRZ upload failed: no internet connection or DNS error."
        qrz_upload_log(f"EXCEPTION={e}")
        qrz_upload_log(msg)
        response = None

    if not showstatus:
        QRZ_status_label.config(
            text=msg,
            fg="red",
            cursor="hand2" if "successfully uploaded" in msg else "",
            font=('Arial', 8, 'underline') if "successfully uploaded" in msg else ('Arial', 8)
        )
        QRZ_status_label.unbind("<Button-1>")
        if "successfully uploaded" in msg:
            QRZ_status_label.bind("<Button-1>", lambda e, cs=callsign: open_qrz_link(e, cs))

    return response






# -------------------------------------------------------------------
# MiniBook → WordPress Live Dashboard updater (threaded, safe)
# -------------------------------------------------------------------
def dashboard_updater():
    """Send live MiniBook data to WordPress every 3 seconds (threaded)."""

    DEBUG_UPDATER = False

    while True:
        try:
            enabled = minibook_web_enabled_var.get()

            if DEBUG_UPDATER:
                print(f"[UPDATER] RUNNING = {enabled}")

            if not enabled:
                time.sleep(1)
                continue

            # Read user settings
            url       = minibook_web_url_var.get().strip()
            radio_id  = minibook_web_id_var.get().strip()
            api_key   = minibook_web_api_var.get().strip()      # <-- NEW

            if not url or not radio_id:
                if DEBUG_UPDATER:
                    print("[UPDATER] Missing URL or ID → Skipping update")
                time.sleep(1)
                continue

            if not api_key:
                if DEBUG_UPDATER:
                    print("[UPDATER] Missing API KEY → Skipping update")
                time.sleep(1)
                continue

            # Build endpoint
            endpoint = f"{url.rstrip('/')}/?minibook_update={radio_id}"

            # Build payload (plugin accepts all fields + api_key)
            payload = {
                "api_key":  api_key,                          # <-- REQUIRED
                "freq":     qso_frequency_var.get().strip(),
                "band":     qso_band_var.get().strip(),
                "mode":     qso_mode_var.get().strip(),
                "mycall":   station_callsign_var.get().strip(),
                "opcall":   station_operator_var.get().strip(),
                "locator":  station_locator_var.get().strip(),
                "location": station_location_var.get().strip(),
                "radio":    radio_id,
                "radio_label": radio_id,
            }

            # Debug print
            if DEBUG_UPDATER:
                print("-----------------------------------------------------")
                print("[UPDATER] Sending POST update to WordPress…")
                print("[UPDATER] URL:", endpoint)
                print("[UPDATER] PAYLOAD:", payload)

            try:
                response = requests.post(endpoint, data=payload, timeout=5)

                if DEBUG_UPDATER:
                    print("[UPDATER] HTTP STATUS:", response.status_code)
                    print("[UPDATER] RESPONSE TEXT:", response.text[:200])

            except Exception as e:
                if DEBUG_UPDATER:
                    print("[UPDATER] EXCEPTION while sending POST:", str(e))

        except Exception as e:
            if DEBUG_UPDATER:
                print("[UPDATER] CRITICAL ERROR:", e)

        try:
            interval = int(config.get("Web_Dashboard", "update_interval", fallback="30"))
            if interval < 5:
                interval = 5
        except:
            interval = 30

        time.sleep(interval)













#########################################################################################
#  _   _ ___  ___     _   ___ ___ ___   _    ___   ___  ___ ___ _  _  ___ 
# | | | |   \| _ \   /_\ |   \_ _| __| | |  / _ \ / __|/ __|_ _| \| |/ __|
# | |_| | |) |  _/  / _ \| |) | || _|  | |_| (_) | (_ | (_ || || .` | (_ |
#  \___/|___/|_|   /_/ \_\___/___|_|   |____\___/ \___|\___|___|_|\_|\___|
#                                                                    
#########################################################################################                                                                     

listener_thread = None
listening = False
sock = None

def load_port_from_config(config):
    """Load the port setting from the configuration."""
    return int(config.get('UDP_ADIF_Settings', 'udp_adif_port', fallback="2338"))

def start_listener(config):
    """
    Start UDP ADIF listener.
    """
    global listening, listener_thread, sock
    if listening:
        print("Listener already running.")
        return

    listening = True
    port = load_port_from_config(config)
    listener_thread = threading.Thread(target=udp_adif_listener, args=(config, port))
    listener_thread.daemon = True
    listener_thread.start()
    print(f"Listener started on port {port}.")

def stop_listener():
    """
    Stop UDP ADIF listener.
    """
    global listening, sock
    if not listening:
        print("Listener not running.")
        return

    listening = False
    if sock:
        sock.close()
        sock = None
    print("Listener stopped.")

def restart_listener(config):
    """
    Restart the UDP ADIF listener upon a port change.
    """
    print("Restart listener...")
    stop_listener()
    time.sleep(1)  # Small pause to make sure everything closes properly
    start_listener(config)

def udp_adif_listener(config, port):
    """
    Listen to and digest UDP ADIF broadcasts.
    """
    global listening, sock, CURRENT_JSON_FILE

    host = "127.0.0.1"
    try:
        sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        sock.bind((host, port))
        print(f"Listen to UDP ADIF broadcasts on {host}:{port}...")

        while listening:
            try:
                data, addr = sock.recvfrom(4096)
                if not data:
                    continue

                # Decode the ADIF message
                try:
                    adif_record = data.decode("utf-8").strip()
                except UnicodeDecodeError:
                    adif_record = data.decode("latin-1").strip()

                if not adif_record:
                    continue

                # Check if it is a valid ADIF message
                if not is_valid_adif(adif_record):
                    print("Invalid ADIF datagram. Message is ignored.")
                    continue

                # Check if a log is open
                if not CURRENT_JSON_FILE:
                    messagebox.showerror(
                        "Error",
                        "UDP ADIF log data received, but no log loaded!\n"
                        "Load a log and try again."
                    )
                    continue

                # Process the QSO data
                process_qso(adif_record)

            except socket.error as e:
                if not listening:
                    break
                print(f"Socket-error: {e}")
    except Exception as e:
        print(f"Error starting the listener: {e}")
    finally:
        if sock:
            sock.close()
        print("Listener terminated.")





# Function that processes an received UDP ADIF record
def process_qso(adif_record):
    global CURRENT_JSON_FILE, entityCode

    callsign = (extract_field(adif_record, "call") or "").upper()
    name = (extract_field(adif_record, "name") or "")

    time_field = (
        import_format_time(extract_field(adif_record, "time_off")) or
        import_format_time(extract_field(adif_record, "time_on")) or
        ""
    )

    # --- BASE STRUCTURE: SAME DEFAULTS AS log_qso() ---
    qso_entry = {
        "Date": import_format_date(extract_field(adif_record, "qso_date")) or "",
        "Time": time_field,
        "Callsign": callsign,
        "Name": name,

        # Station info
        "My Callsign": station_callsign_var.get().upper(),
        "My Operator": station_operator_var.get().upper(),
        "My Locator": station_locator_var.get().upper(),
        "My Location": station_location_var.get(),

        # DXCC fields
        "Country": "",
        "Continent": "",
        "dxcc": "",
        "CQZ": "",
        "ITUZ": "",

        # Reports
        "Sent": (extract_field(adif_record, "rst_sent") or "").upper(),
        "Received": (extract_field(adif_record, "rst_rcvd") or "").upper(),

        # Mode / band / freq
        "Mode": (extract_field(adif_record, "mode") or "").upper(),
        "Submode": (extract_field(adif_record, "submode") or "").upper(),
        "Band": (extract_field(adif_record, "band") or "").lower(),
        "Frequency": extract_field(adif_record, "freq") or "",
        "Locator": (extract_field(adif_record, "gridsquare") or "").upper(),
        "Prop": extract_field(adif_record, "prop_mode") or "",
        "Comment": extract_field(adif_record, "comment") or "",

        # Contest
        "STX": extract_field(adif_record, "stx") or "",
        "SRX": extract_field(adif_record, "srx") or "",

        # QSL defaults (MATCH log_qso!)
        "QS": "No",
        "QSD": "",
        "QR": "No",
        "QRD": "",
        "LWS": "No",
        "LWSD": "",
        "LWR": "No",
        "LWRD": "",
    }

    # DXCC resolving like log_qso
    check_callsign_prefix(callsign, update_ui=False)

    qso_entry["Country"] = qso_country_var.get()
    qso_entry["Continent"] = qso_continent_var.get()
    qso_entry["dxcc"] = entityCode if entityCode else ""
    qso_entry["CQZ"]  = qso_cq_zone_var.get()
    qso_entry["ITUZ"] = qso_itu_zone_var.get()


    # Store
    #add_qso_to_logbook(qso_entry)
    root.after(0, lambda: add_qso_to_logbook(qso_entry))
        
    # Popup
    root.after(0, lambda: show_auto_close_messagebox(
        "New QSO",
        f"QSO logged!\n\n"
        f"Call: {qso_entry['Callsign']}\n"
        f"Time: {qso_entry['Time']}, Band: {qso_entry['Band']}\n"
        f"Mode: {qso_entry['Mode']}\n"
        f"Freq: {qso_entry['Frequency']} MHz",
        duration=3000
    ))

    # QRZ upload same as before
    if upload_qrz_var.get():
        threading.Thread(target=upload_to_qrz, args=(qso_entry, False), daemon=True).start()




def add_qso_to_logbook(qso_entry):
    """
    Add a QSO entry directly to the loaded logbook and update the logbook
    window if open. DXCC is already assigned in process_qso().
    """
    global qso_lines

    try:
        # Add to cache at top
        qso_lines.insert(0, qso_entry)

        # Update Tree instantly (fast)
        add_qso_row(qso_entry)

        # Save cache to disk (async)
        save_async()

        # Update Worked-Before info
        update_worked_before_tree()

        # Update Last QSO label
        last_qso_label.config(
            text=f"Last QSO with {qso_entry['Callsign']} at {qso_entry['Time']} "
                 f"on {float(qso_entry['Frequency']):.3f}MHz in {qso_entry['Mode']}"
        )

        # ------------------------------------------------
        # Build comment for DXCluster spot
        # ------------------------------------------------
        comment_out = qso_entry.get("Comment", "")

        if qso_entry.get("Mode"):
            comment_out += f"{qso_entry['Mode']}"
        '''
        if qso_entry.get("dxcc"):
            comment_out += f" DXCC:{qso_entry['dxcc']}"
        '''
        for refkey in ["WWFF", "POTA", "BOTA", "COTA", "IOTA", "SOTA", "WLOTA"]:
            if qso_entry.get(refkey):
                comment_out += f" {refkey}:{qso_entry[refkey]}"

        # DXCluster update
        if dxspotviewer_window is not None and dxspotviewer_window.winfo_exists():
            if hasattr(dxspotviewer_window, "dxcluster_app"):
                app = dxspotviewer_window.dxcluster_app
                app.spot_callsign_var.set(qso_entry["Callsign"])
                app.spot_freq_var.set(qso_entry["Frequency"])
                app.spot_comment_var.set(comment_out.strip())

    except Exception as e:
        show_auto_close_messagebox(
            "MiniBook",
            f"Error!\n\nFailed to log QSO: {e}",
            duration=3000
        )




def show_auto_close_messagebox(title, message, duration=2000):
    def _show():
        temp_root = tk.Toplevel(root)
        temp_root.title(title)
        temp_root.geometry("300x150")
        temp_root.resizable(False, False)
        temp_root.attributes("-topmost", True)
        temp_root.overrideredirect(True)

        root.update_idletasks()
        root_x = root.winfo_rootx()
        root_y = root.winfo_rooty()
        root_width = root.winfo_width()
        root_height = root.winfo_height()
        x = root_x + (root_width - 300) // 2
        y = root_y + (root_height - 150) // 2
        temp_root.geometry(f"300x150+{x}+{y}")

        border = tk.Frame(temp_root, bg='black', bd=2)
        border.pack(expand=True, fill='both')

        inner = tk.Frame(border, bg='white')
        inner.pack(expand=True, fill='both', padx=1, pady=1)

        label = tk.Label(inner, text=message, wraplength=280, justify="center", bg='white', font=('Arial', 10))
        label.pack(expand=True, pady=20)

        temp_root.after(duration, temp_root.destroy)

    root.after(0, _show)





def is_valid_adif(adif_record):
    """
    Validate if the ADIF record is a valid datagram.
    Checks for the presence of key tags and general structure.
    """
    # Required tags for a minimal valid ADIF record
    required_tags = ["<call:", "<qso_date:", "<band:", "<mode:"]

    # Check for the presence of required tags
    for tag in required_tags:
        if not re.search(re.escape(tag), adif_record, re.IGNORECASE):
            return False

    # Ensure either <time_on> or <time_off> is present
    if not (re.search(r"<time_on:", adif_record, re.IGNORECASE) or re.search(r"<time_off:", adif_record, re.IGNORECASE)):
        return False

    # Check for <EOR> or <eor> to mark the end of the record
    if not re.search(r"<eor>", adif_record, re.IGNORECASE):
        return False

    return True


def extract_adif_field(record, field_name):
    """
    Extract a field's value from the ADIF record in the format <FIELD_NAME:length>value.
    """
    # Search for the field in a case-insensitive manner with an optional length attribute
    pattern = rf"<{field_name}:(\d+)>([^<]*)"
    match = re.search(pattern, record, re.IGNORECASE)
    if match:
        return match.group(2).strip()  # Extract the value and remove any surrounding whitespace
    return ""  # Return an empty string if the field is not found






#########################################################################################
#  ___ ___ ___    ___ ___  _  _ _____ ___  ___  _    
# | _ \_ _/ __|  / __/ _ \| \| |_   _| _ \/ _ \| |   
# |   /| | (_ | | (_| (_) | .` | | | |   / (_) | |__ 
# |_|_\___\___|  \___\___/|_|\_| |_| |_|_\\___/|____|
#
#########################################################################################

# =====================================================================
# Reconnect controls
# =====================================================================
hamlib_reconnect_running = threading.Event()
hamlib_reconnect_thread = None


# ----------------------------------------------------------
# CAT send helper
# ----------------------------------------------------------
def hamlib_send_command(cmd: str,
                        recv_size: int = 1024,
                        timeout: float = 1.0,
                        retries: int = 1) -> str:

    if socket_connection is None:
        hamlib_logger.error(f"CAT send failed (no socket): '{cmd.strip()}'")
        raise ConnectionError("No hamlib socket connection")

    last_exc = None
    for attempt in range(1, retries + 1):
        try:
            socket_connection.sendall(cmd.encode())
            socket_connection.settimeout(timeout)
            return socket_connection.recv(recv_size).decode(errors="ignore")
        except Exception as e:
            last_exc = e
            if attempt == retries:
                hamlib_logger.error(
                    f"CAT send failed after {retries} attempt(s) for '{cmd.strip()}': {e}"
                )
    raise last_exc


# =====================================================================
# High level request to start Hamlib (manual or auto)
# =====================================================================
def hamlib_request_connect(force: bool = False):

    if not config.getboolean('hamlib_settings', 'enable_radio', fallback=False):
        hamlib_logger.info("Hamlib disabled in preferences → not connecting")
        return

    # Manual force reconnect
    if force:
        try:
            disconnect_from_hamlib()
        except:
            pass

    # Always start connect attempt
    connect_to_hamlib_threaded()



def connect_to_hamlib_threaded():
    threading.Thread(target=connect_to_hamlib, daemon=True).start()


def connect_to_hamlib():
    global socket_connection, hamlib_PORT

    stop_frequency_thread.clear()
    hamlib_PORT = config.get('hamlib_settings', 'hamlib_port', fallback="4532")

    server_ip = hamlib_ip_var.get()
    try:
        server_port = int(hamlib_PORT)
    except:
        server_port = 4532

    try:
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.connect((server_ip, server_port))

        socket_connection = sock
        gui_state_control(11)
        hamlib_logger.info(f"Hamlib connected to {server_ip}:{server_port}")

        threading.Thread(target=update_frequency_and_mode_thread,
                         daemon=True).start()

    except Exception as e:
        hamlib_logger.warning(f"Hamlib initial connect failed: {e}")
        hamlib_connection_lost(str(e))


def establish_socket_connection():
    global socket_connection
    server_ip = hamlib_ip_var.get()
    try:
        server_port = int(hamlib_PORT)
    except ValueError:
        server_port = 4532
        hamlib_logger.error(f"Invalid hamlib_port in config: {hamlib_PORT}, using 4532")

    try:
        socket_connection = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        socket_connection.connect((server_ip, server_port))

        hamlib_logger.info(f"Hamlib reconnected to {server_ip}:{server_port}")
        gui_state_control(11)
        threading.Thread(target=update_frequency_and_mode_thread,
                         daemon=True).start()

    except socket.error as e:
        hamlib_logger.error(f"Socket error during reconnect to Hamlib: {e}")
        disconnect_from_hamlib()


def disconnect_from_hamlib():
    global socket_connection, hamlib_process

    freqmode_tracking_var.set(False)
    gui_state_control(12)

    stop_frequency_thread.set()
    time.sleep(0.2)

    if socket_connection:
        try:
            socket_connection.close()
        except:
            pass
        socket_connection = None

    if hamlib_process:
        try:
            hamlib_process.terminate()
        except:
            pass
        hamlib_process = None

    hamlib_logger.info("Hamlib disconnected")


def hamlib_connection_lost(reason: str):
    hamlib_logger.warning(f"Hamlib connection lost: {reason}")
    disconnect_from_hamlib()
    start_hamlib_reconnect_loop()



def start_hamlib_reconnect_loop():
    global hamlib_reconnect_thread

    if hamlib_reconnect_running.is_set():
        return

    hamlib_reconnect_running.set()

    def worker():
        global socket_connection
        attempts = 0
        time.sleep(2)

        while hamlib_reconnect_running.is_set():
            attempts += 1
            server_ip = hamlib_ip_var.get()
            port = int(config.get("hamlib_settings", "hamlib_port", fallback="4532"))

            try:
                sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
                sock.settimeout(3.0)
                sock.connect((server_ip, port))

                socket_connection = sock
                gui_state_control(11)
                hamlib_logger.info(
                    f"Hamlib reconnected to {server_ip}:{port} after {attempts} attempt(s)"
                )

                stop_frequency_thread.clear()
                threading.Thread(target=update_frequency_and_mode_thread,
                                 daemon=True).start()

                hamlib_reconnect_running.clear()
                return

            except Exception as e:
                if attempts == 1 or attempts % 10 == 0:
                    hamlib_logger.warning(
                        f"Hamlib reconnect attempt {attempts} failed: {e}"
                    )

            time.sleep(3)

    hamlib_reconnect_thread = threading.Thread(target=worker, daemon=True)
    hamlib_reconnect_thread.start()

# =====================================================================
# Master HAMLIB start / bootstrap function
# =====================================================================
def start_hamlib_engine():

    if not config.getboolean('hamlib_settings', 'enable_radio', fallback=False):
        hamlib_logger.info("Hamlib engine disabled by preferences.")
        return

    hamlib_logger.info("Starting Hamlib engine...")

    if hamlib_reconnect_running.is_set():
        hamlib_logger.info("Reconnect loop already running → nothing to do.")
        return

    hamlib_request_connect(force=True)
    toggle_tracking()



# ==========================================================
# RPRT error handling
# ==========================================================
def handle_rprtx_error(error_code):
    hamlib_logger.error(f"RPRT {error_code} received from Hamlib")

    if error_code in [-7, -9, -13, -5, -8, -10]:
        hamlib_connection_lost(f"RPRT {error_code}")
        return

    hamlib_logger.warning(f"Non-critical Hamlib error: {error_code}")


# ==========================================================
# Frequency / Mode polling thread
# ==========================================================
def update_frequency_and_mode_thread():
    """Background thread that polls rig frequency and mode."""
    global qso_frequency_var, frequency_mhz, last_passband
    last_mode = None

    while not stop_frequency_thread.is_set():

        # --- safety check -------------------------------------------------
        if socket_connection is None:
            hamlib_logger.info("Hamlib update thread exiting (no socket)")
            break

        # --- Frequency read ---
        try:
            socket_connection.sendall(b"f\n")
            freq_resp = socket_connection.recv(64).decode().strip()

            if freq_resp.startswith("RPRT"):
                try:
                    code = int(freq_resp.split()[-1])
                except ValueError:
                    code = -999
                handle_rprtx_error(code)
                break

            if freq_resp.isdigit():
                frequency_hz = int(freq_resp)
                frequency_mhz = frequency_hz / 1_000_000

                # === NEW: Do not override GUI when manual override active ===
                if tracking_enabled:
                    qso_frequency_var.set(f"{frequency_mhz:.4f}")


            else:
                hamlib_logger.error(
                    f"Invalid frequency response from Hamlib: '{freq_resp}'"
                )

        except Exception as e:
            hamlib_logger.error(f"Hamlib frequency read error: {e}")
            hamlib_connection_lost(str(e))
            break

        time.sleep(0.1)

        # --- safety check again ------------------------------------------
        if socket_connection is None:
            hamlib_logger.info("Hamlib update thread exiting (socket closed)")
            break

        # --- Mode / passband read ---
        try:
            socket_connection.sendall(b"m\n")
            response = socket_connection.recv(256).decode().strip()

            if response.startswith("RPRT"):
                try:
                    code = int(response.split()[-1])
                except ValueError:
                    code = -999
                handle_rprtx_error(code)
                break

            lines = response.splitlines()
            if len(lines) >= 2:
                mode = lines[0].strip()
                passband_str = lines[1].strip()

                # === NEW: No mode override if manual freq override ===
                if tracking_enabled and mode and mode != last_mode:
                    hamlib_logger.info(f"Hamlib mode changed: {last_mode} -> {mode}")
                    qso_mode_var.set(mode)
                    last_mode = mode
                    on_mode_change()

                try:
                    pb = int(passband_str)
                    if pb > 0 and pb != last_passband:
                        last_passband = pb
                except ValueError:
                    hamlib_logger.error(
                        f"Invalid passband from Hamlib: '{passband_str}'"
                    )
            else:
                hamlib_logger.error(
                    f"Incomplete Hamlib mode/passband response: {lines}"
                )

        except Exception as e:
            hamlib_logger.error(f"Hamlib mode read error: {e}")
            hamlib_connection_lost(str(e))
            break

        time.sleep(0.5)



# ==========================================================
# Manual & Spot Frequency / Mode / Callsign input handler
# ==========================================================

def handle_callsign_input(source="manual", callsign=None, freq=None, mode=None, event=None):
    """
    Unified callsign/frequency/mode handler for both manual entry and DXCluster spots.
    source: "manual" or "spot"
    """

    def check_response(response):
        if any(e in response for e in ("RPRT -7", "RPRT -9", "RPRT -13")):
            hamlib_logger.error(f"Hamlib command failed: {response}")
            hamlib_connection_lost(response)
            return False
        return True

    def refocus(clear=True):
        if clear:
            qso_callsign_var.set("")
        callsign_entry.focus_set()
        return "break"

    # ----------------------------------------------------------------------
    # SPOT INPUT (always takes highest priority)
    # ----------------------------------------------------------------------
    if source == "spot":
        try:
            reset_fields()
        except Exception:
            pass

        # Update GUI fields
        qso_callsign_var.set(callsign)
        callsign_entry.focus_set()
        callsign_entry.icursor(tk.END)

        if freq:
            qso_frequency_var.set(str(freq))

        if mode and mode in mode_combobox['values']:
            qso_mode_var.set(mode)

        # QRZ lookup
        use_qrz = config.getboolean("QRZ", "use_qrz_lookup", fallback=False)
        if use_qrz:
            threaded_on_query()

        # Send to hamlib
        if socket_connection is not None and freq:
            try:
                mhz = float(str(freq).replace(",", "."))
                hz = int(mhz * 1_000_000)

                response = hamlib_send_command(f"F {hz}\n", retries=2)

                if not check_response(response):
                    hamlib_logger.error(
                        f"Hamlib refused frequency {hz} Hz from spot"
                    )
                else:
                    hamlib_logger.info(
                        f"Frequency set from DXCluster spot: {hz} Hz"
                    )

            except Exception as e:
                hamlib_logger.error(
                    f"Error setting hamlib freq from spot '{freq}': {e}"
                )

        return

    # ----------------------------------------------------------------------
    # MANUAL ENTRY
    # ----------------------------------------------------------------------
    value = qso_callsign_var.get().strip().upper()

    valid_modes = {
        'USB', 'LSB', 'CW', 'CWR', 'RTTY', 'RTTYR', 'AM', 'FM', 'WFM', 'AMS',
        'PKTLSB', 'PKTUSB', 'PKTFM', 'ECSSUSB', 'ECSSLSB', 'FA',
        'SAM', 'SAL', 'SAH', 'DSB'
    }

    # Frequency +/-
    if re.fullmatch(r'[+-]\d{1,5}', value):
        try:
            step_khz = int(value)
            if frequency_mhz is None:
                return refocus()

            new_freq_hz = int((frequency_mhz + (step_khz / 1000.0)) * 1_000_000)

            if socket_connection is None:
                return refocus()

            response = hamlib_send_command(f"F {new_freq_hz}\n", retries=2)

            if check_response(response) and "RPRT 0" in response:
                hamlib_logger.info(f"Frequency adjusted by {step_khz} kHz")

        except Exception as e:
            hamlib_logger.error(f"Error adjusting frequency: {e}")

        return refocus()

    # MODE input
    if value in valid_modes:
        if socket_connection is None:
            return refocus()

        try:
            mode_response = hamlib_send_command("+m\n", retries=2)

            if not check_response(mode_response):
                return refocus()

            current_mode, current_passband = None, -1

            for line in mode_response.strip().splitlines():
                if line.startswith("Mode:"):
                    current_mode = line.split(":", 1)[1].strip()
                elif line.startswith("Passband:"):
                    try:
                        current_passband = int(line.split(":", 1)[1].strip())
                    except:
                        pass

            if current_mode != value:
                response = hamlib_send_command(
                    f"+M {value} {current_passband}\n",
                    retries=2
                )

                if check_response(response) and "RPRT 0" in response:
                    hamlib_logger.info(
                        f"Mode manually set to {value} (pb {current_passband})"
                    )

        except Exception as e:
            hamlib_logger.error(f"Unexpected mode error '{value}': {e}")

        return refocus()

    # Direct frequency input
    if re.fullmatch(r'\d{4,7}', value):
        try:
            if socket_connection is not None:
                frequency_hz = int(value) * 1000
                response = hamlib_send_command(f"F {frequency_hz}\n", retries=2)

                if check_response(response) and "RPRT 0" in response:
                    hamlib_logger.info(
                        f"Frequency manually set to {frequency_hz} Hz"
                    )

        except Exception as e:
            hamlib_logger.error(f"Frequency error '{value}': {e}")

        return refocus()

    # no modification, treat as callsign
    return refocus(clear=False)


#########################################################################################
#  __  __ ___ _  _ _   _ _ ___ 
# |  \/  | __| \| | | | ( ) __|
# | |\/| | _|| .` | |_| |/\__ \
# |_|  |_|___|_|\_|\___/  |___/
#
#########################################################################################                              

def open_station_setup():
    """Open the Station Setup window centered above the main root window."""
    global Station_Setup_Window

    # If already open → bring to front
    if Station_Setup_Window is not None and Station_Setup_Window.winfo_exists():
        Station_Setup_Window.lift()
        return

    # Create window
    Station_Setup_Window = tk.Toplevel(root)
    Station_Setup_Window.title("Station Setup")

    # Allow auto sizing while building
    Station_Setup_Window.resizable(True, True)

    # Make modal
    Station_Setup_Window.transient(root)
    Station_Setup_Window.grab_set()
    Station_Setup_Window.focus_set()
    Station_Setup_Window.lift()

    # ----------------------------------------------------------------------
    # Center window AFTER layout is ready, geometry is now dynamic
    # ----------------------------------------------------------------------
    station_callsign = tk.StringVar(value=station_callsign_var.get())
    station_operator = tk.StringVar(value=station_operator_var.get())
    station_locator = tk.StringVar(value=station_locator_var.get())
    station_location = tk.StringVar(value=station_location_var.get())

    station_name = tk.StringVar(value=station_name_var.get())
    station_street = tk.StringVar(value=station_street_var.get())
    station_postalcode = tk.StringVar(value=station_postalcode_var.get())
    station_city = tk.StringVar(value=station_city_var.get())
    station_county = tk.StringVar(value=station_county_var.get())
    station_country = tk.StringVar(value=station_country_var.get())
    station_cqzone = tk.StringVar(value=station_cqzone_var.get())
    station_ituzone = tk.StringVar(value=station_ituzone_var.get())

    station_contest = tk.StringVar(value=station_contest_var.get())
    station_pota = tk.StringVar(value=station_pota_var.get())
    station_pota_name = tk.StringVar()
    station_bota = tk.StringVar(value=station_bota_var.get())
    station_bota_name = tk.StringVar()
    station_bota_lat = tk.StringVar()
    station_bota_long = tk.StringVar()
    station_wwff = tk.StringVar(value=station_wwff_var.get())
    station_wwff_name = tk.StringVar()
    station_wwff_lat = tk.StringVar()
    station_wwff_long = tk.StringVar()
    station_iota = tk.StringVar(value=station_iota_var.get())
    station_iota_name = tk.StringVar()
    station_iota_lat = tk.StringVar()
    station_iota_long = tk.StringVar()
    station_cota = tk.StringVar(value=station_cota_var.get())
    station_cota_name = tk.StringVar()
    station_sota = tk.StringVar(value=station_sota_var.get())
    station_sota_name = tk.StringVar()
    station_wlota = tk.StringVar(value=station_wlota_var.get())

    station_qrzapi = tk.StringVar(value=station_qrzapi_var.get())
    station_upload_qrz = tk.BooleanVar(value=upload_qrz_var.get())

    station_callsign.trace_add("write", lambda *args: station_callsign.set(station_callsign.get().upper()))
    station_operator.trace_add("write", lambda *args: station_operator.set(station_operator.get().upper()))
    station_locator.trace_add("write", lambda *args: station_locator.set(station_locator.get().upper()))
    station_postalcode.trace_add("write", lambda *args: station_postalcode.set(station_postalcode.get().upper()))
    station_wwff.trace_add("write", lambda *args: station_wwff.set(station_wwff.get().upper()))
    station_pota.trace_add("write", lambda *args: station_pota.set(station_pota.get().upper()))
    station_bota.trace_add("write", lambda *args: station_bota.set(station_bota.get().upper()))
    station_iota.trace_add("write", lambda *args: station_iota.set(station_iota.get().upper()))
    station_sota.trace_add("write", lambda *args: station_sota.set(station_sota.get().upper()))
    station_wlota.trace_add("write", lambda *args: station_wlota.set(station_wlota.get().upper()))
    station_cota.trace_add("write", lambda *args: station_cota.set(station_cota.get().upper()))

    Station_Setup_Window.grid_columnconfigure(0, weight=1)

    def open_qralocator():
        locator = station_locator.get().strip().upper()
        if locator:
            url = f"https://locator.pd5dj.nl/?locator={locator}"
        else:
            url = "https://locator.pd5dj.nl/"
        webbrowser.open(url)

    # ==================================================
    # Station Information Notebook
    # ==================================================
    station_notebook = ttk.Notebook(Station_Setup_Window)
    station_notebook.grid(row=0, column=0, padx=10, pady=5, sticky="ew")

    # ---------- Tab 1: Station Basic ----------
    basic_frame = tk.Frame(station_notebook, background='#f0f0f0')
    station_notebook.add(basic_frame, text="Basic Info")
    basic_frame.grid_columnconfigure(1, weight=1)

    tk.Label(basic_frame, text="Callsign:", font=('Arial', 10)).grid(row=0, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(basic_frame, textvariable=station_callsign, font=('Arial', 10, 'bold')).grid(row=0, column=1, padx=10, pady=5, sticky='ew')
    tk.Label(basic_frame, text="(Station Callsign)", font=('Arial', 10)).grid(row=0, column=2, padx=10, pady=1, sticky='w')
    
    tk.Label(basic_frame, text="Operator:", font=('Arial', 10)).grid(row=1, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(basic_frame, textvariable=station_operator, font=('Arial', 10, 'bold')).grid(row=1, column=1, padx=10, pady=5, sticky='ew')
    tk.Label(basic_frame, text="(Callsign)", font=('Arial', 10)).grid(row=1, column=2, padx=10, pady=1, sticky='w')

    tk.Label(basic_frame, text="Locator:", font=('Arial', 10)).grid(row=2, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(basic_frame, textvariable=station_locator, font=('Arial', 10, 'bold')).grid(row=2, column=1, padx=10, pady=5, sticky='ew')

    btn = tk.Button(basic_frame, text="Find Your QRA Locator", command=open_qralocator)
    btn.grid(row=2, column=2, padx=10, pady=5)

    tk.Label(basic_frame, text="QSO Location:", font=('Arial', 10)).grid(row=3, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(basic_frame, textvariable=station_location, font=('Arial', 10, 'bold')).grid(row=3, column=1, padx=10, pady=5, sticky='ew')

    # ---------- Tab 2: Address / Zone ----------
    address_frame = tk.Frame(station_notebook, background='#f0f0f0')
    station_notebook.add(address_frame, text="Address / Zone Info")
    address_frame.grid_columnconfigure(1, weight=1)

    tk.Label(address_frame, text="Name:", font=('Arial', 10)).grid(row=0, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(address_frame, textvariable=station_name, font=('Arial', 10, 'bold')).grid(row=0, column=1, padx=10, pady=5, sticky='ew')

    tk.Label(address_frame, text="Street:", font=('Arial', 10)).grid(row=1, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(address_frame, textvariable=station_street, font=('Arial', 10, 'bold')).grid(row=1, column=1, padx=10, pady=5, sticky='ew')

    tk.Label(address_frame, text="Postalcode:", font=('Arial', 10)).grid(row=2, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(address_frame, textvariable=station_postalcode, font=('Arial', 10, 'bold')).grid(row=2, column=1, padx=10, pady=5, sticky='ew')

    tk.Label(address_frame, text="City:", font=('Arial', 10)).grid(row=3, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(address_frame, textvariable=station_city, font=('Arial', 10, 'bold')).grid(row=3, column=1, padx=10, pady=5, sticky='ew')

    tk.Label(address_frame, text="County:", font=('Arial', 10)).grid(row=4, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(address_frame, textvariable=station_county, font=('Arial', 10, 'bold')).grid(row=4, column=1, padx=10, pady=5, sticky='ew')

    tk.Label(address_frame, text="Country:", font=('Arial', 10)).grid(row=5, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(address_frame, textvariable=station_country, font=('Arial', 10, 'bold')).grid(row=5, column=1, padx=10, pady=5, sticky='ew')

    tk.Label(address_frame, text="CQ Zone:", font=('Arial', 10)).grid(row=6, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(address_frame, textvariable=station_cqzone, font=('Arial', 10, 'bold')).grid(row=6, column=1, padx=10, pady=5, sticky='ew')

    tk.Label(address_frame, text="ITU Zone:", font=('Arial', 10)).grid(row=7, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(address_frame, textvariable=station_ituzone, font=('Arial', 10, 'bold')).grid(row=7, column=1, padx=10, pady=5, sticky='ew')

    # ---------------- LabelFrame 3: Contest info ----------------
    lf2 = tk.LabelFrame(Station_Setup_Window, text="Participating Contest", font=('Arial', 10, 'bold'))
    lf2.grid(row=2, column=0, padx=10, pady=5, sticky="ew")
    lf2.grid_columnconfigure(1, weight=1)

    tk.Label(lf2, text="Contest:", font=('Arial', 10)).grid(row=0, column=0, padx=10, pady=1, sticky='w')
    tk.Entry(lf2, textvariable=station_contest, font=('Arial', 10, 'bold')).grid(row=0, column=1, padx=10, pady=5, sticky='ew')

    notebook = ttk.Notebook(Station_Setup_Window)
    notebook.grid(row=3, column=0, padx=10, pady=5, sticky="nsew")

    Station_Setup_Window.grid_rowconfigure(1, weight=0)
    Station_Setup_Window.grid_columnconfigure(0, weight=1)

    # ---------- POTA tab ----------
    pota_frame = tk.Frame(notebook, background='#f0f0f0')
    notebook.add(pota_frame, text="POTA")

    pota_frame.grid_columnconfigure(1, weight=1)

    tk.Label(pota_frame, text="Reference No.:", font=('Arial', 10)).grid(row=0, column=0, padx=10, pady=1, sticky='w')
    pota_entry = tk.Entry(pota_frame, textvariable=station_pota, font=('Arial', 10, 'bold'))
    pota_entry.grid(row=0, column=1, padx=10, pady=5, sticky='ew')
    pota_entry.bind("<FocusOut>", lambda e: auto_fill_pota_prefix(station_pota, station_pota_name, station_callsign))

    tk.Label(pota_frame, text="Park name:", font=('Arial', 10)).grid(row=1, column=0, columnspan=2, padx=10, pady=1, sticky='w')
    tk.Label(pota_frame, textvariable=station_pota_name, font=('Arial', 10, 'bold'), wraplength=360, justify="left").grid(row=1, column=1, columnspan=2, padx=10, pady=1, sticky='w')

    # ---------- WWFF tab ----------
    wwff_frame = tk.Frame(notebook, background='#f0f0f0')
    notebook.add(wwff_frame, text="WWFF")

    wwff_frame.grid_columnconfigure(1, weight=1)

    tk.Label(wwff_frame, text="Reference No.:", font=('Arial', 10)).grid(row=0, column=0, padx=10, pady=1, sticky='w')
    wwff_entry = tk.Entry(wwff_frame, textvariable=station_wwff, font=('Arial', 10, 'bold'))
    wwff_entry.grid(row=0, column=1, padx=10, pady=5, sticky='ew')
    wwff_entry.bind("<FocusOut>", lambda e: auto_fill_wwff_prefix(station_wwff, station_wwff_name, station_wwff_lat, station_wwff_long, station_callsign))

    tk.Label(wwff_frame, text="Park name:", font=('Arial', 10)).grid(row=1, column=0, columnspan=2, padx=10, pady=1, sticky='w')
    tk.Label(wwff_frame, textvariable=station_wwff_name, font=('Arial', 10, 'bold'), wraplength=360, justify="left").grid(row=1, column=1, columnspan=2, padx=10, pady=1, sticky='w')

    # ---------- BOTA tab ----------
    bota_frame = tk.Frame(notebook, background='#f0f0f0')
    notebook.add(bota_frame, text="BOTA")

    bota_frame.grid_columnconfigure(1, weight=1)

    tk.Label(bota_frame, text="Reference No.:", font=('Arial', 10)).grid(row=0, column=0, padx=10, pady=1, sticky='w')
    bota_entry = tk.Entry(bota_frame, textvariable=station_bota, font=('Arial', 10, 'bold'))
    bota_entry.grid(row=0, column=1, padx=10, pady=5, sticky='ew')
    bota_entry.bind("<FocusOut>", lambda e: auto_fill_bota_prefix(station_bota, station_bota_name, station_bota_lat, station_bota_long, station_callsign))

    tk.Label(bota_frame, text="Bunker name:", font=('Arial', 10)).grid(row=1, column=0, columnspan=2, padx=10, pady=1, sticky='w')
    tk.Label(bota_frame, textvariable=station_bota_name, font=('Arial', 10, 'bold'), wraplength=360, justify="left").grid(row=1, column=1, columnspan=2, padx=10, pady=1, sticky='w')

    # ---------- COTA tab ----------
    cota_frame = tk.Frame(notebook, background='#f0f0f0')
    notebook.add(cota_frame, text="COTA")

    bota_frame.grid_columnconfigure(1, weight=1)

    tk.Label(cota_frame, text="Reference No.:", font=('Arial', 10)).grid(row=0, column=0, padx=10, pady=1, sticky='w')
    cota_entry = tk.Entry(cota_frame, textvariable=station_cota, font=('Arial', 10, 'bold'))
    cota_entry.grid(row=0, column=1, padx=10, pady=5, sticky='ew')
    cota_entry.bind("<FocusOut>", lambda e: auto_fill_cota_prefix(station_cota, station_cota_name, station_callsign))

    tk.Label(cota_frame, text="Castle name:", font=('Arial', 10)).grid(row=1, column=0, columnspan=2, padx=10, pady=1, sticky='w')
    tk.Label(cota_frame, textvariable=station_cota_name, font=('Arial', 10, 'bold'), wraplength=360, justify="left").grid(row=1, column=1, columnspan=2, padx=10, pady=1, sticky='w')

    # ---------- IOTA tab ----------
    iota_frame = tk.Frame(notebook, background='#f0f0f0')
    notebook.add(iota_frame, text="IOTA")

    iota_frame.grid_columnconfigure(1, weight=1)

    tk.Label(iota_frame, text="Reference No.:", font=('Arial', 10)).grid(row=0, column=0, padx=10, pady=1, sticky='w')
    bota_entry = tk.Entry(iota_frame, textvariable=station_iota, font=('Arial', 10, 'bold'))
    bota_entry.grid(row=0, column=1, padx=10, pady=5, sticky='ew')
    bota_entry.bind("<FocusOut>", lambda e: auto_fill_iota_prefix(station_iota, station_iota_name, station_iota_lat, station_iota_long, station_callsign))

    tk.Label(iota_frame, text="Island name:", font=('Arial', 10)).grid(row=1, column=0, columnspan=2, padx=10, pady=1, sticky='w')
    tk.Label(iota_frame, textvariable=station_iota_name, font=('Arial', 10, 'bold'), wraplength=360, justify="left").grid(row=1, column=1, columnspan=2, sticky='w')

    # ==================================================
    # SOTA Tab (Station Setup)
    # ==================================================
    sota_frame = tk.Frame(notebook, background='#f0f0f0')
    notebook.add(sota_frame, text="SOTA")

    sota_frame.grid_columnconfigure(0, weight=0, minsize=70)
    sota_frame.grid_columnconfigure(1, weight=1)
    sota_frame.grid_columnconfigure(2, weight=1)

    tk.Label(sota_frame, text="Ref no.:", font=('Arial', 10), width=12).grid(row=0, column=0, padx=5, pady=2, sticky='e')
    station_sota_entry = tk.Entry(sota_frame, textvariable=station_sota, font=('Arial', 14, 'bold'), width=12)
    station_sota_entry.grid(row=0, column=1, padx=5, pady=2, sticky='w')
    station_sota_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
    station_sota_entry.bind("<FocusOut>", lambda e: auto_fill_sota_prefix(station_sota, station_sota_name, None, None, combobox=station_sota_matches))

    station_sota_matches = ttk.Combobox(sota_frame, state="readonly", font=('Arial', 10), width=20)
    station_sota_matches.grid(row=0, column=2, padx=5, pady=2, sticky='ew')
    station_sota_matches.bind("<<ComboboxSelected>>", on_sota_select)

    tk.Label(sota_frame, text="Summit name:", font=('Arial', 10)).grid(row=1, column=0, padx=5, sticky='w')
    tk.Label(sota_frame, textvariable=station_sota_name, font=('Arial', 10, 'bold')).grid(row=1, column=1, columnspan=2, sticky='w')

    # ---------- WLOTA tab ----------
    wlota_frame = tk.Frame(notebook, background='#f0f0f0')
    notebook.add(wlota_frame, text="WLOTA")

    wlota_frame.grid_columnconfigure(1, weight=1)

    tk.Label(wlota_frame, text="Reference No.:", font=('Arial', 10)).grid(row=0, column=0, padx=10, pady=1, sticky='w')
    wlota_entry = tk.Entry(wlota_frame, textvariable=station_wlota, font=('Arial', 10, 'bold'))
    wlota_entry.grid(row=0, column=1, padx=10, pady=5, sticky='ew')

    auto_fill_pota_prefix(station_pota, station_pota_name, station_callsign)
    auto_fill_wwff_prefix(station_wwff, station_wwff_name, station_wwff_lat, station_wwff_long, station_callsign)
    auto_fill_bota_prefix(station_bota, station_bota_name, station_bota_lat, station_bota_long, station_callsign)
    auto_fill_iota_prefix(station_iota, station_iota_name, station_iota_lat, station_iota_long, station_callsign)
    auto_fill_sota_prefix(station_sota, station_sota_name, None, None, station_sota)
    auto_fill_cota_prefix(station_cota, station_cota_name, station_callsign)

    for frame in (pota_frame, wwff_frame, bota_frame):
        frame.grid_columnconfigure(0, minsize=100)
        frame.grid_columnconfigure(1, weight=1)

    # ==================================================
    # QRZ / eQSL Notebook (Upload settings)
    # ==================================================
    upload_notebook = ttk.Notebook(Station_Setup_Window)
    upload_notebook.grid(row=4, column=0, padx=10, pady=5, sticky="ew")

    # ---------- QRZ Upload tab ----------
    qrz_tab = tk.Frame(upload_notebook, background='#f0f0f0')
    upload_notebook.add(qrz_tab, text="QRZ Upload")

    lf_qrz = tk.LabelFrame(qrz_tab, text="QRZ Upload Settings", font=('Arial', 10, 'bold'))
    lf_qrz.pack(fill="x", padx=10, pady=10)
    lf_qrz.grid_columnconfigure(1, weight=1)

    tk.Checkbutton(lf_qrz, text="Upload to QRZ", variable=station_upload_qrz).grid(row=0, column=0, columnspan=2, sticky="w", padx=10)

    tk.Label(lf_qrz, text="QRZ API Key:", font=('Arial', 10)).grid(row=1, column=0, padx=10, pady=1, sticky='w')
    qrzapi_entry = tk.Entry(lf_qrz, textvariable=station_qrzapi, font=('Arial', 10, 'bold'), show="*")
    qrzapi_entry.grid(row=1, column=1, padx=(10, 100), pady=5, sticky='ew')

    def toggle_qrz_visibility():
        if qrzapi_entry.cget('show') == '*':
            qrzapi_entry.config(show='')
            toggle_button.config(text='Hide')
        else:
            qrzapi_entry.config(show='*')
            toggle_button.config(text='Show')

    toggle_button = tk.Button(lf_qrz, text='Show', command=toggle_qrz_visibility, width=10)
    toggle_button.place(in_=qrzapi_entry, relx=1.0, x=5, y=-5, anchor='nw')

    tk.Label(
        lf_qrz,
        text="API key must match Callsign.\nFormat: XXXX-XXXX-XXXX-XXXX",
        font=('Arial', 8, "bold")
    ).grid(row=2, column=0, columnspan=2, padx=10, pady=5, sticky='w')

    def close_window():
        global Station_Setup_Window
        Station_Setup_Window.destroy()
        Station_Setup_Window = None

    def save_setup():
        if not is_valid_locator(station_locator.get().strip()):
            messagebox.showerror("Invalid Locator", "The Maidenhead locator must be at least 4 characters and valid.\nExample: FN31 or FN31TK")
            return

        station_callsign_var.set(station_callsign.get())
        station_operator_var.set(station_operator.get())
        station_locator_var.set(station_locator.get())
        station_location_var.set(station_location.get())

        station_name_var.set(station_name.get())
        station_street_var.set(station_street.get())
        station_postalcode_var.set(station_postalcode.get())
        station_city_var.set(station_city.get())
        station_county_var.set(station_county.get())
        station_country_var.set(station_country.get())
        station_cqzone_var.set(station_cqzone.get())
        station_ituzone_var.set(station_ituzone.get())

        station_contest_var.set(station_contest.get())

        station_pota_var.set(station_pota.get())
        station_wwff_var.set(station_wwff.get())
        station_bota_var.set(station_bota.get())
        station_iota_var.set(station_iota.get())
        station_sota_var.set(station_sota.get())
        station_wlota_var.set(station_wlota.get())
        station_cota_var.set(station_cota.get())

        station_qrzapi_var.set(station_qrzapi.get())
        upload_qrz_var.set(station_upload_qrz.get())

        save_station_setup()
        close_window()

    def cancel_setup():
        close_window()

    tk.Button(Station_Setup_Window, text="Save & Exit", command=save_setup, width=10, height=2).grid(row=5, column=0, padx=20, pady=10, sticky="w")
    tk.Button(Station_Setup_Window, text="Cancel", command=cancel_setup, width=10, height=2).grid(row=5, column=0, padx=20, pady=10, sticky="e")

    Station_Setup_Window.protocol("WM_DELETE_WINDOW", close_window)

    # ----------------------------------------------------------------------
    # NOW autosize → freeze → CENTER
    # ----------------------------------------------------------------------
    Station_Setup_Window.update_idletasks()
    Station_Setup_Window.resizable(False, False)

    w = Station_Setup_Window.winfo_width()
    h = Station_Setup_Window.winfo_height()

    center_window_over(root, Station_Setup_Window, w, h)



#########################################################################################
#   ___ ___  _  _ ___ ___ ___   _    ___   _   ___    __       ___   ___   _____ 
#  / __/ _ \| \| | __|_ _/ __| | |  / _ \ /_\ |   \  / _|___  / __| /_\ \ / / __|
# | (_| (_) | .` | _| | | (_ | | |_| (_) / _ \| |) | > _|_ _| \__ \/ _ \ V /| _| 
#  \___\___/|_|\_|_| |___\___| |____\___/_/ \_\___/  \_____|  |___/_/ \_\_/ |___|
#
#########################################################################################
def open_preferences():
    global Preference_Window, utc_offset_var, hamlib_ip_var, hamlib_port_var, exit_without_confirm

    if Preference_Window is not None and Preference_Window.winfo_exists():
        Preference_Window.lift()
        return

    # -----------------------------------------------------------
    # WINDOW
    # -----------------------------------------------------------
    Preference_Window = tk.Toplevel(root)
    Preference_Window.title("Preferences")
    Preference_Window.resizable(False, False)

    win_w = 450
    win_h = 500
    Preference_Window.geometry(f"{win_w}x{win_h}")

    Preference_Window.transient(root)
    Preference_Window.grab_set()
    Preference_Window.focus_set()
    Preference_Window.lift()

    Preference_Window.after(50, lambda: center_window_over(root, Preference_Window, win_w, win_h))

    # -----------------------------------------------------------
    # NOTEBOOK TABS
    # -----------------------------------------------------------
    nb = ttk.Notebook(Preference_Window)
    nb.grid(row=0, column=0, columnspan=2, sticky="nsew", padx=5, pady=5)

    tab_general = tk.Frame(nb)
    tab_qrz     = tk.Frame(nb)
    tab_web     = tk.Frame(nb)

    nb.add(tab_general, text="General")
    nb.add(tab_qrz, text="QRZ Lookup")
    nb.add(tab_web, text="WordPress Dashboard")

    # -----------------------------------------------------------
    # LOAD SETTINGS
    # -----------------------------------------------------------
    hamlib_port_var = tk.StringVar(value=config.get('hamlib_settings', 'hamlib_port', fallback="4532"))
    hamlib_ip_var   = tk.StringVar(value=config.get('hamlib_settings', 'hamlib_ip', fallback="127.0.0.1"))
    udp_adif_port   = config.get('UDP_ADIF_Settings', 'udp_adif_port', fallback="2333")

    # ====================================================================
    # ======================= TAB GENERAL ================================
    # ====================================================================

    # Reload last logbook
    lf_reload = tk.LabelFrame(tab_general, text="Reload last logbook on startup", font=('Arial', 10, 'bold'))
    lf_reload.pack(fill="x", padx=10, pady=5)

    reload_last_logbook_var = tk.BooleanVar(value=config.getboolean('General', 'reload_last_logbook', fallback=False))
    tk.Label(lf_reload, text="Enable Reload:").grid(row=0, column=0, sticky="w")
    tk.Checkbutton(lf_reload, variable=reload_last_logbook_var).grid(row=0, column=1, sticky="w")

    # UTC
    lf_utc = tk.LabelFrame(tab_general, text="UTC Time Offset", font=('Arial', 10, 'bold'))
    lf_utc.pack(fill="x", padx=10, pady=5)

    tk.Label(lf_utc, text="Offset (Hours):").grid(row=0, column=0, sticky="w")
    utc_offset_var = tk.StringVar(value=config.get('Global_settings', 'utc_offset', fallback='0'))
    utc_offset_menu = ttk.Combobox(lf_utc, textvariable=utc_offset_var, values=[str(i) for i in range(-12, 13)], width=5)
    utc_offset_menu.grid(row=0, column=1, sticky="w")

    utc_auto_var = tk.BooleanVar(value=config.getboolean('Global_settings', 'utc_auto', fallback=True))
    tk.Checkbutton(lf_utc, text="Automatic UTC offset correction", variable=utc_auto_var).grid(row=1, column=0, columnspan=2, sticky="w")

    def toggle_utc_offset_state(*args):
        utc_offset_menu.configure(state="disabled" if utc_auto_var.get() else "readonly")
    utc_auto_var.trace_add("write", toggle_utc_offset_state)
    toggle_utc_offset_state()

    # HAMLIB
    lf_hamlib = tk.LabelFrame(tab_general, text="Hamlib Setup", font=('Arial', 10, 'bold'))
    lf_hamlib.pack(fill="x", padx=10, pady=5)

    enable_radio_var = tk.BooleanVar(value=config.getboolean('hamlib_settings', 'enable_radio', fallback=False))
    tk.Checkbutton(lf_hamlib, text="Enable Radio Control", variable=enable_radio_var).grid(row=0, column=0, sticky="w")

    tk.Label(lf_hamlib, text="Port:").grid(row=1, column=0, sticky="w")
    server_port_frame = tk.Frame(lf_hamlib)
    for value in ["4532", "4536", "4538", "4540"]:
        tk.Radiobutton(server_port_frame, text=value, variable=hamlib_port_var, value=value).pack(side=tk.LEFT)
    server_port_frame.grid(row=1, column=1, sticky="w")

    tk.Label(lf_hamlib, text="IP-address:").grid(row=2, column=0, sticky="w")
    tk.Entry(lf_hamlib, textvariable=hamlib_ip_var).grid(row=2, column=1, sticky="w")

    # UDP ADIF
    lf_udp = tk.LabelFrame(tab_general, text="UDP Receive ADIF QSO Record", font=('Arial', 10, 'bold'))
    lf_udp.pack(fill="x", padx=10, pady=5)

    udp_adif_var = tk.StringVar(value=udp_adif_port)
    tk.Label(lf_udp, text="Port:").grid(row=0, column=0, sticky="w")
    tk.Entry(lf_udp, textvariable=udp_adif_var, width=10).grid(row=0, column=1, sticky="w")

    # BACKUP FOLDER
    lf_backup = tk.LabelFrame(tab_general, text="Backup Folder", font=('Arial', 10, 'bold'))
    lf_backup.pack(fill="x", padx=10, pady=5)

    backup_folder_var = tk.StringVar(value=config.get("General", "backup_folder", fallback=""))
    backup_entry = tk.Entry(lf_backup, textvariable=backup_folder_var, width=40)
    backup_entry.grid(row=0, column=1, sticky="w")

    def choose_backup_folder():
        folder = filedialog.askdirectory(title="Select Backup Folder")
        if folder:
            backup_folder_var.set(folder)

    tk.Button(lf_backup, text="Browse", command=choose_backup_folder).grid(row=0, column=0, sticky="e", padx=5)

    # Exit confirmation
    lf_exit = tk.LabelFrame(tab_general, text="Exit Program Behavior", font=('Arial', 10, 'bold'))
    lf_exit.pack(fill="x", padx=10, pady=5)

    exit_without_confirm_var = tk.BooleanVar(value=config.getboolean("General", "exit_without_confirm", fallback=False))
    tk.Label(lf_exit, text="Exit without confirmation:").grid(row=0, column=0, sticky="w")
    tk.Checkbutton(lf_exit, variable=exit_without_confirm_var).grid(row=0, column=1, sticky="w")

    # ====================================================================
    # ======================= TAB QRZ ====================================
    # ====================================================================
    lf_qrz = tk.LabelFrame(tab_qrz, text="QRZ Lookup Settings", font=('Arial', 10, 'bold'))
    lf_qrz.pack(fill="x", padx=10, pady=10)

    use_qrz_lookup_var = tk.BooleanVar(value=config.getboolean("QRZ", "use_qrz_lookup", fallback=False))
    tk.Label(lf_qrz, text="Use QRZ Lookup:").grid(row=0, column=0, sticky="w")
    tk.Checkbutton(lf_qrz, variable=use_qrz_lookup_var).grid(row=0, column=1, sticky="w")

    qrz_username_var = tk.StringVar(value=config.get("QRZ", "username", fallback=""))
    qrz_password_var = tk.StringVar(value=config.get("QRZ", "password", fallback=""))

    tk.Label(lf_qrz, text="Username:").grid(row=1, column=0, sticky="e")
    tk.Label(lf_qrz, text="Password:").grid(row=2, column=0, sticky="e")

    qrz_username_entry = tk.Entry(lf_qrz, textvariable=qrz_username_var)
    qrz_password_entry = tk.Entry(lf_qrz, textvariable=qrz_password_var, show="*")

    qrz_username_entry.grid(row=1, column=1, sticky="w")
    qrz_password_entry.grid(row=2, column=1, sticky="w")

    def toggle_password():
        if qrz_password_entry.cget("show") == "*":
            qrz_password_entry.config(show="")
            pw_button.config(text="Hide")
        else:
            qrz_password_entry.config(show="*")
            pw_button.config(text="Show")

    # <-- HIER AANGEPAST: dezelfde stijl als API-key knop -->
    pw_button = tk.Button(lf_qrz, text="Show", command=toggle_password, width=6)
    pw_button.grid(row=2, column=2, sticky="w")

    def toggle_qrz(*args):
        state = "normal" if use_qrz_lookup_var.get() else "disabled"
        qrz_username_entry.config(state=state)
        qrz_password_entry.config(state=state)
        pw_button.config(state=state)

    use_qrz_lookup_var.trace_add("write", toggle_qrz)
    toggle_qrz()

    # ====================================================================
    # ======================= TAB WORDPRESS ==============================
    # ====================================================================

    minibook_web_url_var.set(config.get("Web_Dashboard", "web_url", fallback=""))
    minibook_web_id_var.set(config.get("Web_Dashboard", "web_id", fallback=""))
    minibook_web_api_var.set(config.get("Web_Dashboard", "api_key", fallback=""))
    minibook_web_enabled_var.set(config.getboolean("Web_Dashboard", "enable", fallback=False))

    lf_web = tk.LabelFrame(tab_web, text="MiniBook WordPress Dashboard", font=('Arial', 10, 'bold'))
    lf_web.pack(fill="x", padx=10, pady=10)

    tk.Label(lf_web, text="Web URL:").grid(row=0, column=0, sticky="w")
    tk.Entry(lf_web, textvariable=minibook_web_url_var, width=40).grid(row=0, column=1, sticky="w")

    tk.Label(lf_web, text="ID:").grid(row=1, column=0, sticky="w")
    tk.Entry(lf_web, textvariable=minibook_web_id_var, width=20).grid(row=1, column=1, sticky="w")

    tk.Label(lf_web, text="API Key:").grid(row=2, column=0, sticky="w")
    api_entry = tk.Entry(lf_web, textvariable=minibook_web_api_var, width=40, show="*")
    api_entry.grid(row=2, column=1, sticky="w")

    def toggle_api():
        if api_entry.cget("show") == "*":
            api_entry.config(show="")
            api_btn.config(text="Hide")
        else:
            api_entry.config(show="*")
            api_btn.config(text="Show")

    api_btn = tk.Button(lf_web, text="Show", width=6, command=toggle_api)
    api_btn.grid(row=2, column=2, sticky="w")

    tk.Label(lf_web, text="Update interval (sec):").grid(row=3, column=0, sticky="w")
    tk.Entry(lf_web, textvariable=minibook_web_interval_var, width=10).grid(row=3, column=1, sticky="w")

    tk.Checkbutton(lf_web, text="Enable updates", variable=minibook_web_enabled_var).grid(row=4, column=0, columnspan=2, sticky="w")

    # ====================================================================
    # VALIDATIE HELPERS
    # ====================================================================
    def is_valid_ip(ip):
        try:
            ipaddress.ip_address(ip)
            return True
        except:
            return False

    def is_valid_port(port):
        try:
            return 0 <= int(port) <= 65535
        except:
            return False

    # ====================================================================
    # SAVE HANDLER
    # ====================================================================
    def save_preferences():
        old_enable = config.getboolean('hamlib_settings', 'enable_radio', fallback=False)

        config['General']['reload_last_logbook']     = str(reload_last_logbook_var.get())
        config['Global_settings']['utc_offset']      = utc_offset_var.get()
        config['Global_settings']['utc_auto']        = str(utc_auto_var.get())
        config['General']['exit_without_confirm']    = str(exit_without_confirm_var.get())
        globals()['exit_without_confirm']            = exit_without_confirm_var.get()

        # HAMLIB
        config['hamlib_settings']['enable_radio'] = str(enable_radio_var.get())
        config['hamlib_settings']['hamlib_port']  = hamlib_port_var.get()
        config['hamlib_settings']['hamlib_ip']    = hamlib_ip_var.get()

        # UDP
        config["UDP_ADIF_Settings"]['udp_adif_port'] = udp_adif_var.get()

        # QRZ
        config['QRZ']['use_qrz_lookup'] = str(use_qrz_lookup_var.get())
        config['QRZ']['username']       = qrz_username_var.get().strip()
        config['QRZ']['password']       = qrz_password_var.get().strip()

        # Web Dashboard
        config['Web_Dashboard']['enable']          = str(minibook_web_enabled_var.get())
        config['Web_Dashboard']['web_url']         = minibook_web_url_var.get().strip()
        config['Web_Dashboard']['web_id']          = minibook_web_id_var.get().strip()
        config['Web_Dashboard']['api_key']         = minibook_web_api_var.get().strip()
        config['Web_Dashboard']['update_interval'] = minibook_web_interval_var.get().strip()

        # Backup
        config['General']['backup_folder'] = backup_folder_var.get().strip()

        # SAVE FILE
        with open(CONFIG_FILE, 'w') as cfgfile:
            config.write(cfgfile)

        update_datetime()
        restart_listener(config)

        new_enable = enable_radio_var.get()

        if old_enable and not new_enable:
            hamlib_logger.info("Hamlib disabled → disconnecting")
            disconnect_from_hamlib()

        elif new_enable and not old_enable:
            hamlib_logger.info("Hamlib enabled → starting engine")
            start_hamlib_engine()

        elif new_enable and old_enable:
            hamlib_logger.info("Hamlib settings changed → reconnecting")
            hamlib_request_connect(force=True)

        Preference_Window.destroy()

    # ====================================================================
    # SAVE / CANCEL BUTTONS
    # ====================================================================
    tk.Button(
        Preference_Window,
        text="Save & Exit",
        width=10,
        height=2,
        command=lambda: (
            messagebox.showerror("Error", f"{udp_adif_var.get()} is an invalid port.")
            if not is_valid_port(udp_adif_var.get()) else

            messagebox.showerror("Error", f"{hamlib_ip_var.get()} is an invalid IP address.")
            if not is_valid_ip(hamlib_ip_var.get()) else

            messagebox.showerror("Error", "Update interval must be at least 5 seconds.")
            if (not minibook_web_interval_var.get().isdigit()
                or int(minibook_web_interval_var.get()) < 5) else

            save_preferences()
        )
    ).grid(row=90, column=0, padx=20, pady=10, sticky="w")

    tk.Button(
        Preference_Window,
        text="Cancel",
        width=10,
        height=2,
        command=Preference_Window.destroy
    ).grid(row=90, column=1, padx=20, pady=10, sticky="e")

    Preference_Window.protocol("WM_DELETE_WINDOW", Preference_Window.destroy)









def get_current_utc_time():
    """Return UTC time based on automatic or manual offset setting."""
    try:
        if config.getboolean("General", "utc_auto", fallback=True):
            # Automatic correction based on system timezone
            now_local = datetime.now()
            now_utc = datetime.utcnow()
            offset_hours = round((now_local - now_utc).total_seconds() / 3600)
        else:
            offset_hours = int(config.get("General", "utc_offset", fallback="0"))
        corrected_time = datetime.utcnow() + timedelta(hours=offset_hours)
        return corrected_time
    except Exception as e:
        print(f"UTC time calculation error: {e}")
        return datetime.utcnow()


#########################################################################################
#   ___ ___  _  _ ___ ___ ___   _    ___   _   ___    __       ___   ___   _____ 
#  / __/ _ \| \| | __|_ _/ __| | |  / _ \ /_\ |   \  / _|___  / __| /_\ \ / / __|
# | (_| (_) | .` | _| | | (_ | | |_| (_) / _ \| |) | > _|_ _| \__ \/ _ \ V /| _| 
#  \___\___/|_|\_|_| |___\___| |____\___/_/ \_\___/  \_____|  |___/_/ \_\_/ |___|
#
#########################################################################################
def load_config():
    """
    Load parameters from config.ini and ensure every section and key is explicitly checked.
    If the configuration file does not exist, create it with default values.
    Also migrates old [Wsjtx_settings] to [UDP_ADIF_Settings].
    """
    global exit_without_confirm

    file_path = CONFIG_FILE

    # ----------------------------------------------------------
    # CREATE CONFIG IF NOT EXISTS
    # ----------------------------------------------------------
    if not os.path.exists(file_path):
        print("Configuration file not found. Creating a new one with default values.")
        config['Global_settings'] = {}
        config['General'] = {}
        config['hamlib_settings'] = {}
        config['UDP_ADIF_Settings'] = {}
        config['QRZ'] = {}
        config['Web_Dashboard'] = {}

    config.read(file_path)

    # ----------------------------------------------------------
    # MIGRATION: [Wsjtx_settings] → [UDP_ADIF_Settings]
    # ----------------------------------------------------------
    migration_done = False
    try:
        if config.has_section('Wsjtx_settings'):
            print("🔄 Detected old [Wsjtx_settings] — converting to [UDP_ADIF_Settings]...")

            if not config.has_section('UDP_ADIF_Settings'):
                config.add_section('UDP_ADIF_Settings')

            for key, value in config.items('Wsjtx_settings'):
                if key.lower() == 'wsjtx_port':
                    config.set('UDP_ADIF_Settings', 'udp_adif_port', value)
                else:
                    config.set('UDP_ADIF_Settings', key, value)

            config.remove_section('Wsjtx_settings')
            migration_done = True

        if config.has_section('UDP_ADIF_Settings') and config.has_option('UDP_ADIF_Settings', 'wsjtx_port'):
            config.remove_option('UDP_ADIF_Settings', 'wsjtx_port')
            migration_done = True

        if migration_done:
            with open(file_path, 'w', encoding='utf-8') as cfgfile:
                config.write(cfgfile)

    except Exception as e:
        print(f"⚠️ Error while migrating configuration: {e}")

    # ----------------------------------------------------------
    # GLOBAL SETTINGS
    # ----------------------------------------------------------
    if 'Global_settings' not in config:
        config.add_section('Global_settings')
    if 'utc_offset' not in config['Global_settings']:
        config['Global_settings']['utc_offset'] = '0'
    if 'utc_auto' not in config['Global_settings']:
        config['Global_settings']['utc_auto'] = 'True'

    # ----------------------------------------------------------
    # GENERAL SETTINGS
    # ----------------------------------------------------------
    if 'General' not in config:
        config.add_section('General')
    if 'reload_last_logbook' not in config['General']:
        config['General']['reload_last_logbook'] = 'False'
    if 'last_loaded_logbook' not in config['General']:
        config['General']['last_loaded_logbook'] = ''

    if 'backup_folder' not in config['General']:
        os.makedirs(BACKUP_FOLDER, exist_ok=True)
        config['General']['backup_folder'] = str(BACKUP_FOLDER)


    if 'exit_without_confirm' not in config['General']:
        config['General']['exit_without_confirm'] = 'False'

    # ----------------------------------------------------------
    # HAMLIB SETTINGS
    # ----------------------------------------------------------
    if 'hamlib_settings' not in config:
        config.add_section('hamlib_settings')

    if 'hamlib_port' not in config['hamlib_settings']:
        config['hamlib_settings']['hamlib_port'] = '4532'

    if 'hamlib_ip' not in config['hamlib_settings']:
        config['hamlib_settings']['hamlib_ip'] = '127.0.0.1'

    if 'enable_radio' not in config['hamlib_settings']:
        config['hamlib_settings']['enable_radio'] = 'False'

    # ----------------------------------------------------------
    # UDP ADIF SETTINGS
    # ----------------------------------------------------------
    if 'UDP_ADIF_Settings' not in config:
        config.add_section('UDP_ADIF_Settings')

    if 'udp_adif_port' not in config['UDP_ADIF_Settings']:
        config['UDP_ADIF_Settings']['udp_adif_port'] = '2333'

    # ----------------------------------------------------------
    # QRZ SETTINGS
    # ----------------------------------------------------------
    if 'QRZ' not in config:
        config.add_section('QRZ')

    if 'username' not in config['QRZ']:
        config['QRZ']['username'] = ''

    if 'password' not in config['QRZ']:
        config['QRZ']['password'] = ''

    if 'use_qrz_lookup' not in config['QRZ']:
        config['QRZ']['use_qrz_lookup'] = 'False'

    # ----------------------------------------------------------
    # WEB DASHBOARD SETTINGS (NEW: now includes API KEY)
    # ----------------------------------------------------------
    if not config.has_section("Web_Dashboard"):
        config.add_section("Web_Dashboard")

    if not config.has_option("Web_Dashboard", "enable"):
        config.set("Web_Dashboard", "enable", "False")

    if not config.has_option("Web_Dashboard", "web_url"):
        config.set("Web_Dashboard", "web_url", "")

    if not config.has_option("Web_Dashboard", "web_id"):
        config.set("Web_Dashboard", "web_id", "")

    # NEW → API KEY support
    if not config.has_option("Web_Dashboard", "api_key"):
        config.set("Web_Dashboard", "api_key", "")

    if not config.has_option("Web_Dashboard", "update_interval"):
        config.set("Web_Dashboard", "update_interval", "30")



    # ----------------------------------------------------------
    # SAVE UPDATED CONFIG
    # ----------------------------------------------------------
    with open(file_path, 'w', encoding='utf-8') as configfile:
        config.write(configfile)

    # NEW — load into global memory variable
    exit_without_confirm = config.getboolean('General', 'exit_without_confirm', fallback=False)




def load_station_setup():
    log.info("LOAD_STATION_SETUP called")

    # Load station details from JSON logbook if the file is loaded
    if CURRENT_JSON_FILE and os.path.exists(CURRENT_JSON_FILE):
        try:
            with open(CURRENT_JSON_FILE, 'r', encoding='utf-8') as file:
                logbook_data = json.load(file)

            log.debug("Loaded logbook JSON for reading station info")

            station_info = logbook_data.get("Station", {})

            station_operator_var.set(station_info.get("Operator", ""))
            station_callsign_var.set(station_info.get("Callsign", ""))
            station_locator_var.set(station_info.get("Locator", ""))
            station_location_var.set(station_info.get("Location", ""))

            station_name_var.set(station_info.get("Name", ""))
            station_street_var.set(station_info.get("Street", ""))
            station_postalcode_var.set(station_info.get("Postalcode", ""))
            station_city_var.set(station_info.get("City", ""))
            station_county_var.set(station_info.get("County", ""))
            station_country_var.set(station_info.get("Country", ""))
            station_cqzone_var.set(station_info.get("CQ Zone", ""))
            station_ituzone_var.set(station_info.get("ITU Zone", ""))

            station_contest_var.set(station_info.get("Contest", ""))

            station_wwff_var.set(station_info.get("WWFF", ""))
            station_pota_var.set(station_info.get("POTA", ""))
            station_bota_var.set(station_info.get("BOTA", ""))
            station_iota_var.set(station_info.get("IOTA", ""))
            station_sota_var.set(station_info.get("SOTA", ""))
            station_wlota_var.set(station_info.get("WLOTA", ""))
            station_cota_var.set(station_info.get("COTA", ""))
            station_qrzapi_var.set(station_info.get("QRZAPI", ""))
            upload_qrz_var.set(bool(station_info.get("QRZUpload", False)))

            station_callsign_entry.config(textvariable=station_callsign_var)
            station_operator_entry.config(textvariable=station_operator_var)
            station_locator_entry.config(textvariable=station_locator_var)
            station_location_entry.config(textvariable=station_location_var)
            station_wwff_entry.config(textvariable=station_wwff_var)
            station_pota_entry.config(textvariable=station_pota_var)
            station_bota_entry.config(textvariable=station_bota_var)
            station_iota_entry.config(textvariable=station_iota_var)
            station_sota_entry.config(textvariable=station_sota_var)
            station_wlota_entry.config(textvariable=station_wlota_var)
            station_cota_entry.config(textvariable=station_cota_var)

            log.info("LOAD_STATION_SETUP finished successfully")

        except (json.JSONDecodeError, KeyError) as e:
            log.exception("LOAD_STATION_SETUP failed")
            messagebox.showerror("Error", f"Failed to load station details from logbook: {e}")
    else:
        log.warning("LOAD_STATION_SETUP aborted: file missing or not loaded")





def save_station_setup():
    log.info("SAVE_STATION_SETUP called")

    if not CURRENT_JSON_FILE:
        log.warning("SAVE_STATION_SETUP aborted: no CURRENT_JSON_FILE")
        return

    try:
        with open(CURRENT_JSON_FILE, 'r', encoding='utf-8') as f:
            json_data = json.load(f)

        log.debug("Loaded CURRENT_JSON_FILE for writing station info")

        # Only update Station section
        json_data["Station"] = {
            "Callsign": station_callsign_var.get().upper(),
            "Operator": station_operator_var.get().upper(),
            "Locator": station_locator_var.get().upper(),
            "Location": station_location_var.get(),
            "Name": station_name_var.get(),
            "Street": station_street_var.get(),
            "Postalcode": station_postalcode_var.get(),
            "City": station_city_var.get(),
            "County": station_county_var.get(),
            "Country": station_country_var.get(),
            "CQ Zone": station_cqzone_var.get(),
            "ITU Zone": station_ituzone_var.get(),
            "Contest": station_contest_var.get(),
            "WWFF": station_wwff_var.get(),
            "POTA": station_pota_var.get(),
            "BOTA": station_bota_var.get(),
            "IOTA": station_iota_var.get(),
            "SOTA": station_sota_var.get(),
            "WLOTA": station_wlota_var.get(),
            "COTA": station_cota_var.get(),
            "QRZAPI": station_qrzapi_var.get(),
            "QRZUpload": upload_qrz_var.get()
        }

        log.debug("Station setup values collected and updated into JSON data")

        # Update QSO cache:
        qso_lines[:] = json_data.get("Logbook", [])

        save_async(json_data)
        log.info("SAVE_STATION_SETUP finished successfully")

    except Exception as e:
        log.exception("SAVE_STATION_SETUP failed")
        messagebox.showerror("Error", f"Failed to save station details: {e}")




















#########################################################################################
#   ___  ___  ____  _    ___   ___  _  ___   _ ___ 
#  / _ \| _ \|_  / | |  / _ \ / _ \| |/ / | | | _ \
# | (_) |   / / /  | |_| (_) | (_) | ' <| |_| |  _/
#  \__\_\_|_\/___| |____\___/ \___/|_|\_\\___/|_|  
##
#########################################################################################

LOG_QRZ_CALL_DATA = True   # True = ook alle QRZ gegevens loggen

def qrz_log(msg):
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")

    try:
        with open(QRZ_LOOKUP_LOG_FILE, "a", encoding="utf-8") as f:
            f.write(f"{timestamp}  {msg}\n")
    except Exception as e:
        print("QRZ LOGGING ERROR:", e)


def extract_core(callsign):
    callsign = callsign.strip().upper()
    parts = callsign.split('/')
    if len(parts) == 3:
        return parts[1]  # prefix/MainCallsign/suffix
    elif len(parts) == 2:
        # suffix = only letters (like /P of /M or /POTA)
        if re.fullmatch(r'[A-Z]+', parts[1]):
            return parts[0]
        else:
            return parts[1]
    return callsign


def get_session_key(username, password):
    encoded_username = urllib.parse.quote(username)
    encoded_password = urllib.parse.quote(password)

    qrz_log(f"START LOGIN with QRZ credentials username={username}")

    url = f"https://xmldata.qrz.com/xml/current/?username={encoded_username}&password={encoded_password}"

    try:
        response = requests.get(url, timeout=5)
        root_xml = ET.fromstring(response.content)
        ns = {"qrz": "http://xmldata.qrz.com"}

        key = root_xml.findtext(".//qrz:Key", namespaces=ns)
        error = root_xml.findtext(".//qrz:Error", namespaces=ns)

        if key:
            qrz_log(f"LOGIN SUCCESS session_key={key}")
        if error:
            qrz_log(f"LOGIN ERROR={error}")

        return key, error

    except requests.exceptions.RequestException as e:
        qrz_log(f"LOGIN FAILED: {e}")
        return None, "❌ Connection QRZ XML Failed"



def open_qrz_link(event, callsign):
    url = f"https://www.qrz.com/db/{callsign}"
    webbrowser.open_new_tab(url)


def query_callsign(session_key, callsign):
    import re
    ns = {"qrz": "http://xmldata.qrz.com"}

    def do_query(cs):
        qrz_log(f"LOOKUP start callsign={cs}")
        url = f"https://xmldata.qrz.com/xml/current/?s={session_key}&callsign={cs}"
        try:
            response = requests.get(url, timeout=5)
            root = ET.fromstring(response.content)
            qrz_log(f"LOOKUP success callsign={cs}")
            return root.find(".//qrz:Callsign", ns)
        except Exception as e:
            qrz_log(f"LOOKUP FAILED callsign={cs}  ERROR={e}")
            return None

    # Stap 1: originele callsign
    qrz_log(f"PRIMARY lookup callsign={callsign}")
    callsign_element = do_query(callsign)

    fallback_name = ""

    # Als primary geen resultaat
    if callsign_element is None:
        base = extract_core(callsign)
        if base != callsign:
            qrz_log(f"NO MATCH for {callsign}. Trying fallback={base}")
            fallback_element = do_query(base)

            if fallback_element is not None:
                full_name = f"{fallback_element.findtext('qrz:fname', '', ns)} {fallback_element.findtext('qrz:name', '', ns)}".strip()

                qrz_log(f"FALLBACK SUCCESS base={base} name={full_name}")

                return {
                    "callsign": base,
                    "name": full_name,
                    "address": "",
                    "city": "",
                    "zipcode": "",
                    "province": "",
                    "country": "",
                    "email": "",
                    "grid": "",
                    "cq_zone": "",
                    "itu_zone": "",
                    "qslmgr": "",
                    "lat": "",
                    "lon": "",
                    "mapslink": "",
                }

        # helemaal geen match
        qrz_log(f"LOOKUP FAILED for callsign={callsign} AND fallback")
        clear_qrz_fields()
        QRZ_status_label.config(text="⚠️ No Callsign found in QRZ XML.", fg="red")
        return None

    found_call = callsign_element.findtext("qrz:call", default="Unknown", namespaces=ns)

    qrz_log(f"FOUND callsign={found_call} from QRZ")

    if LOG_QRZ_CALL_DATA:
        qrz_log("QRZ DATA fields:")
        for field in ["call", "fname", "name", "addr1", "addr2", "zip",
                      "state", "country", "email", "grid", "cqzone",
                      "ituzone", "qslmgr", "lat", "lon"]:
            txt = callsign_element.findtext(f"qrz:{field}", "", ns)
            qrz_log(f"    {field}={txt}")

    QRZ_status_label.config(text=f"🔍 Found {found_call} in QRZ lookup.", fg="blue", cursor="hand2", font=('Arial', 8, 'underline'))
    QRZ_status_label.unbind("<Button-1>")
    QRZ_status_label.bind("<Button-1>", lambda e, cs=found_call: open_qrz_link(e, cs))

    lat = callsign_element.findtext("qrz:lat", default="", namespaces=ns)
    lon = callsign_element.findtext("qrz:lon", default="", namespaces=ns)
    maps_link = f"https://www.google.com/maps?q={lat},{lon}" if lat and lon else ""

    full_name = f"{callsign_element.findtext('qrz:fname', '', ns)} {callsign_element.findtext('qrz:name', '', ns)}".strip()

    return {
        "callsign": found_call,
        "name": full_name,
        "address": callsign_element.findtext("qrz:addr1", "", ns),
        "city": callsign_element.findtext("qrz:addr2", "", ns),
        "zipcode": callsign_element.findtext("qrz:zip", "", ns),
        "province": callsign_element.findtext("qrz:state", "", ns),
        "country": callsign_element.findtext("qrz:country", "", ns),
        "email": callsign_element.findtext("qrz:email", "", ns),
        "grid": callsign_element.findtext("qrz:grid", "", ns),
        "cq_zone": callsign_element.findtext("qrz:cqzone", "", ns),
        "itu_zone": callsign_element.findtext("qrz:ituzone", "", ns),
        "qslmgr": callsign_element.findtext("qrz:qslmgr", "", ns),
        "lat": lat,
        "lon": lon,
        "mapslink": maps_link,
    }



def threaded_on_query():
    threading.Thread(target=on_query_thread, daemon=True).start()

def on_query_thread():

    # reset file at start of lookup
    with open(QRZ_LOOKUP_LOG_FILE, "w", encoding="utf-8") as f:
        f.write("")   # empty file
    
    use_qrz = config.getboolean("QRZ", "use_qrz_lookup", fallback=False)
    if not use_qrz:
        qrz_log("ABORT lookup: use_qrz_lookup=False")
        return

    username = config.get("QRZ", "username", fallback="").strip()
    password = config.get("QRZ", "password", fallback="").strip()
    callsign = qso_callsign_var.get().strip()

    qrz_log(f"START lookup thread callsign={callsign}")

    if not username or not password:
        qrz_log("ABORT lookup: missing credentials")
        root.after(0, lambda: messagebox.showwarning("Missing Credentials", "QRZ username/password not found in config.ini"))
        return

    if not callsign:
        qrz_log("ABORT lookup: empty callsign")
        return

    qrz_log(f"REQUEST session key for username={username}")
    session_key, error = get_session_key(username, password)

    if not session_key:
        qrz_log(f"LOGIN ERROR: {error}")
        if error:
            def show_error():
                QRZ_status_label.config(text=error, fg="red")
                if any(w in error.lower() for w in ["invalid", "incorrect", "not authorized"]):
                    messagebox.showerror("QRZ Login Failed", "Username or password incorrect.")
                elif "connection" in error.lower():
                    messagebox.showerror("Connection Error", "Unable to reach QRZ XML server.")
                else:
                    messagebox.showerror("QRZ Error", error)
            root.after(0, show_error)
        return

    qrz_log("SESSION KEY OK, starting QRZ XML fetch")

    data = query_callsign(session_key, callsign)
    if data is None:
        qrz_log("NO DATA RETURNED")
        return

    qrz_log(f"LOOKUP DONE callsign={data.get('callsign')}")

    def update_gui():
        qso_locator_var.set(data.get("grid", ""))
        qso_name_var.set(data.get("name", ""))
        qrz_city_var.set(data.get("city", ""))
        qrz_address_var.set(data.get("address", ""))
        qrz_zipcode_var.set(data.get("zipcode", ""))
        qrz_qsl_info_var.set(data.get("qslmgr", ""))

    root.after(0, update_gui)



def clear_qrz_fields():
    qso_locator_var.set("")
    qso_name_var.set("")
    qrz_city_var.set("")
    qrz_address_var.set("")
    qrz_zipcode_var.set("")
    qrz_qsl_info_var.set("")
    QRZ_status_label.config(text="❌ No data", fg="grey", cursor="")
    QRZ_status_label.unbind("<Button-1>")




# About Window
def show_about():
    global About_Window
    # Check if the window is already open
    if About_Window is not None and About_Window.winfo_exists():
        About_Window.lift()  # Bring the existing window to the front
        return

    About_Window = tk.Toplevel(root)
    About_Window.title("About")
    About_Window.resizable(False, False)
    About_Window.geometry("300x400")

    # Center the About_Window relative to root
    About_Window.update_idletasks()
    w = About_Window.winfo_width()

    root_x = root.winfo_x()
    root_y = root.winfo_y()
    root_w = root.winfo_width()

    x = root_x + (root_w // 2) - (w // 2)
    y = root_y

    About_Window.geometry(f"+{x}+{y}") 

    tk.Label(About_Window, text="MiniBook", font=('Arial', 20, 'bold')).pack(pady=10)
    separator = tk.Frame(About_Window, height=2, bd=0, relief='sunken', bg='gray')
    separator.pack(fill='x', pady=5, padx=10)
    tk.Label(About_Window, text=f"Version {VERSION_NUMBER}\n\nA Python based\nJSON Logbook\n\nDeveloped by:\nBjörn Pasteuning\nPD5DJ\n\nCopyright 2024", font=('Arial', 10)).pack(pady=10)

    url = "https://www.pd5dj.nl"
    link = tk.Label(About_Window, text=url, fg="blue", cursor="hand2", font=('Arial', 10))
    link.pack(pady=10)
    link.bind("<Button-1>", lambda e: webbrowser.open(url))
    
    def close_window():
        global About_Window
        About_Window.destroy()
        About_Window = None

    tk.Button(About_Window, text="Close", command=close_window).pack(pady=10)
    
    # Handle window close event
    About_Window.protocol("WM_DELETE_WINDOW", close_window)    





#########################################################################################
#  ___ _  _ ___ _____ 
# |_ _| \| |_ _|_   _|
#  | || .` || |  | |  
# |___|_|\_|___| |_|  
#                     
#########################################################################################

# Function called beginning at start of program
def init():
    global hamlib_ip_var, hamlib_port_var

    #
    load_flag_images()

    #
    entity_check()

    # Checks if cty.day exists
    ctydat_check()
    
    # 
    load_dxcc_to_pota_map()

    # 
    pota_check()

    # 
    wwffref_check()

    #
    bota_check()

    #
    iota_check()
    
    #
    sota_check()

    #
    cota_check()


    # Creating & loading of config.ini
    load_config()

    # Shows locations of folders on users OS
    log.info(f"Settings folder: {SETTINGS_FOLDER}")
    log.info(f"AppData data folder: {APPDATA_DATA_FOLDER}")
    log.info(f"App root data folder: {DATA_FOLDER}")
    log.info(f"Log folder: {LOG_FOLDER}")


    # --- NEW: initialize MiniBook Wordpress Dashboard from config --------------
    try:
        minibook_web_url_var.set(config.get("Web_Dashboard", "web_url", fallback=""))
        minibook_web_id_var.set(config.get("Web_Dashboard", "web_id", fallback=""))
        minibook_web_api_var.set(config.get("Web_Dashboard", "api_key", fallback=""))
        minibook_web_enabled_var.set(config.getboolean("Web_Dashboard", "enable", fallback=False))
        minibook_web_interval_var.set(config.get("Web_Dashboard", "update_interval", fallback="30"))
    except Exception as e:
        log.error(f"MiniBook Wordpress Dashboard: error loading config: {e}")


    no_file_loaded() # Checks if no logbook is loaded
    update_frequency_from_band() # Update Band to Frequency on Startup
    start_listener(config)
    gui_state_control(12) # Shows disconnected Hamlib status
    update_datetime()
    utc_offset_var.set(config.get('Global_settings', 'utc_offset', fallback='0'))
    hamlib_ip_var  = tk.StringVar(value=config.get('hamlib_settings', 'hamlib_ip', fallback="127.0.0.1"))
    hamlib_port_var     = tk.StringVar(value=config.get('hamlib_settings', 'hamlib_port', fallback=4532))
    callsign_entry.focus_set() # Set focus to the callsign entry field
    load_last_logbook_on_startup()
    start_hamlib_engine()



# Save geometries for root and Logbook_Window
def save_window_geometry(window, name):
    """Save window geometry + state. Correctly handles maximized windows."""
    if not window.winfo_exists():
        return

    try:
        config = configparser.ConfigParser()
        file_path = CONFIG_FILE

        if os.path.exists(file_path):
            config.read(file_path)

        if name not in config:
            config[name] = {}

        # Always store the state
        state = window.state()
        config[name]["state"] = state

        if state == "normal":
            # Store the actual normal geometry
            config[name]["geometry"] = window.geometry()
        else:
            # Window is maximized → get the NORMAL geometry
            # Tkinter stores this internally:
            normal_geom = window.wm_geometry()
            config[name]["geometry"] = normal_geom

        with open(file_path, "w") as cfg:
            config.write(cfg)

    except Exception as e:
        print("Error saving geometry:", e)



# Load geometry for a window
def load_window_geometry(window, name):
    """Restore window geometry and state correctly, including multi-monitor support."""
    file_path = CONFIG_FILE
    if not os.path.exists(file_path):
        return

    config = configparser.ConfigParser()
    config.read(file_path)

    if name not in config:
        return

    saved_state = config[name].get("state", "normal")
    saved_geometry = config[name].get("geometry", None)

    # Always restore geometry FIRST (position)
    if saved_geometry:
        try:
            window.geometry(saved_geometry)
            window.update_idletasks()   # force Tk to move it
        except Exception as e:
            print("Invalid geometry:", saved_geometry, e)

    # After geometry is set: apply window state
    if saved_state == "zoomed":
        # Only zoom AFTER the position has been applied
        window.state("zoomed")
    else:
        window.state("normal")

            

def load_window_position(window, name):
    """
    Load the position for a specific window, keeping its size unchanged.
    """
    file_path = CONFIG_FILE
    if os.path.exists(file_path):
        config = configparser.ConfigParser()
        config.read(file_path)
        if name in config and "geometry" in config[name]:
            saved_geometry = config[name]["geometry"]

            # Extract position (x, y) from saved geometry
            try:
                x, y = map(int, saved_geometry.split('+')[1:3])

                # Set only the position, not the size
                window.wm_geometry(f"+{x}+{y}")
            except (ValueError, IndexError):
                print(f"Invalid geometry format in config: {saved_geometry}")


def exit_program():
    global exit_without_confirm

    # Immediate quit mode enabled?
    if exit_without_confirm:
        disconnect_from_hamlib()

        if hamlib_process and hamlib_process.poll() is None:
            hamlib_process.terminate()
            try:
                hamlib_process.wait(timeout=5)
            except subprocess.TimeoutExpired:
                hamlib_process.kill()

        save_window_geometry(root, "MainWindow")
        if Logbook_Window is not None and Logbook_Window.winfo_exists():
            save_window_geometry(Logbook_Window, "LogbookWindow")

        root.destroy()
        sys.exit()

    # Normal mode: confirmation
    response = messagebox.askyesno("Confirmation", "Are you sure you want to quit the program?")
    if not response:
        return

    disconnect_from_hamlib()

    if hamlib_process and hamlib_process.poll() is None:
        hamlib_process.terminate()
        try:
            hamlib_process.wait(timeout=5)
        except subprocess.TimeoutExpired:
            hamlib_process.kill()

    save_window_geometry(root, "MainWindow")
    if Logbook_Window is not None and Logbook_Window.winfo_exists():
        save_window_geometry(Logbook_Window, "LogbookWindow")

    root.destroy()
    sys.exit()


        



#########################################################################################
#  ___  ___ _____ _     ___ ___ ___ ___ ___ ___ _  _  ___ ___ 
# / __|/ _ \_   _/_\   | _ \ __| __| __| _ \ __| \| |/ __| __|
# \__ \ (_) || |/ _ \  |   / _|| _|| _||   / _|| .` | (__| _| 
# |___/\___/ |_/_/ \_\ |_|_\___|_| |___|_|_\___|_|\_|\___|___|
#
#########################################################################################

def download_sota_file():
    """Download the SOTA reference file (summitslist.csv)"""
    try:
        response = requests.get(sota_url, timeout=20)
        response.raise_for_status()

        DATA_FOLDER.mkdir(parents=True, exist_ok=True)

        with open(SOTA_FILE, "wb") as f:
            f.write(response.content)

        root.after(
            0,
            lambda: messagebox.showinfo(
                "Download",
                "SOTA Reference file downloaded successfully."
            )
        )
        return True

    except Exception as e:
        root.after(
            0,
            lambda: messagebox.showerror(
                "Download Error",
                f"Error downloading SOTA Reference file:\n{e}"
            )
        )
        return False



def load_sota_file():
    """Load all SOTA summits into a dictionary: SummitCode -> summit data."""
    global sota_references
    sota_references = {}

    try:
        with open(SOTA_FILE, newline='', encoding="utf-8") as f:
            reader = csv.reader(f)
            for row in reader:
                if not row or len(row) < 10 or row[0].strip().upper() == "SUMMITCODE":
                    continue
                ref = row[0].strip().upper()
                country = row[1].strip()
                region = row[2].strip()
                name = row[3].strip()
                height = row[4].strip()
                points = row[5].strip()
                lon = row[8].strip()
                lat = row[9].strip()
                sota_references[ref] = {
                    "country": country,
                    "region": region,
                    "name": name,
                    "height": height,
                    "points": points,
                    "lon": lon,
                    "lat": lat
                }
        print(f"{len(sota_references)} summits geladen.")
    except Exception as e:
        messagebox.showerror("SOTA Error", f"Error loading SOTA references:\n{e}")
        sota_references = {}


def sota_check():
    """Check if summitslist.csv exists, download if missing, then load into dict."""
    global sota_references
    sota_references = {}

    try:
        if not os.path.exists(SOTA_FILE):
#            messagebox.showinfo("SOTA Reference File Not Found",
 #                               f"The file {SOTA_FILE} was not found. It will now be downloaded.")
            if not download_sota_file():
                return

        load_sota_file()

    except Exception as e:
        messagebox.showerror("SOTA Error", f"Unexpected error:\n{e}")
        sota_references = {}


def normalize_ref(ref):
    parts = re.findall(r'[A-Z]+|\d+', ref.upper())
    normalized = ''
    for p in parts:
        if p.isdigit():
            normalized += str(int(p))
        else:
            normalized += p
    return normalized

def find_sota_matches(input_text):
    input_norm = normalize_ref(input_text)

    letters = "".join(re.findall(r"[A-Z]+", input_norm))
    digits = "".join(re.findall(r"\d+", input_norm))

    exact_match = None
    possible_matches = []

    for ref, data in sota_references.items():
        if normalize_ref(ref) == input_norm:
            exact_match = (ref, data)
            return exact_match, []

    while letters:
        possible_matches = [
            (ref, data)
            for ref, data in sota_references.items()
            if normalize_ref(ref).startswith(letters)
            and re.search(r"-0*{}$".format(digits), ref)
        ]
        if possible_matches:
            break
        letters = letters[:-1]

    if not possible_matches and digits:
        possible_matches = [
            (ref, data)
            for ref, data in sota_references.items()
            if re.search(r"-0*{}$".format(digits), ref)
        ]

    return exact_match, possible_matches


def auto_fill_sota_prefix(sota_var, name_var, lat_var=None, lon_var=None, callsign_var=None, combobox=None):
    user_input = sota_var.get().strip()
    if not user_input:
        if combobox:
            combobox['values'] = []
            combobox.set('')
        name_var.set('')
        if lat_var: lat_var.set('')
        if lon_var: lon_var.set('')
        return

    exact, possibles = find_sota_matches(user_input)

    if exact:
        ref, data = exact
        sota_var.set(ref)
        name_var.set(data["name"])
        if lat_var: lat_var.set(data["lat"])
        if lon_var: lon_var.set(data["lon"])
        if combobox:
            combobox['values'] = []
            combobox.set('')

    elif possibles:
        refs = [f"{ref} ({data['name']})" for ref, data in possibles]
        if combobox:
            combobox['values'] = refs
            combobox.current(0)

        ref, data = possibles[0]
        sota_var.set(ref)
        name_var.set(data["name"])
        if lat_var: lat_var.set(data["lat"])
        if lon_var: lon_var.set(data["lon"])

    else:
        name_var.set('')
        if lat_var: lat_var.set('')
        if lon_var: lon_var.set('')
        if combobox:
            combobox['values'] = []
            combobox.set('')




def on_sota_select(event):
    """Wordt aangeroepen als een summit in de combobox gekozen wordt."""
    sel = sota_matches.get().strip()
    if not sel:
        return

    ref = sel.split()[0].upper()
    data = sota_references.get(ref)

    if data:
        sota_var.set(ref)                   # SummitCode
        sota_name_var.set(data["name"])     # Summit naam
        sota_lat_var.set(data["lat"])       # Latitude
        sota_long_var.set(data["lon"])      # Longitude
    else:
        # Fallback → alles leeg
        sota_var.set("")
        sota_name_var.set("")
        sota_lat_var.set("")
        sota_long_var.set("")
        


#########################################################################################
#   ___ ___ _____ _     ___ ___ ___ ___ ___ ___ _  _  ___ ___ 
#  / __/ _ \_   _/_\   | _ \ __| __| __| _ \ __| \| |/ __| __|
# | (_| (_) || |/ _ \  |   / _|| _|| _||   / _|| .` | (__| _| 
#  \___\___/ |_/_/ \_\ |_|_\___|_| |___|_|_\___|_|\_|\___|___|
#                                                            
#########################################################################################

def cota_check():
    global cota_references
    try:
        if not os.path.exists(COTA_FILE):
#           messagebox.showinfo("COTA Reference File Not Found", "The file WCA_list.csv was not found. It will now be downloaded.")
            download_cota_file()

        cota_references = {}
        with open(COTA_FILE, encoding="utf-8", newline='') as f:
            reader = csv.reader(f)
            for row in reader:
                # We expect at least 4 columns (ref + name in col 3)
                if len(row) >= 4:
                    ref = row[0].strip().upper()
                    name = row[3].strip()

                    if ref:  # save only valid lines
                        cota_references[ref] = {
                            "name": name
                        }

    except Exception as e:
        messagebox.showerror("Error", f"Error loading COTA reference list: {e}")


def download_cota_file():
    try:
        response = requests.get(cota_url)
        response.raise_for_status()

        with open(COTA_FILE, "wb") as f:
            f.write(response.content)

        root.after(0, lambda: messagebox.showinfo("Download", "The file WCA_list.csv has been downloaded successfully."))

    except Exception as e:
        root.after(0, lambda: messagebox.showerror("Download Error",f"Failed to download WCA_list.csv:\n{e}"))



def auto_fill_cota_prefix(cota_var, cota_name_var, qso_callsign_var):
    """
    Auto-fill COTA/COTA reference and castle name based on callsign and numeric part.
    """
    raw_val = cota_var.get().strip()
    call = qso_callsign_var.get().strip().upper()

    # Extract only digits
    cota_digits = re.sub(r"\D", "", raw_val)
    cota_digits = cota_digits.lstrip("0")  # prevent leading zero to "0"

    land_prefix = None
    if cota_digits and call:
        # Find DXCC prefix by callsign
        for entry in dxcc_data:
            match_found = False
            for p in entry.prefixes[1:]:
                if call.startswith(p.upper()):
                    land_prefix = entry.prefixes[0].upper()
                    match_found = True
                    break
            if not match_found and call.startswith(entry.prefixes[0].upper()):
                land_prefix = entry.prefixes[0].upper()
                match_found = True
            if match_found:
                break

        # Fallback: take the letters at the beginning of the call
        if not land_prefix:
            match = re.match(r"^([A-Z]+)", call)
            if match:
                land_prefix = match.group(1).upper()

    # Only fill in if country prefix and digits exist
    if land_prefix and cota_digits:
        ref_code = f"{land_prefix}-{cota_digits.zfill(5)}"  # COTA refs usually have 5 digits
        cota_var.set(ref_code)

        castle_info = cota_references.get(ref_code)
        if castle_info:
            cota_name_var.set(castle_info.get("name", ""))
        else:
            cota_name_var.set("")
    else:
        # Clear if invalid input
        cota_var.set("")
        cota_name_var.set("")







        
        

#########################################################################################
#  ___ ___ _____ _     ___ ___ ___ ___ ___ ___ _  _  ___ ___ 
# |_ _/ _ \_   _/_\   | _ \ __| _ \ __| _ \ __| \| |/ __| __|
#  | | (_) || |/ _ \  |   / _||   / _||   / _|| .` | (__| _| 
# |___\___/ |_/_/ \_\ |_|_\___|_|_\___|_|_\___|_|\_|\___|___|
#
#########################################################################################

def download_iota_file():
    """Download the IOTA reference file from iota-world.org"""
    try:
        response = requests.get(iota_url, timeout=20)
        response.raise_for_status()

        DATA_FOLDER.mkdir(parents=True, exist_ok=True)

        with open(IOTA_FILE, "wb") as f:
            f.write(response.content)

        root.after(0,lambda: messagebox.showinfo("Download","IOTA Reference file downloaded successfully."))
        return True

    except Exception as e:
        root.after(0, lambda: messagebox.showerror("Download Error", f"Error downloading IOTA Reference file:\n{e}"))
        return False



def iota_check():
    """Load IOTA reference data from fulllist.json into a dictionary."""
    global iota_references
    iota_references = {}                
    
    try:
        if not os.path.exists(IOTA_FILE):
#           messagebox.showinfo("IOTA Reference File Not Found", f"The file {IOTA_FILE} was not found. It will now be downloaded.")
            download_iota_file()


        with open(IOTA_FILE, "r", encoding="utf-8") as f:
            data = json.load(f)
        iota_references = {entry["refno"]: entry["name"] for entry in data}
    except Exception as e:
        messagebox.showerror("IOTA Error", f"Error loading IOTA references:\n{e}")
        iota_references = {}

def get_continent_from_callsign(callsign):
    cs = callsign.upper()

    for entry in dxcc_data:
        prefixes = getattr(entry, "prefixes", [])
        if any(cs.startswith(pref) for pref in prefixes):
            return getattr(entry, "continent", None)

    return None


def lookup_iota_ref(callsign, iota_number):
    """
    Build IOTA ref from callsign + number and return (ref, name).
    Example: ("PA1ABC", "032") -> ("EU-032", "Island name")
    """
    continent = get_continent_from_callsign(callsign)

    if not continent:
        return None, None

    ref = f"{continent}-{iota_number.zfill(3)}"
    name = iota_references.get(ref)

    return ref, name



def auto_fill_iota_prefix(iota_var, iota_name_var, iota_lat_var, iota_long_var, qso_callsign_var):
    """
    Auto-fill IOTA reference, name, and coordinates based on callsign and number.
    The coordinates are calculated as the midpoint of the island bounding box from JSON data.
    If no valid ref is found, all fields remain empty.
    """
    callsign = qso_callsign_var.get().strip().upper()
    entry_text = iota_var.get().strip().upper()

    # Extract only the number: everything after a dash or whole string if no dash
    if "-" in entry_text:
        number = entry_text.split("-")[-1]
    else:
        number = entry_text

    # Remove leading zeros
    number = number.lstrip("0")

    # Reset outputs
    iota_var.set("")
    iota_name_var.set("")
    iota_lat_var.set("")
    iota_long_var.set("")

    if callsign and number:
        ref, name = lookup_iota_ref(callsign, number)
        if ref and name:
            # Alleen vullen als er een geldige ref en name is
            iota_var.set(ref)
            iota_name_var.set(name)

            # Lookup coordinates in JSON file
            try:
                with open(IOTA_FILE, "r", encoding="utf-8") as f:
                    data = json.load(f)
                # Find entry matching ref
                entry = next((e for e in data if e["refno"].upper() == ref.upper()), None)
                if entry:
                    lat_max = float(entry.get("latitude_max", 0))
                    lat_min = float(entry.get("latitude_min", 0))
                    lon_max = float(entry.get("longitude_max", 0))
                    lon_min = float(entry.get("longitude_min", 0))

                    # Calculate midpoint
                    mid_lat = (lat_max + lat_min) / 2
                    mid_lon = (lon_max + lon_min) / 2

                    iota_lat_var.set(str(mid_lat))
                    iota_long_var.set(str(mid_lon))
            except Exception as e:
                print(f"IOTA coordinate lookup failed: {e}")






#########################################################################################
#  ___  ___ _____ _     ___ ___ ___ ___ ___ ___ _  _  ___ ___ 
# | _ \/ _ \_   _/_\   | _ \ __| __| __| _ \ __| \| |/ __| __|
# |  _/ (_) || |/ _ \  |   / _|| _|| _||   / _|| .` | (__| _| 
# |_|  \___/ |_/_/ \_\ |_|_\___|_| |___|_|_\___|_|\_|\___|___|
#
#########################################################################################

def download_pota_file():
    try:
        response = requests.get(pota_url)
        response.raise_for_status()

        with open(POTA_FILE, "wb") as f:
            f.write(response.content)

        root.after(0, lambda: messagebox.showinfo("Download",f"The file {POTA_FILE.name} has been downloaded successfully."))

    except Exception as e:
        root.after(0,lambda: messagebox.showerror("Download Error", f"Failed to download {POTA_FILE.name}:\n{e}"))


def pota_check(pota_file=POTA_FILE):
    global pota_references
    pota_references = {}

    try:
        if not os.path.exists(pota_file):
#           messagebox.showinfo("POTA Reference File Not Found", f"The file {pota_file} was not found. It will now be downloaded.")
            download_pota_file()

        with open(pota_file, newline='', encoding='utf-8') as csvfile:
            reader = csv.DictReader(csvfile)
            for row in reader:
                pota_references[row["reference"]] = row["name"]

    except Exception as e:
        messagebox.showerror("Error", f"Error loading POTA reference list: {e}")

def load_dxcc_to_pota_map(json_file=POTA_MAP_FILE):
    """
    Load DXCC to POTA prefix map (prefix ranges → ISO country codes).
    """
    global dxcc_to_pota_map
    try:
        if not os.path.exists(json_file):
            messagebox.showerror("Error", f"Mapping file {json_file} not found!")
            return
        with open(json_file, "r", encoding="utf-8") as f:
            dxcc_to_pota_map = json.load(f)
    except Exception as e:
        messagebox.showerror("Error", f"Failed to load mapping file: {e}")

def prefix_in_range(call, prefix):
    """
    Check if a callsign matches a prefix or prefix range.
    Example:
        prefix = "PA-PI", call = "PD5DJ" → True
        prefix = "ON", call = "ON4UN" → True
    """
    call = call.upper()
    if "-" in prefix:
        start, end = prefix.split("-")
        plen = len(start)
        fragment = call[:plen]
        return start <= fragment <= end
    else:
        return call.startswith(prefix.upper())


def find_landcode_for_call(call, dxcc_map):
    """
    Find ISO country code for a callsign using dxcc_map (prefix ranges).
    """
    call = call.upper()
    for landcode, prefixes in dxcc_map.items():
        for prefix in prefixes:
            if prefix_in_range(call, prefix):
                return landcode
    return None


def get_pota_park_name(ref_code):
    return pota_references.get(ref_code, "")


def auto_fill_pota_prefix(entry_var, park_name_var, qso_callsign_var, event=None):
    """
    Auto-fill POTA reference and park name based on callsign and numeric part.
    Uses prefix-range DXCC mapping.
    """
    raw_val = entry_var.get().strip()
    call = qso_callsign_var.get().strip().upper()

    # Extract only digits from the entry
    pota_digits = re.sub(r"\D", "", raw_val)
    pota_digits = pota_digits.lstrip("0")  # Remove leading zeros

    pota_landcode = None

    if pota_digits and call:
        # Find country code using the new prefix-range lookup
        pota_landcode = find_landcode_for_call(call, dxcc_to_pota_map)

        # Fallback: take letters from the callsign
        if not pota_landcode:
            match = re.match(r"^([A-Z]+)", call)
            if match:
                pota_landcode = match.group(1).upper()

    if pota_landcode and pota_digits:
        # Pad digits to 4
        padded_digits = pota_digits.zfill(4)
        ref_code = f"{pota_landcode}-{padded_digits}"
        entry_var.set(ref_code)
        update_pota_park_name(ref_code, park_name_var)
    else:
        # No valid code → clear field
        entry_var.set("")
        park_name_var.set("")




def update_pota_park_name(ref_code, park_name_var):
    # Finds the POTA name and puts it in the given StringVar
    park_name = pota_references.get(ref_code, "")
    park_name_var.set(park_name)





#########################################################################################
# __      ____      _____ ___   ___ ___ ___ ___ ___ _  _  ___ ___ 
# \ \    / /\ \    / / __| __| | _ \ __| __| _ \ __| \| |/ __| __|
#  \ \/\/ /  \ \/\/ /| _|| _|  |   / _|| _||   / _|| .` | (__| _| 
#   \_/\_/    \_/\_/ |_| |_|   |_|_\___|_| |_|_\___|_|\_|\___|___|
#                                                                 
#########################################################################################

def wwffref_check():
    global wwff_references
    try:
        if not os.path.exists(WWFF_FILE):
#           messagebox.showinfo("WWFF Reference File Not Found", "The file wwff_directory.csv was not found. It will now be downloaded.")
            download_wwffref_file()

        wwff_references = {}
        with open(WWFF_FILE, encoding="utf-8", newline='') as f:
            reader = csv.reader(f)
            for row in reader:
                # We expect at least 12 columns: ref, … lat, lon, …
                if len(row) >= 12:
                    ref = row[0].strip().upper()
                    name = row[2].strip()
                    lat = row[10].strip()
                    lon = row[11].strip()

                    wwff_references[ref] = {
                        "name": name,
                        "lat": lat,
                        "lon": lon
                    }

    except Exception as e:
        messagebox.showerror("Error", f"Error loading WWFF reference list: {e}")



def download_wwffref_file():
    try:
        response = requests.get(wwffref_url)
        response.raise_for_status()

        with open(WWFF_FILE, "wb") as f:
            f.write(response.content)

        root.after(
            0,
            lambda: messagebox.showinfo(
                "Download",
                "The file wwff_directory.csv has been downloaded successfully."
            )
        )

    except Exception as e:
        root.after(
            0,
            lambda: messagebox.showerror(
                "Download Error",
                f"Failed to download wwff_directory.csv:\n{e}"
            )
        )



def auto_fill_wwff_prefix(wwff_var, wwff_park_name_var, wwff_lat_var, wwff_long_var, qso_callsign_var):
    """
    Auto-fill WWFF reference, park name, and coordinates based on callsign and numeric part.
    Only fill fields if a valid land code and digits exist.
    """
    raw_val = wwff_var.get().strip()
    call = qso_callsign_var.get().strip().upper()

    # Extract only digits
    wwff_digits = re.sub(r"\D", "", raw_val)
    wwff_digits = wwff_digits.lstrip("0")  # don't default to "0"

    land_prefix = None
    if wwff_digits and call:
        # Try to find matching DXCC entry for the callsign
        for entry in dxcc_data:
            match_found = False
            for p in entry.prefixes[1:]:
                if call.startswith(p.upper()):
                    land_prefix = entry.prefixes[0].upper()
                    match_found = True
                    break
            if not match_found and call.startswith(entry.prefixes[0].upper()):
                land_prefix = entry.prefixes[0].upper()
                match_found = True
            if match_found:
                break

        # Fallback: take first letter(s) of callsign if no DXCC match
        if not land_prefix:
            match = re.match(r"^([A-Z]+)", call)
            if match:
                land_prefix = match.group(1).upper()

    # Only fill fields if land prefix AND digits exist
    if land_prefix and wwff_digits:
        ref_code = f"{land_prefix}FF-{wwff_digits.zfill(4)}"
        wwff_var.set(ref_code)

        park_info = wwff_references.get(ref_code)
        if park_info:
            wwff_park_name_var.set(park_info.get("name", ""))
            wwff_lat_var.set(park_info.get("lat", ""))
            wwff_long_var.set(park_info.get("lon", ""))
        else:
            wwff_park_name_var.set("")
            wwff_lat_var.set("")
            wwff_long_var.set("")
    else:
        # Clear all fields if invalid
        wwff_var.set("")
        wwff_park_name_var.set("")
        wwff_lat_var.set("")
        wwff_long_var.set("")



#########################################################################################
#  ___  ___ _____ _     ___ ___ ___ ___ ___ ___ _  _  ___ ___ 
# | _ )/ _ \_   _/_\   | _ \ __| __| __| _ \ __| \| |/ __| __|
# | _ \ (_) || |/ _ \  |   / _|| _|| _||   / _|| .` | (__| _| 
# |___/\___/ |_/_/ \_\ |_|_\___|_| |___|_|_\___|_|\_|\___|___|
#
#########################################################################################                                                             

def download_bota_file():
    try:
        response = requests.get(bota_url)
        response.raise_for_status()

        with open(BOTA_FILE, "wb") as f:
            f.write(response.content)

        root.after(0,lambda: messagebox.showinfo("Download",f"The file {BOTA_FILE.name} has been downloaded successfully."))

    except Exception as e:
        root.after(0,lambda: messagebox.showerror("Download Error",f"Failed to download {BOTA_FILE.name}:\n{e}"))


def bota_check():
    "Load bota_directory.csv into bota_references (ref_code → (name, lat, lon))."
    global bota_references
    bota_references.clear()
    try:
        if not os.path.exists(BOTA_FILE):
#           messagebox.showinfo("BOTA Reference File Not Found", f"The file {BOTA_FILE} was not found. It will now be downloaded.")
            download_bota_file()

        with open(BOTA_FILE, encoding="utf-8") as f:
            reader = csv.reader(f)
            next(reader, None)
            for parts in reader:
                if len(parts) >= 7:
                    ref_code = parts[2].strip().upper()
                    name = parts[3].strip()
                    lat = parts[5].strip()
                    lon = parts[6].strip()
                    bota_references[ref_code] = (name, lat, lon)

    except Exception as e:
        messagebox.showerror("Error", f"Error loading BOTA reference list: {e}")

def auto_fill_bota_prefix(bota_var, bota_name_var, bota_lat_var, bota_long_var, qso_callsign_var):
    """
    Auto-fill BOTA reference, park name, and coordinates based on callsign and numeric part.
    Only fill fields if a valid land prefix and digits exist.
    """
    raw_val = (bota_var.get() or "").strip().upper()
    call = (qso_callsign_var.get() or "").strip().upper()

    # Extract only digits; no default "0"
    bota_digits = re.sub(r"\D", "", raw_val).lstrip("0")

    special_prefixes = ["GI", "GW", "GU", "GM", "GJ", "GD"]
    land_prefix = None

    # Reset outputs
    bota_name_var.set("")
    bota_lat_var.set("")
    bota_long_var.set("")

    if bota_digits and call:
        # Check for special UK prefixes first
        for sp in special_prefixes:
            if call.startswith(sp):
                land_prefix = sp
                break

        # Check DXCC data if no special prefix matched
        if not land_prefix:
            for entry in dxcc_data:
                match_found = False
                for p in entry.prefixes[1:]:
                    if call.startswith(p.upper()):
                        land_prefix = entry.prefixes[0].upper()
                        match_found = True
                        break
                if not match_found and call.startswith(entry.prefixes[0].upper()):
                    land_prefix = entry.prefixes[0].upper()
                    match_found = True
                if match_found:
                    break

        # Fallback: take leading alphanumeric part of callsign
        if not land_prefix:
            m = re.match(r"^([A-Z0-9]+)", call)
            if m:
                land_prefix = m.group(1).upper()

    # Special case for Malaysian prefixes
    if land_prefix in ("9M2", "9M4"):
        land_prefix = "9M"

    # Only fill fields if both land prefix and digits exist
    if land_prefix and bota_digits:
        ref_code = f"B/{land_prefix}-{bota_digits.zfill(4)}".upper()
        bota_var.set(ref_code)

        info = bota_references.get(ref_code.upper())
        if info:
            name, lat, lon = info
            bota_name_var.set(name)
            bota_lat_var.set(lat)
            bota_long_var.set(lon)
        else:
            bota_name_var.set("")
            bota_lat_var.set("")
            bota_long_var.set("")
    else:
        # Clear BOTA field if invalid
        bota_var.set("")

def entity_check():
    global prefix_to_dxcc
    try:
        with open(ENTITY_FILE, "r", encoding="utf-8") as f:
            prefix_to_dxcc = json.load(f)
    except FileNotFoundError:
        messagebox.showerror("Error", "prefix_to_dxcc.json not found!")
        prefix_to_dxcc = {}



def get_dxcc_from_callsign(callsign: str):
    cs = callsign.upper()

    # Langste prefix eerst proberen
    for length in range(len(cs), 0, -1):
        prefix = cs[:length]
        if prefix in prefix_to_dxcc:
            return prefix_to_dxcc[prefix]

    return None





#########################################################################################
#
#  ___ ___ _____ ___ _  _   ___ ___ ___ ___ _____  __   ___ _____ _____ ___   _ _____ 
# | __| __|_   _/ __| || | | _ \ _ \ __| __|_ _\ \/ /  / __|_   _|_   _|   \ /_\_   _|
# | _|| _|  | || (__| __ | |  _/   / _|| _| | | >  <  | (__  | |   | |_| |) / _ \| |  
# |_| |___| |_| \___|_||_| |_| |_|_\___|_| |___/_/\_\  \___| |_|   |_(_)___/_/ \_\_|  
#                                                                                     
#########################################################################################

# Function to check if cty.day file exists in root folder
def ctydat_check():
    global dxcc_data, dxcc_prefix_map
    try:
        if not os.path.exists(DXCC_FILE):
#           messagebox.showinfo("File Not Found", "The file cty.dat was not found. It will now be downloaded.")
            download_ctydat_file()  # Automatically download the file after showing the message
    
    except Exception as e:
        messagebox.showerror("Error", f"Error loading data: {e}")
        return {}
    
    # load and parse cty.dat into dxcc_data with cty_parser.py
    dxcc_data = parse_cty_file(DXCC_FILE)
    
    dxcc_prefix_map = {}

    for entry in dxcc_data:
        for raw_prefix in entry.prefixes:
            prefix = re.sub(r'[=;()\[\]*]', '', raw_prefix).strip().upper()
            if prefix:
                dxcc_prefix_map[prefix] = entry




# Function to download the cty.dat file directly into the root folder
def download_ctydat_file():
    try:
        response = requests.get(ctydat_url)
        response.raise_for_status()

        with open(DXCC_FILE, 'wb') as file:
            file.write(response.content)

        root.after(0,lambda: messagebox.showinfo("Download", "The file cty.dat has been downloaded successfully."))

    except Exception as e:
        root.after(0,lambda: messagebox.showerror("Download Error", f"Failed to download file:\n{e}"))




# Continent mapping
continent_map = {
    "EU": "Europe",
    "OC": "Oceania",
    "AF": "Africa",
    "AN": "Antarctica",
    "AS": "Asia",
    "NA": "North America",
    "SA": "South America"
}

def find_coordinates_by_callsign(callsign):
    callsign = callsign.strip().upper()
    for entry in dxcc_data:
        for raw_prefix in entry.prefixes:
            prefix = re.sub(r'[=;()\[\]*]', '', raw_prefix).strip().upper()
            if callsign.startswith(prefix):
                return entry.latitude, entry.longitude
    return None, None




def haversine(lat1, lon1, lat2, lon2):
    R = 6371.0  # Earth radius in kilometers
    phi1 = math.radians(lat1)
    phi2 = math.radians(lat2)
    d_phi = math.radians(lat2 - lat1)
    d_lambda = math.radians(lon2 - lon1)

    a = math.sin(d_phi / 2)**2 + math.cos(phi1) * math.cos(phi2) * math.sin(d_lambda / 2)**2
    c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))

    return R * c


def check_callsign_prefix(callsign, update_ui=True, skip_locator=False):
    """
    Determine DXCC information based on callsign prefix.
    Uses dxcc_data (cty.dat result) for country/continent/zone/lat/lon
    and prefix_to_dxcc.json for the official DXCC Entity code.
    """

    global dxcc_data, entityCode, qso_cq_zone_var, qso_itu_zone_var

    callsign = callsign.strip().upper()
    best_match = None
    best_prefix_len = 0

    # Always reset zones (important for non-UI usage)
    qso_cq_zone_var.set("")
    qso_itu_zone_var.set("")

    # ---------------------------------------------------------------------
    # 1) Find best matching prefix in dxcc_data (cty.dat)
    # ---------------------------------------------------------------------
    for entry in dxcc_data:
        for raw_prefix in entry.prefixes:
            prefix = re.sub(r'[=;()\[\]*]', '', raw_prefix).strip().upper()
            if callsign.startswith(prefix) and len(prefix) > best_prefix_len:
                best_match = entry
                best_prefix_len = len(prefix)

    # ---------------------------------------------------------------------
    # 2) Lookup DXCC ENTITY code from prefix_to_dxcc.json
    # ---------------------------------------------------------------------
    dxcc_entity = None
    for i in range(len(callsign), 0, -1):
        pf = callsign[:i]
        if pf in prefix_to_dxcc:
            dxcc_entity = prefix_to_dxcc[pf]
            break

    # ---------------------------------------------------------------------
    # 3) Fill data if match found
    # ---------------------------------------------------------------------
    if best_match:

        qso_country_var.set(best_match.name)
        qso_continent_var.set(best_match.continent)

        entityCode = dxcc_entity if dxcc_entity is not None else ""

        # ---- FLAG HANDLING ----------------------------------------------
        if entityCode:
            orig = FLAG_IMAGES.get(int(entityCode))
        else:
            orig = None

        if orig is None:
            orig = FLAG_IMAGES.get(-1)

        flag_img = resize_flag(orig, DXCC_FLAG_SIZE)
        if flag_img is None:
            flag_img = orig

        if update_ui:
            flag_label.config(image=flag_img)
            flag_label.image = flag_img

        # -----------------------------------------------------------------
        # CQ / ITU zones from cty.dat
        # -----------------------------------------------------------------
        qso_cq_zone_var.set(str(best_match.cq_zone))
        qso_itu_zone_var.set(str(best_match.itu_zone))

        lat = best_match.latitude
        lon = best_match.longitude

        heading_var = tk.StringVar()
        distance_var_km = tk.StringVar()
        distance_var_miles = tk.StringVar()

        station_call = station_callsign_var.get().strip().upper()
        station_lat, station_lon = find_coordinates_by_callsign(station_call)

        if station_lat is not None and station_lon is not None:
            sp, lp = calculate_headings(station_lat, station_lon, lat, lon)
            heading_var.set(f"SP: {sp}°  /  LP: {lp}°")
            dist_km = haversine(station_lat, station_lon, lat, lon)
            dist_mi = dist_km * 0.621371
            distance_var_km.set(f"{dist_km:.1f}km")
            distance_var_miles.set(f"{dist_mi:.1f}mi")
        else:
            heading_var.set("SP: --°  /  LP: --°")
            distance_var_km.set("-- km")
            distance_var_miles.set("-- mi")

        if update_ui:
            country_continent_label.config(
                text=f"{best_match.name}, ({continent_map.get(best_match.continent, 'Unknown')})"
            )
            heading_label.config(text=heading_var.get())
            distance_label.config(
                text=f"{distance_var_km.get()} / {distance_var_miles.get()}"
            )
            dxcc_cq_itu_label.config(
                text=f"CQ: {qso_cq_zone_var.get()}, ITU: {qso_itu_zone_var.get()}"
            )

            if not skip_locator and qso_locator_var.get().strip():
                calculate_from_locator()

    # ---------------------------------------------------------------------
    # 4) No match → clear fields
    # ---------------------------------------------------------------------
    else:
        qso_country_var.set("")
        qso_continent_var.set("")
        entityCode = ""

        # zones already cleared above

        if update_ui:
            country_continent_label.config(text="----")
            heading_label.config(text="----")
            distance_label.config(text="----")
            dxcc_cq_itu_label.config(text="----")

            orig = FLAG_IMAGES.get(-1)
            flag_img = resize_flag(orig, DXCC_FLAG_SIZE) or orig

            flag_label.config(image=flag_img)
            flag_label.image = flag_img





def calculate_from_locator():
    """
    Calculates heading and distance based on qso_locator_var and station_locator_var.
    If locator is too short (<4), falls back to prefix-based calculation.
    """
    loc = qso_locator_var.get().strip().upper()
    station_loc = station_locator_var.get().strip().upper()

    # Fallback when locator is too short
    if len(loc) < 4:
        check_callsign_prefix(qso_callsign_var.get().strip().upper(), skip_locator=True)
        return

    if not is_valid_locator(loc) or not is_valid_locator(station_loc):
        return

    def locator_to_latlon(locator):
        try:
            locator = locator.strip().upper()
            if len(locator) < 4:
                return None, None

            A = ord('A')
            lon = (ord(locator[0]) - A) * 20 - 180
            lat = (ord(locator[1]) - A) * 10 - 90
            lon += int(locator[2]) * 2
            lat += int(locator[3]) * 1
            if len(locator) >= 6:
                lon += (ord(locator[4]) - A) * 5 / 60
                lat += (ord(locator[5]) - A) * 2.5 / 60
            else:
                lon += 1
                lat += 0.5
            return lat, lon
        except Exception:
            return None, None

    lat1, lon1 = locator_to_latlon(station_loc)
    lat2, lon2 = locator_to_latlon(loc)

    if None in (lat1, lon1, lat2, lon2):
        return

    sp, lp = calculate_headings(lat1, lon1, lat2, lon2)
    heading_label.config(text=f"SP: {sp}°  /  LP: {lp}°")

    distance_km = haversine(lat1, lon1, lat2, lon2)
    distance_miles = distance_km * 0.621371
    distance_label.config(text=f"{distance_km:.1f}km / {distance_miles:.1f}mi")






#########################################################################################
# __      _____  ___ _  _____ ___    ___ ___ ___ ___  ___ ___ 
# \ \    / / _ \| _ \ |/ / __|   \  | _ ) __| __/ _ \| _ \ __|
#  \ \/\/ / (_) |   / ' <| _|| |) | | _ \ _|| _| (_) |   / _| 
#   \_/\_/ \___/|_|_\_|\_\___|___/  |___/___|_| \___/|_|_\___|
#                                                             
#########################################################################################

def normalize_callsign(call):
    return call.upper().split('/')[0]  # Remove any suffix like /P, /M, /QRP, etc.

def update_worked_before_tree(*args):
    global qso_lines

    entered_call = qso_callsign_var.get().strip().upper()
    if not entered_call:
        workedb4_tree.delete(*workedb4_tree.get_children())
        return

    matches = [qso for qso in qso_lines if qso.get("Callsign", "").upper().startswith(entered_call)]
    
    from datetime import datetime
    def sort_key(qso):
        date = qso.get("Date", "1900-01-01")
        time = qso.get("Time", "00:00:00")
        try:
            return datetime.strptime(f"{date} {time}", "%Y-%m-%d %H:%M:%S")
        except ValueError:
            return datetime.min

    matches.sort(key=sort_key, reverse=True)
    workedb4_tree.delete(*workedb4_tree.get_children())

    # Configure Treeview tags for coloring
    workedb4_tree.tag_configure("oddrow", background="#f0f0f0")
    workedb4_tree.tag_configure("evenrow", background="white")
    workedb4_tree.tag_configure("dupe_date_band_mode_match", foreground="white", background="red", font=("TkDefaultFont", 9, "bold"))  # RED
    workedb4_tree.tag_configure("exact_callsign_today", foreground="black", background="darkorange", font=("TkDefaultFont", 9, "bold"))          # ORANGE
    workedb4_tree.tag_configure("base_callsign_today", foreground="white", background="blue", font=("TkDefaultFont", 9, "bold"))                   # BLUE

    # Current inputs for matching
    today = qso_date_var.get().strip()
    current_band = qso_band_var.get().strip() if 'qso_band_var' in globals() else ""
    current_mode = qso_mode_var.get().strip().upper() if 'qso_mode_var' in globals() else ""

    row_color = True
    for qso in matches:
        tag = "oddrow" if row_color else "evenrow"
        row_color = not row_color

        qso_call = qso.get("Callsign", "").upper()
        base_qso_call = normalize_callsign(qso_call)
        base_entered_call = normalize_callsign(entered_call)

        date_match = qso.get("Date", "").strip() == today
        band_match = qso.get("Band", "").strip().lower() == current_band.lower()
        mode_match = qso.get("Mode", "").strip().upper() == current_mode

        # Assign color tags:
        # 1) Exact callsign + date + band + mode => RED
        if qso_call == entered_call and date_match and band_match and mode_match:
            tag = "dupe_date_band_mode_match"  # RED

        # 2) Exact callsign + date only => ORANGE
        elif qso_call == entered_call and date_match:
            tag = "exact_callsign_today"       # ORANGE

        # 3) Base callsign match + date only + different suffix => BLUE
        elif base_qso_call == base_entered_call and qso_call != entered_call and date_match:
            tag = "base_callsign_today"        # BLUE

        workedb4_tree.insert("", "end", values=(
            qso_call,
            qso.get("Date", ""),
            qso.get("Time", ""),
            qso.get("Band", ""),
            qso.get("Mode", ""),
            qso.get("Frequency", ""),
            qso.get("Country", "")
        ), tags=(tag,))



def show_color_legend():
    legend_win = tk.Toplevel(root)
    legend_win.title("Worked Before Color Legend")
    legend_win.resizable(False, False)
    legend_win.configure(padx=10, pady=10)

    width, height = 380, 200

    root_x = root.winfo_rootx()
    root_y = root.winfo_rooty()
    root_w = root.winfo_width()
    root_h = root.winfo_height()

    x = root_x + (root_w // 2) - (width // 2)
    y = root_y + (root_h // 2) - (height // 2)

    legend_win.geometry(f"{width}x{height}+{x}+{y}")

    legend_items = [
        ("Exact callsign match (including suffix)\nFull match on Date, Band, Mode",
         "red", "black"),
        ("Base callsign match (ignores suffix)\nFull match on Date, Band, Mode",
         "blue", "black"),
        ("Exact callsign match (including suffix)\nMatch on Date only",
         "darkorange", "black"),
        ("Callsign match\nJust worked before",
         "white", "black"),
    ]

    for i, (desc, bg, fg) in enumerate(legend_items):
        color_box = tk.Canvas(legend_win, width=25, height=25, bg=bg, highlightthickness=1, highlightbackground="black")
        color_box.grid(row=i, column=0, padx=(0,10), pady=5)

        label = tk.Label(legend_win, text=desc, justify="left", fg=fg, bg=legend_win.cget("bg"), font=("Arial", 10))
        label.grid(row=i, column=1, sticky="w")

    legend_win.transient(root)
    legend_win.grab_set()
    legend_win.focus_set()




#########################################################################################
#  __  __   _   ___ _  _ 
# |  \/  | /_\ |_ _| \| |
# | |\/| |/ _ \ | || .` |
# |_|  |_/_/ \_\___|_|\_|
#
#########################################################################################                        

def ensure_single_instance():
    """
    Windows file lock based single instance.
    Works with python.exe, pythonw.exe and EXE.
    """
    global _lock_handle

    _lock_handle = open(LOCK_FILE, "w")

    try:
        msvcrt.locking(_lock_handle.fileno(), msvcrt.LK_NBLCK, 1)
    except OSError:
        ctypes.windll.user32.MessageBoxW(None, "MiniBook is already running.", "MiniBook", 0x10)
        sys.exit(0)

ensure_single_instance()

# Main window
root                        = tk.Tk()

style                       = ttk.Style()
style.theme_use("clam")

# ComboBox and Notebook Styling in main window
root.option_add("*TCombobox*Font", ("Arial", 10))
style.configure('TNotebook.Tab', font=('Arial', 10, 'bold'))

# Load saved geometry for the main window
load_window_position(root, "MainWindow")

# Global Configuration & Variables
config                      = configparser.ConfigParser()

upload_qrz_var              = tk.BooleanVar(value=False)
use_serial_var              = tk.BooleanVar(value=False)
utc_offset_var              = tk.StringVar(value="0")
radio_status_var            = tk.StringVar()

datetime_tracking_enabled   = tk.BooleanVar(value=True)
freqmode_tracking_var       = tk.BooleanVar(value=False)

tracking_enabled_var        = tk.BooleanVar(value=False)

qso_date_var                = tk.StringVar()
qso_time_var                = tk.StringVar()
qso_callsign_var            = tk.StringVar()
qso_callsign_var.trace_add('write', lambda *args: check_callsign_prefix(qso_callsign_var.get(),True)) # Trigger check on input change with the callsign value passed to the function
qso_callsign_var.trace_add("write", update_worked_before_tree)
qso_callsign_var.trace_add("write", lambda *args: qso_callsign_var.set(qso_callsign_var.get().upper()))
qso_name_var                = tk.StringVar()
qso_locator_var             = tk.StringVar()
qso_locator_var.trace_add("write", lambda *args: qso_locator_var.set(qso_locator_var.get().upper()))
qso_locator_var.trace_add("write", lambda *args: calculate_from_locator())
qso_country_var             = tk.StringVar()
qso_continent_var           = tk.StringVar()
qso_rst_sent_var = tk.StringVar(value="59")
qso_rst_received_var        = tk.StringVar(value="59")
qso_sent_exchange_var       = tk.StringVar(value="1")
qso_receive_exchange_var = tk.StringVar()
qso_receive_exchange_var.trace_add("write", lambda *args: qso_receive_exchange_var.set(qso_receive_exchange_var.get().upper()))
qso_comment_var             = tk.StringVar()
qso_frequency_var           = tk.StringVar()
qso_cq_zone_var                 = tk.StringVar(value="")
qso_itu_zone_var                = tk.StringVar(value="")


qso_band_var                = tk.StringVar(value="20m")
qso_band_var.trace_add("write", update_worked_before_tree)
qso_mode_var                = tk.StringVar(value="USB")
qso_mode_var.trace_add("write", update_worked_before_tree)
qso_submode_var             = tk.StringVar(value="")
qso_prop_mode_var           = tk.StringVar(value="")
qso_satellite_var           = tk.StringVar(value="")

qrz_username_var            = tk.StringVar()
qrz_password_var            = tk.StringVar()
qrz_city_var                = tk.StringVar()
qrz_zipcode_var             = tk.StringVar()
qrz_address_var             = tk.StringVar()
qrz_qsl_info_var            = tk.StringVar()

station_locator_var         = tk.StringVar()
station_location_var        = tk.StringVar()
station_callsign_var        = tk.StringVar()
station_operator_var        = tk.StringVar()

station_name_var            = tk.StringVar()
station_city_var            = tk.StringVar()
station_street_var          = tk.StringVar()
station_postalcode_var      = tk.StringVar()
station_county_var          = tk.StringVar()
station_country_var         = tk.StringVar()
station_cqzone_var          = tk.StringVar()
station_ituzone_var         = tk.StringVar()

station_contest_var         = tk.StringVar()

station_bota_var            = tk.StringVar()
station_cota_var            = tk.StringVar()
station_iota_var            = tk.StringVar()
station_pota_var            = tk.StringVar()
station_sota_var            = tk.StringVar()
station_wlota_var           = tk.StringVar()
station_wwff_var            = tk.StringVar()
station_qrzapi_var          = tk.StringVar()


# --- NEW: MiniBook Wordpress Dashboard settings (Preferences + runtime) ---
minibook_web_url_var        = tk.StringVar()
minibook_web_id_var         = tk.StringVar()
minibook_web_api_var        = tk.StringVar()
minibook_web_interval_var   = tk.StringVar()

minibook_web_enabled_var    = tk.BooleanVar(value=False)

# Job ID for Tkinter after() scheduler
minibook_web_job            = None


# Bunkers On The Air
bota_var                    = tk.StringVar()
bota_var.trace_add("write", lambda *args: bota_var.set(bota_var.get().upper()))
bota_name_var               = tk.StringVar()
bota_lat_var                = tk.StringVar()
bota_long_var               = tk.StringVar()

# Castles On The Air
cota_var                    = tk.StringVar()
cota_var.trace_add("write", lambda *args: cota_var.set(cota_var.get().upper()))
cota_name_var               = tk.StringVar()
cota_lat_var                = tk.StringVar()
cota_long_var               = tk.StringVar()

# Islands On The Air
iota_var                    = tk.StringVar()
iota_var.trace_add("write", lambda *args: iota_var.set(iota_var.get().upper()))
iota_name_var               = tk.StringVar()
iota_lat_var                = tk.StringVar()
iota_long_var               = tk.StringVar()

# Parks On The Air
pota_var                    = tk.StringVar()
pota_var.trace_add("write", lambda *args: pota_var.set(pota_var.get().upper()))
pota_park_name_var          = tk.StringVar()

# Summits On The Air
sota_var                    = tk.StringVar()
sota_var.trace_add("write", lambda *args: sota_var.set(sota_var.get().upper()))
sota_name_var               = tk.StringVar()
sota_lat_var                = tk.StringVar()
sota_long_var               = tk.StringVar()

# World Lighthouse On The Air
wlota_var                   = tk.StringVar()
wlota_var.trace_add("write", lambda *args: wlota_var.set(wlota_var.get().upper()))

# World Wide Flora Fauna
wwff_var                    = tk.StringVar()
wwff_var.trace_add("write", lambda *args: wwff_var.set(wwff_var.get().upper()))
wwff_lat_var                = tk.StringVar()
wwff_long_var               = tk.StringVar()
wwff_park_name_var          = tk.StringVar()

# Preparation of hamlib in Threaded mode
hamlib_process              = None
socket_connection           = None

stop_frequency_thread       = threading.Event()

# Updates Main window title
update_title(root, VERSION_NUMBER, CURRENT_JSON_FILE, radio_status_var.get())




# Adding Menu to main window
menu_bar = tk.Menu(root)

# --- File Menu ---
file_menu = tk.Menu(menu_bar, tearoff=0)
file_menu.add_command(label="New logbook", command=create_new_json)
file_menu.add_command(label="Load logbook", command=load_json)
file_menu.add_separator()
file_menu.add_command(label="Open Backup Folder", command=open_backup_folder)
file_menu.add_separator()
file_menu.add_command(label="Station setup", command=open_station_setup, state='disabled')
file_menu.add_separator()
file_menu.add_command(label="Preferences", command=open_preferences)
file_menu.add_separator()
file_menu.add_command(label="Exit", command=exit_program)
menu_bar.add_cascade(label="File", menu=file_menu)


# --- Reference Menu ---
reference_menu = tk.Menu(menu_bar, tearoff=0)
reference_menu.add_command(label="POTA Activations", command=lambda: webbrowser.open("https://pota.app/#/"))
reference_menu.add_command(label="POTA Map", command=lambda: webbrowser.open("https://pota.app/#/map"))
reference_menu.add_command(label="POTA List", command=lambda: webbrowser.open("https://pota.app/#/parklist"))
reference_menu.add_separator()
reference_menu.add_command(label="WWFF Agenda", command=lambda: webbrowser.open("https://wwff.co/agenda/"))
reference_menu.add_command(label="WWFF Dutch Map", command=lambda: webbrowser.open("https://www.google.com/maps/d/u/0/view?hl=en&mid=1yXBN79NWlsI-wrZaDfyeXA2zg129nEU&ll=52.12186480765635%2C5.251994000000018&z=8"))
reference_menu.add_command(label="WWFF Global Map", command=lambda: webbrowser.open("https://ham-map.com/"))
reference_menu.add_command(label="WWFF Announce activation", command=lambda: webbrowser.open("https://www.cqgma.org/alertwwfflight.php"))
reference_menu.add_separator()
reference_menu.add_command(label="WW BOTA Map", command=lambda: webbrowser.open("https://wwbota.org/map/"))
reference_menu.add_separator()
reference_menu.add_command(label="IOTA", command=lambda: webbrowser.open("https://www.iota-world.org/"))
reference_menu.add_separator()
reference_menu.add_command(label="SOTA Watch", command=lambda: webbrowser.open("https://sotawatch.sota.org.uk/"))
reference_menu.add_command(label="SOTA Maps", command=lambda: webbrowser.open("https://www.sotamaps.org/"))
reference_menu.add_separator()
reference_menu.add_command(label="World lighthouse OTA", command=lambda: webbrowser.open("https://www.wlota.com/"))
menu_bar.add_cascade(label="References", menu=reference_menu)

# --- Help Menu ---
help_menu = tk.Menu(menu_bar, tearoff=0)
help_menu.add_command(label="Open Log Folder", command=open_log_folder)
help_menu.add_separator()
help_menu.add_command(label="Worked Before Color Legend", command=show_color_legend)
help_menu.add_separator()
help_menu.add_command(label="Download latest CTY.DAT file", command=download_ctydat_file)
help_menu.add_command(label="Download latest Pota reference file", command=download_pota_file)
help_menu.add_command(label="Download latest WWFF reference file", command=download_wwffref_file)
help_menu.add_command(label="Download latest BOTA reference file", command=download_bota_file)
help_menu.add_command(label="Download latest COTA reference file", command=download_cota_file)
help_menu.add_command(label="Download latest IOTA reference file", command=download_iota_file)
help_menu.add_command(label="Download latest SOTA reference file", command=download_sota_file)
help_menu.add_separator()
help_menu.add_command(label="About", command=show_about)
menu_bar.add_cascade(label="Help", menu=help_menu)
root.config(menu=menu_bar)






# =====================================================
# UNIVERSAL COLLAPSABLE SECTION ENGINE
# =====================================================

def expand_main_window():
    """Force the main application window to auto-expand."""
    root.update_idletasks()
    root.geometry("")


def collapse_main_window():
    """Force the main application window to auto-shrink."""
    root.update_idletasks()
    root.geometry("")


def register_collapsable_section(button, section_widget, default_open=True):
    """
    Generic collapsable handler.
    button: Collapse/expand button
    section_widget: The frame or notebook to show/hide
    default_open: If True, section is visible at startup
    """

    # Local state container
    state = {"collapsed": not default_open}

    # Set initial button text
    if default_open:
        button.config(text="▼ " + button.collapse_title)
    else:
        button.config(text="▶ " + button.collapse_title)
        section_widget.grid_remove()

    def toggle():
        """Toggle the section between expanded and collapsed states."""
        if state["collapsed"]:
            section_widget.grid()
            button.config(text="▼ " + button.collapse_title)
            state["collapsed"] = False
            expand_main_window()
        else:
            section_widget.grid_remove()
            button.config(text="▶ " + button.collapse_title)
            state["collapsed"] = True
            collapse_main_window()

    # Assign command handler
    button.config(command=toggle)

    # Hover behavior
    def hover_in(event):
        button.config(bg="#cfcfcf")

    def hover_out(event):
        button.config(bg="#d9d9d9")

    button.bind("<Enter>", hover_in)
    button.bind("<Leave>", hover_out)


def create_collapse_button(parent, title):
    """Create a styled collapse/expand button."""
    btn = tk.Button(parent, text="", font=('Arial', 10, 'bold'), bg="#d9d9d9", fg="black", relief="raised", bd=2, activebackground="#c0c0c0", activeforeground="black")
    btn.collapse_title = title  # Used by the generic toggle engine
    return btn



# --------------------------------------------------------------
# Collapsable: My Station / Activation Info
# --------------------------------------------------------------

# Create the collapse/expand button (same style as Reference section)
info_toggle_button = create_collapse_button(root, "My Station / Activation Info")
info_toggle_button.grid(row=0, column=0, columnspan=7, sticky="ew", padx=5, pady=(5, 0))

# Create the notebook for Station and Activation information
info_notebook = ttk.Notebook(root)
info_notebook.grid(row=0 + 1, column=0, columnspan=7, sticky='nsew', padx=5, pady=2)

# Register this notebook as a collapsable section
register_collapsable_section(button=info_toggle_button, section_widget=info_notebook, default_open=True)

# ==============================
# STATION INFO TAB
# ==============================
station_info_frame = tk.Frame(info_notebook, bd=2, relief='groove', pady=2, bg='lightgrey')
info_notebook.add(station_info_frame, text="My Station Information")

# Configure columns for stretching
for i in range(6):
    station_info_frame.grid_columnconfigure(i, weight=1)

# Row 0
tk.Label(station_info_frame, text="Callsign:", font=('Arial', 10), bg='lightgrey').grid(row=0, column=0, sticky='e', padx=5)

station_callsign_entry = tk.Entry(station_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_callsign_entry.grid(row=0, column=1, sticky='w', padx=5)

tk.Label(station_info_frame, text="Operator:", font=('Arial', 10), bg='lightgrey').grid(row=0, column=2, sticky='e', padx=5)

station_operator_entry = tk.Entry(station_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_operator_entry.grid(row=0, column=3, sticky='w', padx=5)

tk.Label(station_info_frame, text="Locator:", font=('Arial', 10), bg='lightgrey').grid(row=0, column=4, sticky='e', padx=5)

station_locator_entry = tk.Entry(station_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_locator_entry.grid(row=0, column=5, sticky='w', padx=5)

# Row 1
tk.Label(station_info_frame, text="Location:", font=('Arial', 10), bg='lightgrey').grid(row=1, column=0, sticky='e', padx=5)

station_location_entry = tk.Entry(station_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_location_entry.grid(row=1, column=1, sticky='w', padx=5)

# ==============================
# ACTIVATION INFO TAB
# ==============================
activation_info_frame = tk.Frame(info_notebook, bd=2, relief='groove', pady=2, bg='lightgrey')
info_notebook.add(activation_info_frame, text="My Activation Information")

# Configure columns for stretching
for i in range(6):
    activation_info_frame.grid_columnconfigure(i, weight=1)

# Row 0
tk.Label(activation_info_frame, text="BOTA:", font=('Arial', 10), bg='lightgrey').grid(row=0, column=0, sticky='e', padx=5)

station_bota_entry = tk.Entry(activation_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_bota_entry.grid(row=0, column=1, sticky='w', padx=5)

tk.Label(activation_info_frame, text="COTA:", font=('Arial', 10), bg='lightgrey').grid(row=0, column=2, sticky='e', padx=5)

station_cota_entry = tk.Entry(activation_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_cota_entry.grid(row=0, column=3, sticky='w', padx=5)

tk.Label(activation_info_frame, text="IOTA:", font=('Arial', 10), bg='lightgrey').grid(row=0, column=4, sticky='e', padx=5)

station_iota_entry = tk.Entry(activation_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_iota_entry.grid(row=0, column=5, sticky='w', padx=5)

# Row 1
tk.Label(activation_info_frame, text="POTA:", font=('Arial', 10), bg='lightgrey').grid(row=1, column=0, sticky='e', padx=5)

station_pota_entry = tk.Entry(activation_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_pota_entry.grid(row=1, column=1, sticky='w', padx=5)

tk.Label(activation_info_frame, text="SOTA:", font=('Arial', 10), bg='lightgrey').grid(row=1, column=2, sticky='e', padx=5)

station_sota_entry = tk.Entry(activation_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_sota_entry.grid(row=1, column=3, sticky='w', padx=5)

tk.Label(activation_info_frame, text="WLOTA:", font=('Arial', 10), bg='lightgrey').grid(row=1, column=4, sticky='e', padx=5)

station_wlota_entry = tk.Entry(activation_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_wlota_entry.grid(row=1, column=5, sticky='w', padx=5)

# Row 2
tk.Label(activation_info_frame, text="WWFF:", font=('Arial', 10), bg='lightgrey').grid(row=2, column=0, sticky='e', padx=5)

station_wwff_entry = tk.Entry(activation_info_frame, font=('Arial', 10, 'bold'), state='disabled', disabledbackground='white', disabledforeground='black', relief='sunk')
station_wwff_entry.grid(row=2, column=1, sticky='w', padx=5)





DateTime_frame = tk.LabelFrame(root, bd=2, font=('Arial', 10, 'bold'), relief='groove', text="QSO Date & Time", labelanchor="n", pady=2)
DateTime_frame.grid(row=2, column=0, columnspan=7, sticky="ew", padx=5, pady=2)


shared_column_widths = [70, 100, 50, 80, 50, 80]
for i, width in enumerate(shared_column_widths):
    DateTime_frame.grid_columnconfigure(i, weight=0, minsize=width)

# DATE
tk.Label(DateTime_frame, text="Date:", font=('Arial', 10)).grid(row=0, column=0, padx=5, pady=2, sticky='e')
date_entry = DateEntry(DateTime_frame, textvariable=qso_date_var, date_pattern='yyyy-mm-dd', font=('Arial', 14, 'bold'))
date_entry.grid(row=0, column=1, padx=5, pady=2, sticky='w')
date_entry.configure(takefocus=False)

def toggle_datetime_tracking():
    """Toggle date time updates and colors based on checkbox state."""
    if datetime_tracking_enabled.get():
        # tracking enabled: start updates + red
        time_entry.config(fg="red")
        update_datetime()  # start/update loop
    else:
        # tracking disabled: no updates + blue
        time_entry.config(fg="black")

# TIME
tk.Label(DateTime_frame, text="Time:", font=('Arial', 10)).grid(
    row=0, column=3, padx=5, pady=2, sticky='e'
)
time_entry = tk.Entry(DateTime_frame, textvariable=qso_time_var, font=('Arial', 14, 'bold'), width=10)
time_entry.grid(row=0, column=4, padx=5, pady=2, sticky='w')
time_entry.configure(takefocus=False)

# initial state
datetime_tracking_enabled.set(True)
time_entry.config(fg="red")
update_datetime()

# TRACKING CHECKBOX (was in menu, staat nu in frame)
datetime_tracking_checkbox = tk.Checkbutton(DateTime_frame, text="Track Time", variable=datetime_tracking_enabled, command=toggle_datetime_tracking, font=('Arial', 10, 'bold'))
datetime_tracking_checkbox.grid(row=0, column=5, pady=2, sticky='w')





MainEntry_frame = tk.LabelFrame(root, bd=2, font=('Arial', 10, 'bold'), relief='groove', text="QSO Entry", labelanchor="n", pady=2)
MainEntry_frame.grid(row=3, column=0, columnspan=7, sticky='ew', padx=5, pady=2)

shared_column_widths = [70, 100, 50, 80, 50, 80]
for i, width in enumerate(shared_column_widths):
    MainEntry_frame.grid_columnconfigure(i, weight=0, minsize=width)


def callsign_on_tab_press(event):
    send_callsign_to_dxcluster()
    use_qrz = config.getboolean("QRZ", "use_qrz_lookup", fallback=True)
    if not use_qrz:
        return  # QRZ lookup disabled, do nothing
    threaded_on_query()
    return None

# CALLSIGN
tk.Label(MainEntry_frame, text="Callsign:", font=('Arial', 10)).grid(row=0, column=0, padx=5, sticky='e')
callsign_entry = tk.Entry(MainEntry_frame, textvariable=qso_callsign_var, font=('Arial', 14, 'bold'), width=14)
callsign_entry.grid(row=0, column=1, padx=5, pady=2, sticky='w')
callsign_entry.bind("<Return>", handle_callsign_input)
callsign_entry.bind("<Tab>", callsign_on_tab_press)
callsign_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
callsign_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))
ToolTip(callsign_entry,
        "This field supports multiple inputs:\n\n"
        "- Callsign entry (e.g. PD5DJ, K1JT, G0ABC/P)\n"
        "- Mode entry (e.g. CW, RTTY, USB, LSB, FM, AM)\n"
        "- Frequency entry (in kHz) (e.g. 14074, 7100, 3500)\n"
        "- Frequency shift input (in kHz) (e.g. +600, -1600, +10000)")


def toggle_serial_fields():
    """
    Enable/disable the serial exchange entries.
    When enabling, calculate the next STX based on the highest STX found in the logbook.
    This is robust even if the logbook is stored newest-first or oldest-first.
    """
    state = "normal" if use_serial_var.get() else "disabled"
    sent_exchange_entry.config(state=state)
    receive_exchange_entry.config(state=state)

    # If serial is enabled, set the next STX based on the logbook contents
    if use_serial_var.get() and CURRENT_JSON_FILE:
        try:
            with open(CURRENT_JSON_FILE, "r", encoding="utf-8") as file:
                data = json.load(file)
                logbook = data.get("Logbook", [])

            max_stx = 0
            for qso in logbook:
                stx_val = str(qso.get("STX", "")).strip()
                if stx_val.isdigit():
                    max_stx = max(max_stx, int(stx_val))

            # Next serial = highest found + 1 (default to 1 if none found)
            qso_sent_exchange_var.set(str(max_stx + 1 if max_stx > 0 else 1))

        except Exception as e:
            print(f"Error retrieving last STX: {e}")
            # Safe fallback
            qso_sent_exchange_var.set("1")




tk.Label(MainEntry_frame, text="Contest:", font=('Arial', 10)).grid(row=1, column=0, padx=5, sticky='e')

use_serial_checkbox = tk.Checkbutton(MainEntry_frame, text='Use Serial',variable=use_serial_var, command=toggle_serial_fields)
use_serial_checkbox.grid(row=1, column=1, padx=5, sticky='w')
use_serial_checkbox.config(takefocus=False)

# SENT
tk.Label(MainEntry_frame, text="Sent:", font=('Arial', 10)).grid(row=0, column=2, padx=5, sticky='e')
rst_sent_entry = tk.Entry(MainEntry_frame, textvariable=qso_rst_sent_var, font=('Arial', 14, 'bold'), width=6)
rst_sent_entry.grid(row=0, column=3, padx=5, pady=2, sticky='w')

tk.Button(MainEntry_frame, text="👆",font=("Segoe UI Emoji", 10, 'bold'),bg="#00FF80",fg="#000000",activebackground="#00AA80",activeforeground="#000000",command=lambda: open_selector(root, qso_rst_sent_var, rst_options, "Select RST"),width=3,takefocus=False).grid(row=0, column=3, padx=(80, 5), pady=2, sticky='w')

# SENT Exchange
tk.Label(MainEntry_frame, text="Sent #:", font=('Arial', 10)).grid(row=1, column=2, padx=5, sticky='e')
sent_exchange_entry = tk.Entry(MainEntry_frame, textvariable=qso_sent_exchange_var, font=('Arial', 14, 'bold'), width=6)
sent_exchange_entry.grid(row=1, column=3, padx=5, pady=2, sticky='w')
sent_exchange_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
sent_exchange_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))

# RECEIVED
tk.Label(MainEntry_frame, text="Received:", font=('Arial', 10)).grid(row=0, column=4, padx=5, sticky='e')
rst_received_entry = tk.Entry(MainEntry_frame, textvariable=qso_rst_received_var, font=('Arial', 14, 'bold'), width=6)
rst_received_entry.grid(row=0, column=5, padx=5, pady=2, sticky='w')

tk.Button(MainEntry_frame, text="👆", font=("Segoe UI Emoji", 10, 'bold'), bg="#00FF80", fg="#000000", activebackground="#00AA80", activeforeground="#000000", command=lambda: open_selector(root, qso_rst_received_var, rst_options, "Select RST"), width=3, takefocus=False).grid(row=0, column=5, padx=(80, 5), pady=2, sticky='w')

# RECEIVE Exchange
tk.Label(MainEntry_frame, text="Rcv. #:", font=('Arial', 10)).grid(row=1, column=4, padx=5, sticky='e')
receive_exchange_entry = tk.Entry(MainEntry_frame, textvariable=qso_receive_exchange_var, font=('Arial', 14, 'bold'), width=6)
receive_exchange_entry.grid(row=1, column=5, padx=5, pady=2, sticky='w')
receive_exchange_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
receive_exchange_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))

toggle_serial_fields() #Turn off entry fields by default

# LOCATOR
tk.Label(MainEntry_frame, text="Locator:", font=('Arial', 10)).grid(row=2, column=0, padx=5, pady=2, sticky='e')
locator_entry = tk.Entry(MainEntry_frame, textvariable=qso_locator_var, font=('Arial', 14, 'bold'), width=12)
locator_entry.grid(row=2, column=1, padx=5, pady=2, sticky='w')
locator_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
locator_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))

# MODE
tk.Label(MainEntry_frame, text="Mode:", font=('Arial', 10)).grid(row=2, column=2, padx=5, pady=2, sticky='e')
mode_combobox = ttk.Combobox(MainEntry_frame, textvariable=qso_mode_var, values=mode_options, font=('Arial', 14, 'bold'), width=8, state='readonly')
mode_combobox.grid(row=2, column=3, padx=5, pady=2, sticky='w')
mode_combobox.bind("<<ComboboxSelected>>", on_mode_change)
bind_jump_to_letter(mode_combobox)



# SUBMODE
tk.Label(MainEntry_frame, text="Submode:", font=('Arial', 10)).grid(row=2, column=4, padx=5, pady=2, sticky='e')
submode_combobox = ttk.Combobox(MainEntry_frame, textvariable=qso_submode_var, values=submode_options, font=('Arial', 14, 'bold'), width=8, state='readonly')
submode_combobox.grid(row=2, column=5, padx=5, pady=2, sticky='w')
submode_combobox.configure(takefocus=False)
bind_jump_to_letter(submode_combobox)


# BAND
tk.Label(MainEntry_frame, text="Band:", font=('Arial', 10)).grid(row=3, column=0, padx=5, pady=2, sticky='e')
band_combobox = ttk.Combobox(MainEntry_frame, width=8, textvariable=qso_band_var, values=list(band_to_frequency.keys()), font=('Arial', 14, 'bold'), state='readonly')
band_combobox.grid(row=3, column=1, padx=5, pady=2, sticky='w')
band_combobox.bind("<<ComboboxSelected>>", update_frequency_from_band)


def on_keypress(event):
    current_value = qso_frequency_var.get()
    
    if event.keysym in ("BackSpace","Tab","Return"):
        # Allow normal backspace behavior
        return True
    
    # Only allow numbers and one decimal point
    if event.char in "0123456789.":
        # Allow the decimal point if there is no decimal already in the entry
        if event.char == "." and current_value.count('.') >= 1:
            return "break"  # Prevent the second decimal point from being inserted
        return True
    else:
        return "break"  # Block other characters

# FREQUENCY
tk.Label(MainEntry_frame, text="Freq (MHz):", font=('Arial', 10)).grid(row=3, column=2, padx=5, pady=2, sticky='e')

freq_entry = tk.Entry(MainEntry_frame, textvariable=qso_frequency_var, font=('Arial', 14, 'bold'), width=10)
freq_entry.grid(row=3, column=3, padx=5, pady=2, sticky='w')

# Function to handle backspace and other key events in the entry
def on_focus(event):
    event.widget.selection_clear()  # Clear any text selection
    event.widget.icursor(tk.END)    # Move cursor to the end

freq_entry.bind("<FocusIn>", on_focus)
freq_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
freq_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))
freq_entry.bind("<KeyPress>", on_keypress)

# keep trace alive
qso_frequency_var.trace_add("write", update_band_from_frequency)


def toggle_tracking():
    global tracking_enabled

    if tracking_checkbox['state'] == 'disabled':
        return

    tracking_enabled = tracking_enabled_var.get()

    if tracking_enabled:
        freq_entry.config(state='readonly', fg="red")
        band_combobox.config(state='disabled')
    else:
        freq_entry.config(state='normal', fg="blue")
        band_combobox.config(state='normal')


tracking_checkbox = tk.Checkbutton(MainEntry_frame, text="Track", variable=tracking_enabled_var, command=toggle_tracking, onvalue=True, offvalue=False, font=('Arial', 10, 'bold'))
tracking_checkbox.grid(row=3, column=4, padx=5)

# --- initial startup state ---
tracking_enabled_var.set(False)
tracking_checkbox.config(state="disabled")
freq_entry.config(state='normal', fg="black")







# NAME
tk.Label(MainEntry_frame, text="Name:", font=('Arial', 10)).grid(row=4, column=0, padx=5, pady=2, sticky='e')
name_entry = tk.Entry(MainEntry_frame, textvariable=qso_name_var, font=('Arial', 12, 'bold'))
name_entry.grid(row=4, column=1, padx=5, pady=2, sticky='w')
name_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
name_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))

# COMMENT
tk.Label(MainEntry_frame, text="Comment:", font=('Arial', 10)).grid(row=4, column=2, padx=5, pady=2, sticky='e')
comment_entry = tk.Entry(MainEntry_frame, textvariable=qso_comment_var, font=('Arial', 12, 'bold'))
comment_entry.grid(row=4, column=3, columnspan=3, padx=5, pady=2, sticky='ew')
comment_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
comment_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))

# SATELLITE
tk.Label(MainEntry_frame, text="Satellite:", font=('Arial', 10)).grid(row=5, column=0, padx=5, pady=2, sticky='e')
satellite_list = load_satellite_names()
satellite_list.insert(0, "")
qso_satellite_var = tk.StringVar()
satellite_dropdown = ttk.Combobox(MainEntry_frame, textvariable=qso_satellite_var, values=satellite_list, font=('Arial', 12, 'bold'), width=8, state="readonly")
satellite_dropdown.grid(row=5, column=1, padx=5, pady=2, sticky='ew')
bind_jump_to_letter(satellite_dropdown)


sat_button = ttk.Button(MainEntry_frame, text="◀ EDIT", command=edit_satellite_names, width=7)
sat_button.grid(row=5, column=2, padx=5, sticky="w")
sat_button.configure(takefocus=False)

# PROPAGATION MODE
tk.Label(MainEntry_frame, text="Propagation:", font=('Arial', 10)).grid(row=5, column=3, padx=5, pady=2, sticky='e')

propagation_modes = [
    "", "AS", "AUE", "AUR", "BS", "ECH", "EME", "ES", "F2",
    "FAI", "GWAVE", "INTERNET", "ION", "IRL", "LOS", "MS",
    "RPT", "RS", "SAT", "TEP", "TR"
]

prop_mode_combo = ttk.Combobox(MainEntry_frame, textvariable=qso_prop_mode_var, values=propagation_modes, font=('Arial', 12, 'bold'), width=12, state="readonly")
prop_mode_combo.grid(row=5, column=4, padx=5, columnspan=2, pady=2, sticky='w')
prop_mode_combo.current(0)
bind_jump_to_letter(prop_mode_combo)


# ==========================================================
# COLLAPSABLE NOTEBOOK BLOCK (POTA / WWFF / IOTA / SOTA etc.)
# ==========================================================

# Create collapsable button for the entire reference section block
pota_toggle_button = create_collapse_button(root, "Reference Sections")
pota_toggle_button.grid(row=5 - 1, column=0, columnspan=7, sticky="ew", padx=5, pady=(0, 3))

# Frame that wraps the notebook content (this collapses)
pota_block_frame = tk.Frame(root)
pota_block_frame.grid(row=5, column=0, columnspan=99, sticky="nsew", padx=5, pady=2)

# Register collapsable behavior using shared engine
register_collapsable_section(button=pota_toggle_button, section_widget=pota_block_frame, default_open=False)      # Start hidden, same as your original code

# Notebook inside collapsable frame
notebook = ttk.Notebook(pota_block_frame)
notebook.pack(fill="both", expand=True)

# ----------------------------------------------------------
# BOTA Tab
# ----------------------------------------------------------
bota_tab = tk.Frame(notebook)
notebook.add(bota_tab, text="BOTA")

bota_tab.grid_columnconfigure(0, weight=0, minsize=70)
bota_tab.grid_columnconfigure(1, weight=1)
bota_tab.grid_columnconfigure(2, weight=0)

tk.Label(bota_tab, text="Ref no.:", font=('Arial', 10), width=12).grid(row=0, column=0, padx=5, pady=2, sticky='e')
bota_entry = tk.Entry(bota_tab, textvariable=bota_var, font=('Arial', 14, 'bold'), width=12)
bota_entry.grid(row=0, column=1, padx=5, pady=2, sticky='w')
bota_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
bota_entry.bind("<FocusOut>", lambda e: auto_fill_bota_prefix(bota_var, bota_name_var, bota_lat_var, bota_long_var, qso_callsign_var))

bota_map_button = tk.Button(bota_tab, text="Show Bunker on Map", takefocus=False,
    command=lambda: open_osm_map(bota_lat_var, bota_long_var, station_locator_var, station_callsign_var, bota_var.get(), bota_name_var.get()))
bota_map_button.grid(row=0, column=2, rowspan=2, padx=5, pady=2, sticky='ew')

tk.Label(bota_tab, text="Bunker Name:", font=('Arial', 10)).grid(row=1, column=0, padx=5, sticky='w')
tk.Label(bota_tab, textvariable=bota_name_var, font=('Arial', 10, 'bold')).grid(row=1, column=1, padx=5, sticky='w')

# ----------------------------------------------------------
# COTA Tab
# ----------------------------------------------------------
cota_tab = tk.Frame(notebook)
notebook.add(cota_tab, text="COTA/WCA")

cota_tab.grid_columnconfigure(0, weight=0, minsize=70)
cota_tab.grid_columnconfigure(1, weight=1)
cota_tab.grid_columnconfigure(2, weight=0)

tk.Label(cota_tab, text="Ref no.:", font=('Arial', 10), width=12).grid(row=0, column=0, padx=5, pady=2, sticky='e')
cota_entry = tk.Entry(cota_tab, textvariable=cota_var, font=('Arial', 14, 'bold'), width=12)
cota_entry.grid(row=0, column=1, padx=5, pady=2, sticky='w')
cota_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))

cota_entry.bind("<FocusOut>", lambda e: auto_fill_cota_prefix(cota_var, cota_name_var, qso_callsign_var))

tk.Label(cota_tab, text="Castle Name:", font=('Arial', 10)).grid(row=1, column=0, padx=5, sticky='w')
tk.Label(cota_tab, textvariable=cota_name_var, font=('Arial', 10, 'bold')).grid(row=1, column=1, padx=5, sticky='w')

# ----------------------------------------------------------
# IOTA Tab
# ----------------------------------------------------------
iota_tab = tk.Frame(notebook)
notebook.add(iota_tab, text="IOTA")

iota_tab.grid_columnconfigure(0, weight=0, minsize=70)
iota_tab.grid_columnconfigure(1, weight=1)
iota_tab.grid_columnconfigure(2, weight=0)

tk.Label(iota_tab, text="Ref no.:", font=('Arial', 10), width=12).grid(row=0, column=0, padx=5, pady=2, sticky='e')
iota_entry = tk.Entry(iota_tab, textvariable=iota_var, font=('Arial', 14, 'bold'), width=12)
iota_entry.grid(row=0, column=1, padx=5, pady=2, sticky='w')
iota_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
iota_entry.bind("<FocusOut>", lambda e: auto_fill_iota_prefix(iota_var, iota_name_var, iota_lat_var, iota_long_var, qso_callsign_var))

iota_map_button = tk.Button(iota_tab, text="Show Island on Map", takefocus=False,
    command=lambda: open_osm_map(iota_lat_var, iota_long_var, station_locator_var, station_callsign_var, iota_var.get(), iota_name_var.get()))
iota_map_button.grid(row=0, column=2, rowspan=2, padx=5, pady=2, sticky='ew')

tk.Label(iota_tab, text="Island Name:", font=('Arial', 10)).grid(row=1, column=0, padx=5, sticky='w')
tk.Label(iota_tab, textvariable=iota_name_var, font=('Arial', 10, 'bold')).grid(row=1, column=1, padx=5, sticky='w')

# ----------------------------------------------------------
# POTA Tab
# ----------------------------------------------------------
pota_tab = tk.Frame(notebook)
notebook.add(pota_tab, text="POTA")

pota_tab.grid_columnconfigure(0, weight=0, minsize=70)
pota_tab.grid_columnconfigure(1, weight=1)

tk.Label(pota_tab, text="Ref no.:", font=('Arial', 10), width=12).grid(row=0, column=0, padx=5, pady=2, sticky='e')
pota_entry = tk.Entry(pota_tab, textvariable=pota_var, font=('Arial', 14, 'bold'), width=12)
pota_entry.grid(row=0, column=1, padx=5, pady=2, sticky='w')
pota_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
pota_entry.bind("<FocusOut>", lambda e: auto_fill_pota_prefix(pota_var, pota_park_name_var, qso_callsign_var))

tk.Label(pota_tab, text="Park Name:", font=('Arial', 10)).grid(row=1, column=0, padx=5, sticky='w')
tk.Label(pota_tab, textvariable=pota_park_name_var, font=('Arial', 10, 'bold')).grid(row=1, column=1, padx=5, sticky='w')

# ----------------------------------------------------------
# SOTA Tab
# ----------------------------------------------------------
sota_tab = tk.Frame(notebook)
notebook.add(sota_tab, text="SOTA")

sota_tab.grid_columnconfigure(0, weight=0, minsize=70)
sota_tab.grid_columnconfigure(1, weight=1)
sota_tab.grid_columnconfigure(2, weight=1)
sota_tab.grid_columnconfigure(3, weight=1)

tk.Label(sota_tab, text="Ref no.:", font=('Arial', 10), width=12).grid(row=0, column=0, padx=5, pady=2, sticky='e')

sota_entry = tk.Entry(sota_tab, textvariable=sota_var, font=('Arial', 14, 'bold'), width=12)
sota_entry.grid(row=0, column=1, padx=5, pady=2, sticky='w')
sota_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
sota_entry.bind("<FocusOut>", lambda e: auto_fill_sota_prefix(sota_var, sota_name_var, sota_lat_var, sota_long_var, qso_callsign_var, sota_matches))

sota_matches = ttk.Combobox(sota_tab, state="readonly", font=('Arial', 10), width=20)
sota_matches.grid(row=0, column=2, padx=5, pady=2, sticky='ew')
sota_matches.bind("<<ComboboxSelected>>", on_sota_select)

sota_map_button = tk.Button(sota_tab, text="Show Summit on Map", takefocus=False,
    command=lambda: open_osm_map(sota_lat_var, sota_long_var, station_locator_var, station_callsign_var, sota_var.get(), sota_name_var.get()))
sota_map_button.grid(row=0, column=3, rowspan=2, padx=5, pady=2, sticky='ew')

tk.Label(sota_tab, text="Summit name:", font=('Arial', 10)).grid(row=1, column=0, padx=5, sticky='w')
tk.Label(sota_tab, textvariable=sota_name_var, font=('Arial', 10, 'bold')).grid(row=1, column=1, sticky='w')

# ----------------------------------------------------------
# WLOTA Tab
# ----------------------------------------------------------
wlota_tab = tk.Frame(notebook)
notebook.add(wlota_tab, text="WLOTA")

wlota_tab.grid_columnconfigure(0, weight=0, minsize=70)
wlota_tab.grid_columnconfigure(1, weight=1)

tk.Label(wlota_tab, text="Ref no.:", font=('Arial', 10), width=12).grid(row=0, column=0, padx=5, pady=2, sticky='e')
wlota_entry = tk.Entry(wlota_tab, textvariable=wlota_var, font=('Arial', 14, 'bold'), width=12)
wlota_entry.grid(row=0, column=1, padx=5, pady=2, sticky='w')
wlota_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))

# ----------------------------------------------------------
# WWFF Tab
# ----------------------------------------------------------
wwff_tab = tk.Frame(notebook)
notebook.add(wwff_tab, text="WWFF")

wwff_tab.grid_columnconfigure(0, weight=0, minsize=70)
wwff_tab.grid_columnconfigure(1, weight=1)
wwff_tab.grid_columnconfigure(2, weight=0)

tk.Label(wwff_tab, text="Ref no.:", font=('Arial', 10), width=12).grid(row=0, column=0, padx=5, pady=2, sticky='e')

wwff_entry = tk.Entry(wwff_tab, textvariable=wwff_var, font=('Arial', 14, 'bold'), width=12)
wwff_entry.grid(row=0, column=1, padx=5, pady=2, sticky='w')
wwff_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
wwff_entry.bind("<FocusOut>", lambda e: auto_fill_wwff_prefix(wwff_var, wwff_park_name_var, wwff_lat_var, wwff_long_var, qso_callsign_var))

wwff_map_button = tk.Button(wwff_tab, text="Show Park on Map", takefocus=False,
    command=lambda: open_osm_map(wwff_lat_var, wwff_long_var, station_locator_var, station_callsign_var, wwff_var.get(), wwff_park_name_var.get()))
wwff_map_button.grid(row=0, column=2, rowspan=2, padx=5, pady=2, sticky='ew')

tk.Label(wwff_tab, text="Park Name:", font=('Arial', 10)).grid(row=1, column=0, padx=5, sticky='w')
tk.Label(wwff_tab, textvariable=wwff_park_name_var, font=('Arial', 10, 'bold')).grid(row=1, column=1, padx=5, sticky='w')









notebook = ttk.Notebook(root, takefocus=False)
notebook.grid(row=7, column=0, columnspan=99, sticky='nsew', padx=5, pady=5)

# Tab 1: DXCC Info
dxcc_tab = tk.Frame(notebook, bg=root.cget("bg"))
dxcc_tab.grid_columnconfigure(0, weight=1)
notebook.add(dxcc_tab, text="DXCC Info")

# Tab 2: Extra QRZ Lookup results
qrz_tab = tk.Frame(notebook, bg=root.cget("bg"))
notebook.add(qrz_tab, text="Extra QRZ Lookup results")

# Country/Continent label
country_continent_label = tk.Label(dxcc_tab, text="----", font=('Arial', 14, 'bold'), bg=root.cget("bg"))
country_continent_label.grid(row=0, column=0, sticky='ew', pady=(0, 5))

# Subframe-container
subframe_container = tk.Frame(dxcc_tab, bg=root.cget("bg"))
subframe_container.grid(row=1, column=0, sticky='ew')
subframe_container.grid_columnconfigure((0, 1, 2), weight=1, uniform="equal")

# Zones Frame
zones_frame = tk.LabelFrame(subframe_container, text="Zones", font=('Arial', 9, 'bold'),
                            labelanchor="n", bg=root.cget("bg"), relief='groove', bd=2)
zones_frame.grid(row=0, column=0, padx=5, pady=2, sticky='nsew')
zones_frame.grid_rowconfigure(0, weight=1)
zones_frame.grid_columnconfigure(0, weight=1)

dxcc_cq_itu_label = tk.Label(zones_frame, text="----", font=('Arial', 12, 'bold'), bg=root.cget("bg"))
dxcc_cq_itu_label.grid(row=0, column=0, sticky='nsew')

# Distance Frame
distance_frame = tk.LabelFrame(subframe_container, text="Distance", font=('Arial', 9, 'bold'),
                               labelanchor="n", bg=root.cget("bg"), relief='groove', bd=2)
distance_frame.grid(row=0, column=1, padx=5, pady=2, sticky='nsew')
distance_frame.grid_rowconfigure(0, weight=1)
distance_frame.grid_columnconfigure(0, weight=1)

distance_label = tk.Label(distance_frame, text="----", font=('Arial', 12, 'bold'), bg=root.cget("bg"))
distance_label.grid(row=0, column=0, sticky='nsew')

# Heading Frame
heading_frame = tk.LabelFrame(subframe_container, text="Heading", font=('Arial', 9, 'bold'),
                              labelanchor="n", bg=root.cget("bg"), relief='groove', bd=2)
heading_frame.grid(row=0, column=2, padx=5, pady=2, sticky='nsew')
heading_frame.grid_rowconfigure(0, weight=1)
heading_frame.grid_columnconfigure(0, weight=1)

heading_label = tk.Label(heading_frame, text="----", font=('Arial', 16, 'bold'), bg=root.cget("bg"))
heading_label.grid(row=0, column=0, sticky='nsew')

# DXCC Flag
DXCC_FLAG_SIZE = 36
orig = FLAG_IMAGES.get(-1)
resized = resize_flag(orig, DXCC_FLAG_SIZE)
flag_label = tk.Label(dxcc_tab, image=resized, bg=root.cget("bg"))
flag_label.image = resized
flag_label.grid(row=0, column=0, rowspan=2, padx=5, sticky="nw")


tk.Label(qrz_tab, text="Address:", font=('Arial', 10)).grid(row=0, column=0, sticky='e', padx=5, pady=2)
address_entry = tk.Entry(qrz_tab, textvariable=qrz_address_var, font=('Arial', 10, 'bold'), width=30)
address_entry.grid(row=0, column=1, columnspan=2, padx=5, pady=2, sticky='w')
address_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
address_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))
address_entry.configure(takefocus=False)

tk.Label(qrz_tab, text="City:", font=('Arial', 10)).grid(row=1, column=0, sticky='e', padx=5, pady=2)
city_entry = tk.Entry(qrz_tab, textvariable=qrz_city_var, font=('Arial', 10, 'bold'), width=30)
city_entry.grid(row=1, column=1, columnspan=2, sticky='w', padx=5, pady=2)
city_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
city_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))
city_entry.configure(takefocus=False)

tk.Label(qrz_tab, text="Zipcode:", font=('Arial', 10)).grid(row=0, column=3, sticky='e', padx=5, pady=2)
zipcode_entry = tk.Entry(qrz_tab, textvariable=qrz_zipcode_var, font=('Arial', 10, 'bold'), width=15)
zipcode_entry.grid(row=0, column=4, sticky='w', padx=5, pady=2)
zipcode_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
zipcode_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))
zipcode_entry.configure(takefocus=False)

tk.Label(qrz_tab, text="QSL Info:", font=('Arial', 10)).grid(row=1, column=3, sticky='e', padx=5, pady=2)
qsl_info_entry = tk.Entry(qrz_tab, textvariable=qrz_qsl_info_var, font=('Arial', 10, 'bold'), width=35)
qsl_info_entry.grid(row=1, column=4, columnspan=1, padx=5, pady=2, sticky='w')
qsl_info_entry.bind("<FocusIn>", lambda e: e.widget.configure(background="#FFCCDD"))
qsl_info_entry.bind("<FocusOut>", lambda e: e.widget.configure(background="white"))
qsl_info_entry.configure(takefocus=False)

qrz_tab.grid_columnconfigure(0, weight=0, minsize=70)
qrz_tab.grid_columnconfigure(1, weight=0, minsize=80)
qrz_tab.grid_columnconfigure(2, weight=0, minsize=10)
qrz_tab.grid_columnconfigure(3, weight=1)
qrz_tab.grid_columnconfigure(4, weight=1)






#----------- WORKED BEFORE FRAME ------------

workedb4_frame = tk.LabelFrame(root, bg=root.cget("bg"), bd=2, font=('Arial', 10, 'bold'), relief='groove', text="Worked Before Lookup", labelanchor="n", padx=5, pady=2)
workedb4_frame.grid(row=8, column=0, columnspan=99, sticky="nsew", pady=2, padx=5)

tree_frame = tk.Frame(workedb4_frame)
tree_frame.pack(fill="both", expand=True, padx=10, pady=0)

# Scrollbars
y_scroll = tk.Scrollbar(tree_frame, orient="vertical")
y_scroll.pack(side="right", fill="y")

# TreeView
cols = ("Callsign", "Date", "Time", "Band", "Mode", "Frequency", "Country")
workedb4_tree = ttk.Treeview(tree_frame, columns=cols, show='headings', height=3, yscrollcommand=y_scroll.set)

y_scroll.config(command=workedb4_tree.yview)

for col in cols:
    workedb4_tree.heading(col, text=col)
    workedb4_tree.column(col, anchor="center", width=80)

workedb4_tree.pack(fill="both", expand=True)



Button_frame = tk.Frame(root, bd=2, relief='groove', padx=5, pady=2)
Button_frame.grid(row=9, column=0, columnspan=7, pady=2, padx=5, sticky='ew')

Button_frame.grid_columnconfigure(0, weight=1)
Button_frame.grid_columnconfigure(1, weight=1)
Button_frame.grid_columnconfigure(2, weight=1)
Button_frame.grid_columnconfigure(3, weight=1)

# Detecteer platform en pas stijl toe
use_ttk = False
if platform.system() == "Darwin":
    use_ttk = True
    style = ttk.Style()
    style.theme_use("clam")

    # Lookup button
    style.configure("Lookup.TButton", foreground="black", background="#FFFF80", font=('Arial', 10, 'bold'), anchor="center", justify="center", padding=(0, 6))
    style.map("Lookup.TButton", background=[("active", "#e0e066")])

    # Log button
    style.configure("Log.TButton", foreground="black", background="#00FF80", font=('Arial', 10, 'bold'), anchor="center", justify="center", padding=(0, 6))
    style.map("Log.TButton", background=[("active", "#00cc66")])

    # Wipe button
    style.configure("Wipe.TButton", foreground="black", background="#FF8080", font=('Arial', 10, 'bold'), anchor="center", justify="center", padding=(0, 6))
    style.map("Wipe.TButton", background=[("active", "#cc6666")])

    # Exit button
    style.configure("Exit.TButton", foreground="white", background="grey", font=('Arial', 10, 'bold'), anchor="center", justify="center", padding=(0, 12))
    style.map("Exit.TButton", background=[("active", "#666666")])


# --- Buttons ---
if use_ttk:
    # ttk variant met vaste afmeting in tekens

    log_button = ttk.Button(Button_frame, text="Log QSO\nF1", command=log_qso, style="Log.TButton", width=10)
    log_button.grid(row=0, column=0, padx=10)
    log_button.configure(takefocus=False)

    lookup_button = ttk.Button(Button_frame, text="Lookup\nF2", command=threaded_on_query, style="Lookup.TButton", width=10)
    lookup_button.grid(row=0, column=1, padx=10)
    lookup_button.configure(takefocus=False)

    logbook_button = ttk.Button(Button_frame, text="Logbook\nF3", command=view_logbook, style="Log.TButton", width=10)
    logbook_button.grid(row=0, column=2, padx=10)
    logbook_button.configure(takefocus=False)

    dxcluster_button = ttk.Button(Button_frame, text="DX Cluster\nF4", command=open_dxspotviewer, style="Log.TButton", width=10)
    dxcluster_button.grid(row=0, column=3, padx=10)
    dxcluster_button.configure(takefocus=False)    

    wipe_button = ttk.Button(Button_frame, text="Wipe\nF5", command=reset_fields, style="Wipe.TButton", width=10)
    wipe_button.grid(row=0, column=4, padx=10)
    wipe_button.configure(takefocus=False)

    EXIT_button = ttk.Button(Button_frame, text="EXIT", command=exit_program, style="Exit.TButton", width=10)
    EXIT_button.grid(row=0, column=5, padx=10)
    EXIT_button.configure(takefocus=False)

else:

    # Log
    log_button = tk.Button(Button_frame, text="Log QSO\nF1", command=log_qso, bd=3, relief='raised', width=10, height=2, bg='#00FF80', fg='black', font=('Arial', 10, 'bold'))
    log_button.grid(row=0, column=0, padx=10)
    log_button.configure(takefocus=False)

    # Lookup
    lookup_button = tk.Button(Button_frame, text="Lookup\nF2", command=threaded_on_query, bd=3, relief='raised', width=10, height=2, bg='#FFFF80', fg='black', font=('Arial', 10, 'bold'))
    lookup_button.grid(row=0, column=1, padx=10)
    lookup_button.configure(takefocus=False)

    # Logbook
    logbook_button = tk.Button(Button_frame, text="Logbook\nF3", command=view_logbook, bd=3, relief='raised', width=10, height=2, bg='#00FFFF', fg='black', font=('Arial', 10, 'bold'))
    logbook_button.grid(row=0, column=2, padx=10)
    logbook_button.configure(takefocus=False)    

    # DX Cluster
    dxcluster_button = tk.Button(Button_frame, text="DX Cluster\nF4", command=open_dxspotviewer, bd=3, relief='raised', width=10, height=2, bg='#8080FF', fg='white', font=('Arial', 10, 'bold'))
    dxcluster_button.grid(row=0, column=3, padx=10)
    dxcluster_button.configure(takefocus=False)    

     # Wipe
    wipe_button = tk.Button(Button_frame, text="Wipe\nF5", command=reset_fields, bd=3, relief='raised', width=10, height=2, bg='#FF8080', fg='black', font=('Arial', 10, 'bold'))
    wipe_button.grid(row=0, column=4, padx=10)
    wipe_button.configure(takefocus=False)


    # EXIT
    EXIT_button = tk.Button(Button_frame, text="EXIT", command=exit_program, bd=3, relief='raised', width=10, height=2, bg='grey', fg='white', font=('Arial', 10, 'bold'))
    EXIT_button.grid(row=0, column=5, padx=10)
    EXIT_button.configure(takefocus=False)



status_frame = tk.Frame(root, bd=2, relief='groove', padx=5, pady=2)
status_frame.grid(row=10, column=0, columnspan=7, rowspan=1, sticky="ew", padx=5, pady=2)

status_frame.grid_columnconfigure(0, weight=1)
status_frame.grid_columnconfigure(1, weight=1)

# last_qso_label 
last_qso_label = tk.Label(status_frame, fg="blue", font=('Arial', 8, 'bold'), text="Last QSO info")
last_qso_label.grid(row=0, column=0, sticky='w', padx=5)

# QRZ_status_label 
QRZ_status_label = tk.Label(status_frame, font=('Arial', 8, 'bold'), text="QRZ Status")
QRZ_status_label.grid(row=0, column=1, sticky='e', padx=5)


# Let Tkinter calculate the required window size after all widgets are placed
root.update_idletasks()

# Fix the window size to the calculated size
final_width = root.winfo_width()
final_height = root.winfo_height()
root.geometry(f"{final_width}x{final_height}")

# Disable resizing (window is now fixed)
root.resizable(False, False)


# Buttons Bindings

# Bind F1 key to log_button function
def invoke_log_button(event=None):
    log_button.invoke()
root.bind("<F1>", invoke_log_button)

# Bind F2 key to Lookup function
def invoke_lookup_button(event=None):
    lookup_button.invoke()
root.bind("<F2>", invoke_lookup_button)

# Bind F3 key to logbook_button function
def invoke_logbook_button(event=None):
    logbook_button.invoke()
root.bind("<F3>", invoke_logbook_button)

# Bind F4 key to dxcluster_button function
def invoke_dxcluster_button(event=None):
    dxcluster_button.invoke()
root.bind("<F4>", invoke_dxcluster_button)

# Bind F5 key to reset_fields function
def invoke_reset_fields(event=None):
    reset_fields()
root.bind("<F5>", invoke_reset_fields)

def on_minibook_exit():
    global dxspotviewer_window
    if dxspotviewer_window and dxspotviewer_window.winfo_exists():
        dxspotviewer_window.destroy()
    root.destroy()

root.protocol("WM_DELETE_WINDOW", on_minibook_exit)


# Call init function to Preset / Preload  variables / Tasks
init()

# Bind the on_close function to the "X" button
root.protocol("WM_DELETE_WINDOW", exit_program)

# --------------------------------------------------------
# Start background thread for WordPress updater
# --------------------------------------------------------
threading.Thread(target=dashboard_updater, daemon=True).start()


# Start main loop
root.mainloop()