<?php
// ID wordt door plugin doorgegeven via $atts of $instance
$id = isset($atts['id']) ? $atts['id'] : (isset($instance['id']) ? $instance['id'] : '');
$id = (string) $id;
$id_attr = esc_attr($id);

// JSON live endpoint (blijft op basis van het MiniBook ID)
$live_url = add_query_arg('minibook_live', rawurlencode($id), home_url('/'));

// Style uit shortcode of widget (default = "default")
$style = isset($atts['style']) ? $atts['style'] : (isset($instance['style']) ? $instance['style'] : 'default');
$style = (string) $style;
$style_attr = esc_attr($style);

// Alle bekende banden
$all_bands = array(
    '160' => '160m',
    '80'  => '80m',
    '60'  => '60m',
    '40'  => '40m',
    '30'  => '30m',
    '20'  => '20m',
    '17'  => '17m',
    '15'  => '15m',
    '12'  => '12m',
    '10'  => '10m',
    '6'   => '6m',
    '4'   => '4m',
    '2'   => '2m',
    '125' => '1.25m',
    '70'  => '70cm',
    '33'  => '33cm',
    '23'  => '23cm',
    '13'  => '13cm',
);

// Default: alle banden tonen
$bands_for_view = array_keys($all_bands);

// Widget / shortcode kan een bandfilter doorgeven
if (isset($instance['selected_bands']) && is_array($instance['selected_bands']) && !empty($instance['selected_bands'])) {
    $bands_for_view = array();
    foreach ($instance['selected_bands'] as $b) {
        if (isset($all_bands[$b])) {
            $bands_for_view[] = $b;
        }
    }
    if (empty($bands_for_view)) {
        $bands_for_view = array_keys($all_bands);
    }
} elseif (isset($selected_bands) && is_array($selected_bands) && !empty($selected_bands)) {
    $bands_for_view = array();
    foreach ($selected_bands as $b) {
        if (isset($all_bands[$b])) {
            $bands_for_view[] = $b;
        }
    }
    if (empty($bands_for_view)) {
        $bands_for_view = array_keys($all_bands);
    }
}

// Interne band-ID’s (zonder domId suffix)
$band_id_list = array();
foreach ($bands_for_view as $b) {
    $band_id_list[] = 'b' . $b;
}

// *** IMPORTANT ***
// Maak een unieke DOM-ID per instance, zodat meerdere dashboards
// met hetzelfde MiniBook ID op één pagina geen botsende element-ID’s hebben.
if (!isset($GLOBALS['minibook_instance_counter'])) {
    $GLOBALS['minibook_instance_counter'] = 0;
}
$GLOBALS['minibook_instance_counter']++;
$dom_id = $id_attr . '-' . $GLOBALS['minibook_instance_counter'];
$dom_id_attr = esc_attr($dom_id);
?>

<div class="minibook-wrapper minibook-style-<?php echo $style_attr; ?>" data-minibook-id="<?php echo $id_attr; ?>">
    <div class="minibook-info-box">
        <table>
            <tr><td>Frequency:</td> <td id="freq-<?php echo $dom_id_attr; ?>">-</td></tr>
<?php if ($style === 'text' || $style === 'text-compact') : ?>
            <tr class="band-row"><td>Band:</td> <td id="band-<?php echo $dom_id_attr; ?>">-</td></tr>
<?php endif; ?>
            <tr><td>Mode:</td>      <td id="mode-<?php echo $dom_id_attr; ?>">-</td></tr>
            <tr><td>Callsign:</td>  <td id="mycall-<?php echo $dom_id_attr; ?>">-</td></tr>
            <tr><td>Operator:</td>  <td id="opcall-<?php echo $dom_id_attr; ?>">-</td></tr>
            <tr><td>Locator:</td>   <td id="locator-<?php echo $dom_id_attr; ?>">-</td></tr>
            <tr><td>Location:</td>  <td id="location-<?php echo $dom_id_attr; ?>">-</td></tr>
            <tr><td>Last update:</td> <td id="ts-<?php echo $dom_id_attr; ?>">-</td></tr>
        </table>
    </div>

    <div class="minibook-bands" aria-label="Bands overview">
        <?php foreach ($bands_for_view as $b) : ?>
            <div class="band" id="b<?php echo $b; ?>-<?php echo $dom_id_attr; ?>">
                <?php echo esc_html($all_bands[$b]); ?>
            </div>
        <?php endforeach; ?>
    </div>

    <p class="minibook-status" id="status-<?php echo $dom_id_attr; ?>">Waiting for data…</p>
</div>

<script>
(function() {
    // dataId = MiniBook ID (voor JSON endpoint)
    const dataId = <?php echo json_encode($id); ?>;
    // domId = unieke DOM-suffix per instance
    const domId = <?php echo json_encode($dom_id); ?>;
    const liveUrl = <?php echo json_encode($live_url); ?>;

    if (!dataId) {
        return;
    }

    const elFreq     = document.getElementById("freq-" + domId);
    const elMode     = document.getElementById("mode-" + domId);
    const elMyCall   = document.getElementById("mycall-" + domId);
    const elOpCall   = document.getElementById("opcall-" + domId);
    const elLocator  = document.getElementById("locator-" + domId);
    const elLocation = document.getElementById("location-" + domId);
    const elTs       = document.getElementById("ts-" + domId);
    const elStatus   = document.getElementById("status-" + domId);

    const bandIds = <?php echo json_encode($band_id_list); ?>;

    function clearFields() {
        elFreq.textContent     = "-";
        elMode.textContent     = "-";

        // Reset band text (voor text / text-compact styles)
        const elBand = document.getElementById("band-" + domId);
        if (elBand) { elBand.textContent = "-"; }

        elMyCall.textContent   = "-";
        elOpCall.textContent   = "-";
        elLocator.textContent  = "-";
        elLocation.textContent = "-";
        elTs.textContent       = "-";
    }


	function setStatus(state) {
		const elStatus = document.getElementById("status-" + domId);
		if (!elStatus) return;

		elStatus.classList.remove("is-live", "is-offair");

		if (state === "ON AIR") {
			elStatus.classList.add("is-live");
		} else if (state === "OFF AIR") {
			elStatus.classList.add("is-offair");
		}

		elStatus.textContent = state;
	}


    function setActiveBand(band) {
        // Clear all
        bandIds.forEach(function(b) {
            const el = document.getElementById(b + "-" + domId);
            if (el) {
                el.classList.remove("active");
            }
        });

        if (!band) {
            return;
        }

        let norm = band.toString().toLowerCase().trim();

        // Normalize some common band strings
        if (norm.startsWith("1.8") || norm.indexOf("160") !== -1) norm = "160";
        else if (norm.startsWith("3.")  || norm.indexOf("80")  !== -1) norm = "80";
        else if (norm.startsWith("5.3") || norm.indexOf("60")  !== -1) norm = "60";
        else if (norm.startsWith("7.")  || norm.indexOf("40")  !== -1) norm = "40";
        else if (norm.startsWith("10.") || norm.indexOf("30")  !== -1) norm = "30";
        else if (norm.startsWith("14")  || norm.indexOf("20")  !== -1) norm = "20";
        else if (norm.startsWith("18.") || norm.indexOf("17")  !== -1) norm = "17";
        else if (norm.startsWith("21")  || norm.indexOf("15")  !== -1) norm = "15";
        else if (norm.startsWith("24.") || norm.indexOf("12")  !== -1) norm = "12";
        else if (norm.startsWith("28")  || norm.indexOf("10")  !== -1) norm = "10";
        else if (norm.indexOf("6m") !== -1  || norm.indexOf("50")   !== -1) norm = "6";
        else if (norm.indexOf("4m") !== -1) norm = "4";
        else if (norm.startsWith("70.") || norm.indexOf("70mhz") !== -1) norm = "4";
        else if (norm.indexOf("2m") !== -1  || norm.indexOf("144")  !== -1) norm = "2";
        else if (norm.indexOf("1.25m") !== -1 || norm.indexOf("220") !== -1) norm = "125";
        else if (norm.indexOf("33cm") !== -1  || norm.indexOf("902") !== -1) norm = "33";
        else if (norm.indexOf("70") !== -1 || norm.indexOf("430")  !== -1 || norm.indexOf("432") !== -1) norm = "70";
        else if (norm.indexOf("23cm") !== -1 || norm.indexOf("1296") !== -1 || norm.indexOf("1.2") !== -1) norm = "23";
        else if (norm.indexOf("13cm") !== -1 || norm.indexOf("2320") !== -1 || norm.indexOf("2.3") !== -1) norm = "13";

        const mapId = {
            "160": "b160",
            "80":  "b80",
            "60":  "b60",
            "40":  "b40",
            "30":  "b30",
            "20":  "b20",
            "17":  "b17",
            "15":  "b15",
            "12":  "b12",
            "10":  "b10",
            "6":   "b6",
            "4":   "b4",
            "2":   "b2",
            "125": "b125",
            "70":  "b70",
            "33":  "b33",
            "23":  "b23",
            "13":  "b13"
        };

        const bid = mapId[norm];
        if (bid) {
            const el = document.getElementById(bid + "-" + domId);
            if (el) {
                el.classList.add("active");
            }
        }
    }

    async function fetchMiniBook() {
        try {
            const response = await fetch(liveUrl, {
                cache: "no-store",
                headers: { "Accept": "application/json" }
            });

            if (!response.ok) {
                setStatus("OFF AIR");
                clearFields();
                setActiveBand(null);
            } else {
                const json = await response.json();

                if (!json || !json.success) {
                    // Could be OFFAIR from server
                    setStatus("OFF AIR");
                    clearFields();
                    setActiveBand(null);
                } else {
                    const data = json.data || json; // support both styles

                    elFreq.textContent = data.freq ? (data.freq + " MHz") : "-";
                    elMode.textContent     = data.mode || "-";

                    // Band text ALWAYS comes from payload (no frequency -> band logic)
                    const bandLabel = (data.band || "-");
                    const elBand = document.getElementById("band-" + domId);
                    if (elBand) { elBand.textContent = bandLabel; }

                    elMyCall.textContent   = data.mycall   || "-";
                    elOpCall.textContent   = data.opcall   || "-";
                    elLocator.textContent  = data.locator  || "-";
                    elLocation.textContent = data.location || "-";
                    elTs.textContent       = data.ts_human || "-";

                    setStatus("ON AIR");
                    setActiveBand(bandLabel);
                }
            }
        } catch (err) {
            console.error("MiniBook Live error:", err);
            setStatus("OFF AIR");
            clearFields();
            setActiveBand(null);
        }

        // Poll again in ~2 seconds
        setTimeout(fetchMiniBook, 2000);
    }

    fetchMiniBook();
})();
</script>
