<?php
/*
Plugin Name: PD5DJ MiniBook Dashboard
Plugin URI: https://pd5dj.nl/
Description: Live MiniBook dashboard receiving POST updates every few seconds and displaying real-time radio data in WordPress via shortcode or widget.
Version: 1.8.1
Author: PD5DJ
Author URI: https://pd5dj.nl/
License: GPL2
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: pd5dj-minibook-dashboard

Requires at least: 5.0
Tested up to: 6.9
Requires PHP: 7.4
*/

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Returns the upload directory for MiniBook JSON files.
 *
 * @return array { 'dir' => string, 'url' => string }
 */
function minibook_dashboard_get_upload_dir() {
    $upload = wp_upload_dir();
    $dir    = trailingslashit($upload['basedir']) . 'minibook';
    $url    = trailingslashit($upload['baseurl']) . 'minibook';

    if (!file_exists($dir)) {
        wp_mkdir_p($dir);
    }

    return array(
        'dir' => $dir,
        'url' => $url,
    );
}

/**
 * Get current API key from options.
 *
 * @return string
 */
function minibook_dashboard_get_api_key() {
    $key = get_option('minibook_api_key', '');
    if (empty($key)) {
        // Generate a new key if not present yet.
        $key = wp_generate_password(32, false, false);
        update_option('minibook_api_key', $key);
    }
    return $key;
}

/**
 * Plugin activation hook: ensure an API key exists.
 */
function minibook_dashboard_activate() {
    minibook_dashboard_get_api_key();
}
register_activation_hook(__FILE__, 'minibook_dashboard_activate');


/* ============================================================
   POST RECEIVER — MiniBook → WordPress (secured by API key)
   ============================================================ */

/**
 * Handle incoming POST updates from MiniBook.
 *
 * Endpoint example:
 *   https://example.com/?minibook_update=IC7610
 *
 * Required POST field:
 *   api_key  - must match the key from the settings page.
 *
 * Suggested POST fields:
 *   freq, band, mode, mycall, opcall, locator, location, radio, radio_label, app, station
 */
add_action('init', function () {
    if (!isset($_GET['minibook_update'])) {
        return;
    }

    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        wp_die('Only POST allowed.');
    }

    // Validate ID
    $id = preg_replace('/[^a-zA-Z0-9_-]/', '', $_GET['minibook_update']);
    if ($id === '') {
        wp_die('Invalid ID.');
    }

    // Validate API key
    $posted_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
    $stored_key = minibook_dashboard_get_api_key();

    if (empty($posted_key) || !hash_equals($stored_key, $posted_key)) {
        status_header(403);
        wp_die('Unauthorized (invalid API key).');
    }

    // Build data structure from POST
    $fields = array(
        'freq',
        'band',
        'mode',
        'mycall',
        'opcall',
        'locator',
        'location',
        'radio',
        'radio_label',
        'app',
        'station',
    );

    $data = array();
    foreach ($fields as $field) {
        $data[$field] = isset($_POST[$field]) ? sanitize_text_field(wp_unslash($_POST[$field])) : '';
    }

    // Timestamp (Unix + human)
    $data['ts']         = time();
    $data['ts_human']   = current_time('mysql');
    $data['id']         = $id;
    $data['plugin_ver'] = '1.8.1';

    // Save JSON file
    $upload = minibook_dashboard_get_upload_dir();
    $file   = trailingslashit($upload['dir']) . 'minibook_' . $id . '.json';

    $json   = wp_json_encode($data);
    file_put_contents($file, $json);

    // Respond with simple OK JSON
    wp_send_json_success(array(
        'saved' => true,
        'id'    => $id,
    ));
	
	// GENEREER SNAPSHOT EERST
	generate_minibook_qrz_snapshot($data);

	// DAARNA JSON RESPONSE
	wp_send_json_success(array(
		'saved' => true,
		'id'    => $id,
	));


});


/* ============================================================
   LIVE ENDPOINT — JSON polling for dashboard (with OFFAIR)
   ============================================================ */

add_action('init', function () {

    if (!isset($_GET['minibook_live'])) {
        return;
    }

    $id = preg_replace('/[^a-zA-Z0-9_-]/', '', $_GET['minibook_live']);
    if ($id === '') {
        wp_send_json_error(array('error' => 'Invalid ID.'));
    }

    $upload = minibook_dashboard_get_upload_dir();
    $file   = trailingslashit($upload['dir']) . 'minibook_' . $id . '.json';

    if (!file_exists($file)) {

        // ❌ No data at all → OFFAIR snapshot
        generate_minibook_qrz_snapshot(array(
            'mycall' => '',
            'freq'   => '',
            'band'   => '',
            'mode'   => '',
            'radio'  => '',
            'status' => 'OFFAIR',
        ));

        wp_send_json_error(array(
            'error'  => 'No data yet.',
            'offair' => true,
        ));
    }

    $raw  = file_get_contents($file);
    $data = json_decode($raw, true);

    if (!is_array($data)) {

        // ❌ Corrupt → OFFAIR snapshot
        generate_minibook_qrz_snapshot(array(
            'mycall' => '',
            'freq'   => '',
            'band'   => '',
            'mode'   => '',
            'radio'  => '',
            'status' => 'OFFAIR',
        ));

        wp_send_json_error(array(
            'error'  => 'Corrupt data.',
            'offair' => true,
        ));
    }

    // OFFAIR detection based on timestamp
    $now         = time();
    $last_update = isset($data['ts']) ? (int) $data['ts'] : 0;

    $timeout = (int) get_option('minibook_offair_timeout', 300);
    if ($timeout < 1) {
        $timeout = 300;
    }

    // ❌ OFFAIR (timeout)
    if ($last_update === 0 || ($now - $last_update) > $timeout) {

        // Mark OFFAIR explicitly for snapshot
        $data['status'] = 'OFFAIR';
        generate_minibook_qrz_snapshot($data);

        wp_send_json_error(array(
            'error'       => 'OFF AIR',
            'offair'      => true,
            'last_update' => $last_update,
        ));
    }

    // ✅ LIVE — optional: keep QRZ snapshot in sync
    $data['status'] = 'LIVE';
    generate_minibook_qrz_snapshot($data);

    wp_send_json_success($data);
});


function generate_minibook_qrz_snapshot($data)
{
    // Resolve ID (per-ID snapshot)
    $id = isset($data['id']) && $data['id'] !== ''
        ? preg_replace('/[^a-zA-Z0-9_-]/', '', $data['id'])
        : 'default';

    $upload_dir = wp_upload_dir();
    $dir  = trailingslashit($upload_dir['basedir']) . 'minibook';
    $file = trailingslashit($dir) . 'qrz-' . $id . '.html';

    if (!file_exists($dir)) {
        wp_mkdir_p($dir);
    }

    /* =========================
       STATUS + TIMESTAMP
       ========================= */

    $on_air = !isset($data['status']) || $data['status'] !== 'OFFAIR';

    $status_text  = $on_air ? 'ON AIR' : 'OFF AIR';
    $status_color = $on_air ? '#ff2b2b' : '#666';
    $status_glow  = $on_air
        ? 'text-shadow: 0 0 6px rgba(255,43,43,0.9), 0 0 14px rgba(255,43,43,0.6);'
        : '';

    // Last update (altijd tonen)
    $last_update = isset($data['ts_human'])
        ? esc_html($data['ts_human'])
        : date('Y-m-d H:i:s');

    /* =========================
       DATA (OFF AIR → alles "-")
       ========================= */

	if ($on_air && !empty($data['freq'])) {
		$frequency = esc_html($data['freq']) . ' MHz';
	} else {
		$frequency = '-';
	}

    $band      = $on_air ? esc_html($data['band'] ?? '-') : '-';
    $mode      = $on_air ? esc_html($data['mode'] ?? '-') : '-';
    $callsign  = $on_air ? esc_html($data['mycall'] ?? '-') : '-';
    $operator  = $on_air ? esc_html($data['opcall'] ?? '-') : '-';
    $locator   = $on_air ? esc_html($data['locator'] ?? '-') : '-';
    $location  = $on_air ? esc_html($data['location'] ?? '-') : '-';

    /* =========================
       HTML (Cards style)
       ========================= */

    $html = <<<HTML
<!DOCTYPE html>
<html>
<head>
<meta charset="utf-8">
<style>
body {
    margin: 0;
    padding: 10px;
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Arial, sans-serif;
    background: transparent;
}

.card {
    background: #ffffff;
    border-radius: 10px;
    border: 1px solid rgba(0,0,0,0.22);
    box-shadow:
        0 1px 3px rgba(0,0,0,0.25),
        0 3px 8px rgba(0,0,0,0.15);
    padding: 14px;
    max-width: 300px;
    margin: 0 auto;
    box-sizing: border-box;
}

.status {
    text-align: center;
    font-size: 1.15rem;
    font-weight: 700;
    margin-bottom: 12px;
    color: {$status_color};
    {$status_glow}
}

.row {
    display: flex;
    justify-content: space-between;
    font-size: 0.95rem;
    margin: 4px 0;
}

.label {
    color: #666;
    font-weight: 600;
    margin-right: 10px;
    white-space: nowrap;
}

.value {
    color: #000;
    font-weight: 500;
    text-align: right;
    flex: 1;
}

.footer {
    margin-top: 10px;
    text-align: center;
    font-size: 0.7rem;
    color: #999;
    opacity: 0.85;
}
</style>
</head>
<body>
<div class="card">

    <div class="status">{$status_text}</div>

    <div class="row"><div class="label">Frequency</div><div class="value">{$frequency}</div></div>
    <div class="row"><div class="label">Band</div><div class="value">{$band}</div></div>
    <div class="row"><div class="label">Mode</div><div class="value">{$mode}</div></div>
    <div class="row"><div class="label">Callsign</div><div class="value">{$callsign}</div></div>
    <div class="row"><div class="label">Operator</div><div class="value">{$operator}</div></div>
    <div class="row"><div class="label">Locator</div><div class="value">{$locator}</div></div>
    <div class="row"><div class="label">Location</div><div class="value">{$location}</div></div>

    <div class="footer">
        Last update: {$last_update}<br>
        generated by MiniBook logging software
    </div>

</div>
</body>
</html>
HTML;

    file_put_contents($file, $html);
}




/* ============================================================
   SHORTCODE — [minibook_dashboard id="IC7610"]
   ============================================================ */

/**
 * Enqueue frontend inline CSS for dashboard.
 */
function minibook_dashboard_enqueue_styles() {
    wp_register_style(
        'minibook-dashboard-inline-style',
        plugins_url('inline-style.css', __FILE__),
        array(),
        '1.8.1'
    );
    wp_enqueue_style('minibook-dashboard-inline-style');
}

/**
 * Shortcode callback.
 *
 * @param array $atts
 * @return string
 */
function minibook_dashboard_shortcode($atts) {
    minibook_dashboard_enqueue_styles();

    $atts = shortcode_atts(
        array(
            'id'    => '',
            'style' => 'default',
            'bands' => '',
        ),
        $atts,
        'minibook_dashboard'
    );

    if (empty($atts['id'])) {
        return '<p>MiniBook Dashboard: missing <code>id</code> attribute.</p>';
    }

    // Define all known bands and groups
    $all_bands = array('160','80','60','40','30','20','17','15','12','10','6','4','2','125','70','33','23','13');

    $groups = array(
        'hf'  => array('160','80','60','40','30','20','17','15','12','10'),
        'vhf' => array('6','4','2','125'),
        'uhf' => array('70','33','23'),
        'shf' => array('13'),
    );

    $bands_param = strtolower(trim($atts['bands']));
    $selected_bands = array();

    if ($bands_param === '' || $bands_param === 'all') {
        $selected_bands = $all_bands;
    } else {
        $parts = explode(',', $bands_param);
        foreach ($parts as $part) {
            $part = strtolower(trim($part));
            if ($part === '') {
                continue;
            }

            // Group name?
            if (isset($groups[$part])) {
                $selected_bands = array_merge($selected_bands, $groups[$part]);
                continue;
            }

            // Individual band notation like "20m", "70cm", "23", etc.
            $normalized = str_replace(array('m', 'cm', ' '), '', $part);
            if (in_array($normalized, $all_bands, true)) {
                $selected_bands[] = $normalized;
            }
        }

        $selected_bands = array_values(array_unique($selected_bands));

        // If nothing valid was parsed, fall back to all bands
        if (empty($selected_bands)) {
            $selected_bands = $all_bands;
        }
    }

    ob_start();
    $instance = array(
        'id'             => $atts['id'],
        'style'          => $atts['style'],
        'selected_bands' => $selected_bands,
    );
    include __DIR__ . '/html-template.php';
    return ob_get_clean();
}
add_shortcode('minibook_dashboard', 'minibook_dashboard_shortcode');


/* ============================================================
   WIDGET — also loads template with JS live updater
   ============================================================ */

class MiniBookDashboardWidget extends WP_Widget {

    public function __construct() {
        parent::__construct(
            'minibook_dashboard_widget',
            __('MiniBook Live Dashboard', 'pd5dj-minibook-dashboard'),
            array(
                'description' => __('Displays live MiniBook radio data.', 'pd5dj-minibook-dashboard'),
            )
        );
    }

    /**
     * Frontend output of the widget.
     */
    public function widget($args, $instance) {
        minibook_dashboard_enqueue_styles();

        $id = isset($instance['id']) ? $instance['id'] : '';

        echo $args['before_widget'];

        if (!empty($args['before_title'])) {
            echo $args['before_title'] . esc_html__('MiniBook Live', 'pd5dj-minibook-dashboard') . $args['after_title'];
        }

        if (empty($id)) {
            echo '<p>MiniBook Dashboard: configure an ID in the widget settings.</p>';
        } else {
            // Default: show all bands in widget context
            $selected_bands = array('160','80','60','40','30','20','17','15','12','10','6','4','2','70','23','13');
            $atts = array(
                'id'    => $id,
                'style' => 'default',
            );
            include __DIR__ . '/html-template.php';
        }

        echo $args['after_widget'];
    }

    /**
     * Backend widget form.
     */
    public function form($instance) {
        $id = isset($instance['id']) ? $instance['id'] : '';
        $field_id   = $this->get_field_id('id');
        $field_name = $this->get_field_name('id');
        ?>
        <p>
            <label for="<?php echo esc_attr($field_id); ?>"><?php esc_html_e('MiniBook ID', 'pd5dj-minibook-dashboard'); ?></label>
            <input class="widefat"
                   id="<?php echo esc_attr($field_id); ?>"
                   name="<?php echo esc_attr($field_name); ?>"
                   type="text"
                   value="<?php echo esc_attr($id); ?>"
                   placeholder="IC7610, IC7300, Shack1, etc." />
        </p>
        <?php
    }

    /**
     * Save widget options.
     */
    public function update($new_instance, $old_instance) {
        $instance        = array();
        $instance['id']  = isset($new_instance['id']) ? sanitize_text_field($new_instance['id']) : '';
        return $instance;
    }
}

/**
 * Register the widget.
 */
add_action('widgets_init', function () {
    register_widget('MiniBookDashboardWidget');
});


/* ============================================================
   SETTINGS PAGE — API key management (view + regenerate)
   ============================================================ */

/**
 * Add settings page under Settings → MiniBook Dashboard.
 */
function minibook_dashboard_add_settings_page() {
    add_options_page(
        __('MiniBook Dashboard', 'pd5dj-minibook-dashboard'),
        __('MiniBook Dashboard', 'pd5dj-minibook-dashboard'),
        'manage_options',
        'minibook-dashboard-settings',
        'minibook_dashboard_render_settings_page'
    );
}
add_action('admin_menu', 'minibook_dashboard_add_settings_page');


/**
 * Register setting for API key.
 */
function minibook_dashboard_register_settings() {
    register_setting(
        'minibook_dashboard_settings_group',
        'minibook_api_key',
        array(
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => '',
        )
    );

    register_setting(
        'minibook_dashboard_settings_group',
        'minibook_offair_timeout',
        array(
            'type'              => 'integer',
            'sanitize_callback' => 'absint',
            'default'           => 300,
        )
    );
}
add_action('admin_init', 'minibook_dashboard_register_settings');


/**
 * Render settings page.
 */
function minibook_dashboard_render_settings_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    // Handle regenerate action
    if (isset($_POST['minibook_regenerate_key'])) {
        check_admin_referer('minibook_dashboard_regenerate_key');

        $new_key = wp_generate_password(32, false, false);
        update_option('minibook_api_key', $new_key);
        add_settings_error(
            'minibook_dashboard_messages',
            'minibook_dashboard_key_regenerated',
            __('New API key generated.', 'pd5dj-minibook-dashboard'),
            'updated'
        );
    }

    settings_errors('minibook_dashboard_messages');

    $api_key = minibook_dashboard_get_api_key();

    ?>
    <div class="wrap">
        <h1><?php esc_html_e('MiniBook Dashboard Settings', 'pd5dj-minibook-dashboard'); ?></h1>

        <p><?php esc_html_e('Use this API key in your MiniBook or external script to authenticate POST updates to this site.', 'pd5dj-minibook-dashboard'); ?></p>

        <form method="post" action="options.php">
            <?php
            settings_fields('minibook_dashboard_settings_group');
            do_settings_sections('minibook_dashboard_settings_group');
            ?>

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="minibook_api_key"><?php esc_html_e('API Key', 'pd5dj-minibook-dashboard'); ?></label>
                    </th>
                    <td>
                        <input type="text"
                               id="minibook_api_key"
                               name="minibook_api_key"
                               value="<?php echo esc_attr($api_key); ?>"
                               class="regular-text"
                               readonly />
                        <p class="description">
                            <?php esc_html_e('This key is required as the "api_key" POST field when sending updates.', 'pd5dj-minibook-dashboard'); ?>
                        </p>
                    </td>
                </tr>
                <?php $timeout = (int) get_option('minibook_offair_timeout', 10); if ($timeout < 1) { $timeout = 300; } ?>
                <tr>
                    <th scope="row">
                        <label for="minibook_offair_timeout"><?php esc_html_e('OFF AIR timeout (seconds)', 'pd5dj-minibook-dashboard'); ?></label>
                    </th>
                    <td>
                        <input type="number"
                               id="minibook_offair_timeout"
                               name="minibook_offair_timeout"
                               value="<?php echo esc_attr($timeout); ?>"
                               min="1"
                               class="small-text" />
                        <p class="description">
                            <?php esc_html_e('If no new update is received within this number of seconds, the dashboard goes to OFF AIR.', 'pd5dj-minibook-dashboard'); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <?php submit_button(__('Save Settings', 'pd5dj-minibook-dashboard')); ?>
        </form>

        <hr />

        <form method="post">
            <?php wp_nonce_field('minibook_dashboard_regenerate_key'); ?>
            <p>
                <strong><?php esc_html_e('Regenerate API Key', 'pd5dj-minibook-dashboard'); ?></strong>
            </p>
            <p>
                <?php esc_html_e('If you suspect your key has leaked, you can generate a new one. Be sure to update it in all connected clients (MiniBook, scripts, etc.).', 'pd5dj-minibook-dashboard'); ?>
            </p>
            <p>
                <input type="submit"
                       name="minibook_regenerate_key"
                       class="button button-secondary"
                       value="<?php esc_attr_e('Regenerate API Key', 'pd5dj-minibook-dashboard'); ?>"
                       onclick="return confirm('<?php echo esc_js(__('Are you sure you want to generate a new API key?', 'pd5dj-minibook-dashboard')); ?>');" />
            </p>
        </form>

        <hr />

        <h2><?php esc_html_e('How to use', 'pd5dj-minibook-dashboard'); ?></h2>
        <ol>
            <li><?php esc_html_e('Copy the API key above.', 'pd5dj-minibook-dashboard'); ?></li>
            <li><?php esc_html_e('Configure your MiniBook or external sender to POST to:', 'pd5dj-minibook-dashboard'); ?>
                <br />
                <code><?php echo esc_html( home_url('/?minibook_update=IC7610') ); ?></code>
            </li>
            <li><?php esc_html_e('Include the POST field "api_key" with the value from this page.', 'pd5dj-minibook-dashboard'); ?></li>
            <li><?php esc_html_e('Place the shortcode on a page to display the live dashboard:', 'pd5dj-minibook-dashboard'); ?>
                <br />
                <code>[minibook_dashboard id="IC7610"]</code>
            </li>
        </ol>
    </div>
    <?php
}


/* ============================================================
   VIEW DETAILS LINK + LOCAL PLUGIN INFORMATION POPUP
   ============================================================ */

/**
 * Provide local plugin information for the "View details" thickbox.
 */
add_filter('plugins_api', function($result, $action, $args) {

    if ($action !== 'plugin_information' || empty($args->slug) || $args->slug !== 'pd5dj-minibook-dashboard') {
        return $result;
    }

    $info = new stdClass();

    $info->name          = 'PD5DJ MiniBook Dashboard';
    $info->slug          = 'pd5dj-minibook-dashboard';
    $info->version       = '1.8.1';
    $info->author        = '<a href="https://pd5dj.nl/">PD5DJ</a>';
    $info->homepage      = 'https://pd5dj.nl/';
    $info->requires      = '5.0';
    $info->tested        = '6.7';
    $info->requires_php  = '7.4';
    $info->last_updated  = current_time('mysql');
    $info->download_link = '';

    // Build absolute URLs for screenshots inside the plugin.
    $assets_url = plugin_dir_url(__FILE__) . 'assets/';

    $screenshots_html = sprintf('
        <ol>
            <li>Wordpress Admin page<br>
                <img src="%1$s" alt="Wordpress Admin page" />
            </li>

            <li>MiniBook Preference WordPress dashboard tab<br>
                <img src="%2$s" alt="MiniBook Preference WordPress dashboard tab" />
            </li>

            <li>Example Sidebar widget<br>
                <img src="%3$s" alt="Example Sidebar widget" />
            </li>
        </ol>
    ',
        esc_url($assets_url . 'screenshot-1.png'),
        esc_url($assets_url . 'screenshot-2.png'),
        esc_url($assets_url . 'screenshot-3.png')
    );



    $info->sections = array(
		'description' => '
			<h3>MiniBook Live Dashboard</h3>

			<p>
				The MiniBook Dashboard plugin provides a real-time, modern interface for displaying live radio information
				received from the MiniBook Desktop Application. It is designed for amateur radio operators who want to show
				frequency, band, mode, callsign, operator name, locator, radio label, and LIVE / OFF AIR status directly
				on their WordPress website using a clean and responsive dashboard.
			</p>

			<p>
				The MiniBook application securely transmits updates every few seconds using a unique API key.
				This plugin receives those POST updates and updates the dashboard instantly, without requiring a page reload.
				The same incoming data is also used to generate a dedicated static HTML snapshot for QRZ.com.
			</p>

			<p>
				In addition to the live WordPress dashboard, the plugin automatically creates a
				<strong>QRZ HTML Snapshot</strong>. This is a static HTML file that can be safely embedded on QRZ.com
				using an iframe. The snapshot contains pure HTML and inline CSS, does not execute PHP, and does not depend
				on WordPress themes.
			</p>

			<p>
				The dashboard can be displayed anywhere using a <strong>shortcode</strong>, or inside a sidebar or footer
				using the <strong>built-in widget</strong>. Multiple dashboards can run at the same time, each with their own
				unique ID (<code>YOURID</code>), style, and band visibility configuration.
			</p>

			<p>
				Several display styles are available, including default band icon layouts, compact sidebar-friendly views,
				text-only modes, dark mode, and card-based designs.
				Band icons and band text are always based on the transmitted <strong>band payload</strong>
				and support HF, VHF, UHF, and SHF bands.
			</p>

			<p>
				For more information about the MiniBook application, downloads, and documentation, visit:<br>
				<a href="https://pd5dj.nl" target="_blank">https://pd5dj.nl</a>
			</p>
		',



		'installation' => '
			<h3>MiniBook Dashboard – Installation & Usage Manual</h3>

			<h3>Installation</h3>
			<ol>
				<li>Upload the plugin ZIP via <strong>Plugins → Add New → Upload Plugin</strong>.</li>
				<li>Activate the plugin.</li>
				<li>Open <strong>Settings → MiniBook Dashboard</strong> and copy the generated API key into your MiniBook application.</li>
				<li>
					Place one of the following shortcodes:
					<br>
					<code>[minibook_dashboard id="YOURID"]</code><br>
					<code>[minibook_dashboard id="YOURID" style="cards"]</code><br>
					<code>[minibook_dashboard id="YOURID" style="dark"]</code><br>
					<code>[minibook_dashboard id="YOURID" style="compact"]</code><br>
					<code>[minibook_dashboard id="YOURID" style="text"]</code><br>
					<code>[minibook_dashboard id="YOURID" style="text-compact"]</code><br>
				</li>
				<li>Or add the widget via <strong>Appearance → Widgets</strong>.</li>
			</ol>

			<hr>

			<h3>Styles</h3>

			<p>
				The <code>style</code> attribute defines how the dashboard is displayed.
				If omitted, <strong>default</strong> is used.
			</p>

			<ul>
				<li><code>default</code> – Standard view with band icons</li>
				<li><code>compact</code> – Smaller version for sidebars</li>
				<li><code>cards</code> – Card-based layout</li>
				<li><code>dark</code> – Dark themed dashboard</li>
				<li><code>text</code> – Text-only view</li>
				<li><code>text-compact</code> – Minimal text-only layout</li>
			</ul>

			<hr>

			<h3>Bands</h3>

			<p>
				The <code>bands</code> attribute controls which band icons are visible in the dashboard.
				If the attribute is omitted, <strong>all supported bands</strong> are displayed.
			</p>

			<p>
				<strong>Important:</strong> Band display is entirely based on the transmitted
				<code>band</code> payload received from the MiniBook Desktop Application.
				No frequency-based band calculation is performed anywhere in this plugin.
			</p>

			<h4>Supported band groups</h4>

			<p>
				For convenience, commonly used bands can be grouped using predefined keywords:
			</p>

			<ul>
				<li><code>hf</code> → 160m, 80m, 60m, 40m, 30m, 20m, 17m, 15m, 12m, 10m</li>
				<li><code>vhf</code> → 6m, 4m, 2m, 1.25m</li>
				<li><code>uhf</code> → 70cm, 33cm</li>
				<li><code>shf</code> → 23cm, 13cm</li>
			</ul>

			<h4>Individual band selection</h4>

			<p>
				Individual bands can be specified using comma-separated values.
				The values must match the band identifiers transmitted by MiniBook.
			</p>

			<pre><code>bands="20,40,80"</code></pre>
			<pre><code>bands="2,70,23"</code></pre>

			<h4>Combining band groups and individual bands</h4>

			<p>
				Band groups and individual bands can be freely combined in the same
				<code>bands</code> attribute. The resulting dashboard will display the union
				of all specified groups and bands.
			</p>

			<pre><code>bands="hf,13"</code></pre>
			<p>Displays all HF bands plus the 13cm band.</p>

			<pre><code>bands="vhf,23"</code></pre>
			<p>Displays all VHF bands plus the 23cm band.</p>

			<pre><code>bands="hf,23,13"</code></pre>
			<p>Displays all HF bands plus both 23cm and 13cm.</p>

			<pre><code>bands="uhf,shf,20"</code></pre>
			<p>Displays all UHF and SHF bands plus the 20m band.</p>

			<h4>Examples</h4>

			<pre><code>[minibook_dashboard id="YOURID" bands="hf"]</code></pre>
			<p>Displays only HF band icons.</p>

			<pre><code>[minibook_dashboard id="YOURID" bands="vhf,shf"]</code></pre>
			<p>Displays VHF and SHF bands.</p>

			<pre><code>[minibook_dashboard id="YOURID" bands="hf,13"]</code></pre>
			<p>Displays HF bands plus 13cm.</p>

			<pre><code>[minibook_dashboard id="YOURID" bands="hf,23,13"]</code></pre>
			<p>Displays HF bands plus 23cm and 13cm.</p>

			<pre><code>[minibook_dashboard id="YOURID" style="text" bands="hf,shf"]</code></pre>
			<p>
				Text-only layout (icons hidden). Band information is still displayed
				using the transmitted band payload.
			</p>

			<p>
				Band highlighting always reflects the currently active band reported
				by the MiniBook Desktop Application.
			</p>


			<h3>QRZ HTML Snapshot</h3>

			<p>
				Each time MiniBook sends an update, the plugin automatically generates a static HTML file
				intended for QRZ.com embedding.
			</p>

			<p>
				<strong>Snapshot location:</strong><br>
				<code>/wp-content/uploads/minibook/</code>
			</p>

			<p>
				<strong>Snapshot filename format:</strong><br>
				<code>qrz-YOURID.html</code>
			</p>

			<p>
				Example:
				<br>
				<code>/wp-content/uploads/minibook/qrz-YOURID.html</code>
			</p>

			<p>
				The snapshot uses the same <code>YOURID</code> as:
				<ul>
					<li>The MiniBook Desktop configuration</li>
					<li>The WordPress shortcode</li>
					<li>The QRZ iframe URL</li>
				</ul>
			</p>

			<h4>Embedding on QRZ.com</h4>

			<p>
				Edit your QRZ.com biography or HTML section and insert the following iframe:
			</p>

<pre><code>&lt;iframe
 src="https://yourdomain.com/wp-content/uploads/minibook/qrz-YOURID.html"
 width="100%"
 height="300"
 frameborder="0"
 scrolling="no"&gt;
&lt;/iframe&gt;</code></pre>

			<p>
				Adjust the height if needed. QRZ loads the static file directly; no WordPress login or PHP execution
				is required.
			</p>

			<hr>

			<p>
				This manual reflects the current MiniBook Dashboard behaviour and supported features.
			</p>
		',



        'screenshots' => $screenshots_html,

		'changelog' => '
			<h3>Version 1.8.1</h3>
			<ul>
				<li>Added QRZ-compatible static HTML snapshot feature.</li>
				<li>Introduced unified YOURID usage across MiniBook, WordPress, and QRZ.</li>
				<li>Added professional documentation for QRZ iframe embedding.</li>
				<li>Documentation fully aligned with current plugin behaviour.</li>
			</ul>

			<h3>Version 1.8.1</h3>
			<ul>
				<li>Removed deprecated layout styles (wide, horizontal, horizontal-compact).</li>
				<li>Band handling is fully payload-based; no frequency-to-band logic remains.</li>
				<li>Added full support for 1.25m and 33cm bands.</li>
				<li>Improved compact layout band sizing and grid behaviour.</li>
				<li>LIVE status now displays with centered red glow.</li>
			</ul>

			<h3>Version 1.7.0 – 1.7.7</h3>
			<ul>
				<li>Added shortcode band filtering.</li>
				<li>Improved text-only styles.</li>
				<li>Enhanced SHF band handling.</li>
				<li>General layout and UI improvements.</li>
			</ul>
		',

    );


    return $info;

}, 10, 3);

/**
 * Add a "View details" link in the plugin meta row (next to Visit plugin site).
 */
add_filter('plugin_row_meta', function($links, $file) {
    if ($file === plugin_basename(__FILE__)) {
        $details_url = admin_url(
            'plugin-install.php?tab=plugin-information&plugin=pd5dj-minibook-dashboard&TB_iframe=true&width=600&height=550'
        );
        $links[] = '<a href="' . esc_url($details_url) . '" class="thickbox">View details</a>';
    }
    return $links;
}, 10, 2);

/**
 * Hide "not tested with your version of WordPress" warning
 * for this custom plugin.
 */

/* Force WP to think this plugin is tested up to the current WP version */
add_filter('plugins_api_result', function($res, $action, $args) {
    if (!empty($args->slug) && $args->slug === 'pd5dj-minibook-dashboard') {
        $res->tested = get_bloginfo('version');
        $res->requires = '5.0';
        $res->requires_php = '7.4';
    }
    return $res;
}, 10, 3);

/* Modify the plugin row meta data output */
add_filter('plugin_row_meta', function($links, $file) {
    if (strpos($file, 'pd5dj-minibook-dashboard.php') !== false) {
        // Inject a hidden element that prevents JS from marking it incompatible
        $links['compat'] = '<span class="pd5dj-plugin-compatible" data-compatible="true"></span>';
    }
    return $links;
}, 10, 2);

/* Modify the internal plugin data used by WP-admin JavaScript */
add_filter('wp_prepare_plugins_for_js', function($plugins) {
    foreach ($plugins as &$plugin) {
        if (!empty($plugin['slug']) && $plugin['slug'] === 'pd5dj-minibook-dashboard') {
            $plugin['tested'] = get_bloginfo('version');
            $plugin['requires'] = '5.0';
            $plugin['requires_php'] = '7.4';

            // Force WordPress to consider it compatible
            $plugin['compatibility'] = array(
                'compatible' => true,
                'tested'     => true,
            );
        }
    }
    return $plugins;
});
